/*
 * Copyright (c) 2023 Oracle and/or its affiliates.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package io.helidon.inject.api;

import java.util.ArrayList;
import java.util.Collections;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.Consumer;
import java.util.function.Supplier;

import io.helidon.builder.api.Prototype;
import io.helidon.common.Errors;
import io.helidon.common.Generated;
import io.helidon.common.types.Annotation;
import io.helidon.common.types.TypeName;
import io.helidon.common.types.TypedElementInfo;

import jakarta.inject.Provider;

/**
 * Used by {@link Interceptor}.
 *
 * @see #builder()
 * @see #create()
 */
@Generated(value = "io.helidon.builder.processor.BlueprintProcessor", trigger = "io.helidon.inject.api.InvocationContextBlueprint")
public interface InvocationContext extends InvocationContextBlueprint, Prototype.Api {

    /**
     * Create a new fluent API builder to customize configuration.
     *
     * @return a new builder
     */
    static InvocationContext.Builder builder() {
        return new InvocationContext.Builder();
    }

    /**
     * Create a new fluent API builder from an existing instance.
     *
     * @param instance an existing instance used as a base for the builder
     * @return a builder based on an instance
     */
    static InvocationContext.Builder builder(InvocationContext instance) {
        return InvocationContext.builder().from(instance);
    }

    /**
     * Create a new instance with default values.
     *
     * @return a new instance
     */
    static InvocationContext create() {
        return InvocationContext.builder().buildPrototype();
    }

    /**
     * Fluent API builder base for {@link InvocationContext}.
     *
     * @param <BUILDER> type of the builder extending this abstract builder
     * @param <PROTOTYPE> type of the prototype interface that would be built by {@link #buildPrototype()}
     */
    abstract class BuilderBase<BUILDER extends InvocationContext.BuilderBase<BUILDER, PROTOTYPE>, PROTOTYPE extends InvocationContext> implements Prototype.Builder<BUILDER, PROTOTYPE> {

        private final List<Annotation> classAnnotations = new ArrayList<>();
        private final List<TypedElementInfo> elementArgInfo = new ArrayList<>();
        private final List<Provider<Interceptor>> interceptors = new ArrayList<>();
        private final Map<String, Object> contextData = new LinkedHashMap<>();
        private ServiceProvider<?> serviceProvider;
        private TypedElementInfo elementInfo;
        private TypeName serviceTypeName;

        /**
         * Protected to support extensibility.
         */
        protected BuilderBase() {
        }

        /**
         * Update this builder from an existing prototype instance.
         *
         * @param prototype existing prototype to update this builder from
         * @return updated builder instance
         */
        public BUILDER from(InvocationContext prototype) {
            serviceProvider(prototype.serviceProvider());
            serviceTypeName(prototype.serviceTypeName());
            addClassAnnotations(prototype.classAnnotations());
            elementInfo(prototype.elementInfo());
            addElementArgInfo(prototype.elementArgInfo());
            addInterceptors(prototype.interceptors());
            addContextData(prototype.contextData());
            return self();
        }

        /**
         * Update this builder from an existing prototype builder instance.
         *
         * @param builder existing builder prototype to update this builder from
         * @return updated builder instance
         */
        public BUILDER from(InvocationContext.BuilderBase<?, ?> builder) {
            builder.serviceProvider().ifPresent(this::serviceProvider);
            builder.serviceTypeName().ifPresent(this::serviceTypeName);
            addClassAnnotations(builder.classAnnotations());
            builder.elementInfo().ifPresent(this::elementInfo);
            addElementArgInfo(builder.elementArgInfo());
            addInterceptors(builder.interceptors());
            addContextData(builder.contextData());
            return self();
        }

        /**
         * The service provider being intercepted.
         *
         * @param serviceProvider the service provider being intercepted
         * @return updated builder instance
         * @see #serviceProvider()
         */
        public BUILDER serviceProvider(ServiceProvider<?> serviceProvider) {
            Objects.requireNonNull(serviceProvider);
            this.serviceProvider = serviceProvider;
            return self();
        }

        /**
         * The service type name for the root service provider.
         *
         * @param serviceTypeName the service type name for the root service provider
         * @return updated builder instance
         * @see #serviceTypeName()
         */
        public BUILDER serviceTypeName(TypeName serviceTypeName) {
            Objects.requireNonNull(serviceTypeName);
            this.serviceTypeName = serviceTypeName;
            return self();
        }

        /**
         * The service type name for the root service provider.
         *
         * @param consumer consumer of builder for
         *                 the service type name for the root service provider
         * @return updated builder instance
         * @see #serviceTypeName()
         */
        public BUILDER serviceTypeName(Consumer<TypeName.Builder> consumer) {
            Objects.requireNonNull(consumer);
            var builder = TypeName.builder();
            consumer.accept(builder);
            this.serviceTypeName(builder.build());
            return self();
        }

        /**
         * The service type name for the root service provider.
         *
         * @param supplier supplier of
         *                 the service type name for the root service provider
         * @return updated builder instance
         * @see #serviceTypeName()
         */
        public BUILDER serviceTypeName(Supplier<? extends TypeName> supplier) {
            Objects.requireNonNull(supplier);
            this.serviceTypeName(supplier.get());
            return self();
        }

        /**
         * The annotations on the enclosing type.
         *
         * @param classAnnotations the annotations on the enclosing type
         * @return updated builder instance
         * @see #classAnnotations()
         */
        public BUILDER classAnnotations(List<? extends Annotation> classAnnotations) {
            Objects.requireNonNull(classAnnotations);
            this.classAnnotations.clear();
            this.classAnnotations.addAll(classAnnotations);
            return self();
        }

        /**
         * The annotations on the enclosing type.
         *
         * @param classAnnotations the annotations on the enclosing type
         * @return updated builder instance
         * @see #classAnnotations()
         */
        public BUILDER addClassAnnotations(List<? extends Annotation> classAnnotations) {
            Objects.requireNonNull(classAnnotations);
            this.classAnnotations.addAll(classAnnotations);
            return self();
        }

        /**
         * The element info represents the method (or the constructor) being invoked.
         *
         * @param elementInfo the element info represents the method (or the constructor) being invoked
         * @return updated builder instance
         * @see #elementInfo()
         */
        public BUILDER elementInfo(TypedElementInfo elementInfo) {
            Objects.requireNonNull(elementInfo);
            this.elementInfo = elementInfo;
            return self();
        }

        /**
         * The element info represents the method (or the constructor) being invoked.
         *
         * @param consumer consumer of builder for
         *                 the element info represents the method (or the constructor) being invoked
         * @return updated builder instance
         * @see #elementInfo()
         */
        public BUILDER elementInfo(Consumer<TypedElementInfo.Builder> consumer) {
            Objects.requireNonNull(consumer);
            var builder = TypedElementInfo.builder();
            consumer.accept(builder);
            this.elementInfo(builder.build());
            return self();
        }

        /**
         * The element info represents the method (or the constructor) being invoked.
         *
         * @param supplier supplier of
         *                 the element info represents the method (or the constructor) being invoked
         * @return updated builder instance
         * @see #elementInfo()
         */
        public BUILDER elementInfo(Supplier<? extends TypedElementInfo> supplier) {
            Objects.requireNonNull(supplier);
            this.elementInfo(supplier.get());
            return self();
        }

        /**
         * The method/element argument info.
         *
         * @param elementArgInfo the method/element argument info
         * @return updated builder instance
         * @see #elementArgInfo()
         */
        public BUILDER elementArgInfo(List<? extends TypedElementInfo> elementArgInfo) {
            Objects.requireNonNull(elementArgInfo);
            this.elementArgInfo.clear();
            this.elementArgInfo.addAll(elementArgInfo);
            return self();
        }

        /**
         * The method/element argument info.
         *
         * @param elementArgInfo the method/element argument info
         * @return updated builder instance
         * @see #elementArgInfo()
         */
        public BUILDER addElementArgInfo(List<? extends TypedElementInfo> elementArgInfo) {
            Objects.requireNonNull(elementArgInfo);
            this.elementArgInfo.addAll(elementArgInfo);
            return self();
        }

        /**
         * The interceptor chain.
         *
         * @param interceptors the interceptor chain
         * @return updated builder instance
         * @see #interceptors()
         */
        public BUILDER interceptors(List<Provider<Interceptor>> interceptors) {
            Objects.requireNonNull(interceptors);
            this.interceptors.clear();
            this.interceptors.addAll(interceptors);
            return self();
        }

        /**
         * The interceptor chain.
         *
         * @param interceptors the interceptor chain
         * @return updated builder instance
         * @see #interceptors()
         */
        public BUILDER addInterceptors(List<Provider<Interceptor>> interceptors) {
            Objects.requireNonNull(interceptors);
            this.interceptors.addAll(interceptors);
            return self();
        }

        /**
         * This method replaces all values with the new ones.
         *
         * @param contextData the read/write contextual data that is passed between each chained interceptor
         * @return updated builder instance
         * @see #contextData()
         */
        public BUILDER contextData(Map<? extends String, ?> contextData) {
            Objects.requireNonNull(contextData);
            this.contextData.clear();
            this.contextData.putAll(contextData);
            return self();
        }

        /**
         * This method keeps existing values, then puts all new values into the map.
         *
         * @param contextData the read/write contextual data that is passed between each chained interceptor
         * @return updated builder instance
         * @see #contextData()
         */
        public BUILDER addContextData(Map<? extends String, ?> contextData) {
            Objects.requireNonNull(contextData);
            this.contextData.putAll(contextData);
            return self();
        }

        /**
         * The service provider being intercepted.
         *
         * @return the service provider
         */
        public Optional<ServiceProvider<?>> serviceProvider() {
            return Optional.ofNullable(serviceProvider);
        }

        /**
         * The service type name for the root service provider.
         *
         * @return the service type name
         */
        public Optional<TypeName> serviceTypeName() {
            return Optional.ofNullable(serviceTypeName);
        }

        /**
         * The annotations on the enclosing type.
         *
         * @return the class annotations
         */
        public List<Annotation> classAnnotations() {
            return classAnnotations;
        }

        /**
         * The element info represents the method (or the constructor) being invoked.
         *
         * @return the element info
         */
        public Optional<TypedElementInfo> elementInfo() {
            return Optional.ofNullable(elementInfo);
        }

        /**
         * The method/element argument info.
         *
         * @return the element arg info
         */
        public List<TypedElementInfo> elementArgInfo() {
            return elementArgInfo;
        }

        /**
         * The interceptor chain.
         *
         * @return the interceptors
         */
        public List<Provider<Interceptor>> interceptors() {
            return interceptors;
        }

        /**
         * The contextual info that can be shared between interceptors.
         *
         * @return the context data
         */
        public Map<String, Object> contextData() {
            return contextData;
        }

        @Override
        public String toString() {
            return "InvocationContextBuilder{"
                    + "serviceProvider=" + serviceProvider + ","
                    + "serviceTypeName=" + serviceTypeName + ","
                    + "classAnnotations=" + classAnnotations + ","
                    + "elementInfo=" + elementInfo + ","
                    + "elementArgInfo=" + elementArgInfo + ","
                    + "interceptors=" + interceptors + ","
                    + "contextData=" + contextData
                    + "}";
        }

        /**
         * Handles providers and decorators.
         */
        protected void preBuildPrototype() {
        }

        /**
         * Validates required properties.
         */
        protected void validatePrototype() {
            Errors.Collector collector = Errors.collector();
            if (serviceProvider == null) {
                collector.fatal(getClass(), "Property \"serviceProvider\" must not be null, but not set");
            }
            if (serviceTypeName == null) {
                collector.fatal(getClass(), "Property \"serviceTypeName\" must not be null, but not set");
            }
            if (elementInfo == null) {
                collector.fatal(getClass(), "Property \"elementInfo\" must not be null, but not set");
            }
            collector.collect().checkValid();
        }

        /**
         * Generated implementation of the prototype, can be extended by descendant prototype implementations.
         */
        protected static class InvocationContextImpl implements InvocationContext {

            private final List<Annotation> classAnnotations;
            private final List<TypedElementInfo> elementArgInfo;
            private final List<Provider<Interceptor>> interceptors;
            private final Map<String, Object> contextData;
            private final ServiceProvider<?> serviceProvider;
            private final TypedElementInfo elementInfo;
            private final TypeName serviceTypeName;

            /**
             * Create an instance providing a builder.
             *
             * @param builder extending builder base of this prototype
             */
            protected InvocationContextImpl(InvocationContext.BuilderBase<?, ?> builder) {
                this.serviceProvider = builder.serviceProvider().get();
                this.serviceTypeName = builder.serviceTypeName().get();
                this.classAnnotations = List.copyOf(builder.classAnnotations());
                this.elementInfo = builder.elementInfo().get();
                this.elementArgInfo = List.copyOf(builder.elementArgInfo());
                this.interceptors = List.copyOf(builder.interceptors());
                this.contextData = Collections.unmodifiableMap(new LinkedHashMap<>(builder.contextData()));
            }

            @Override
            public ServiceProvider<?> serviceProvider() {
                return serviceProvider;
            }

            @Override
            public TypeName serviceTypeName() {
                return serviceTypeName;
            }

            @Override
            public List<Annotation> classAnnotations() {
                return classAnnotations;
            }

            @Override
            public TypedElementInfo elementInfo() {
                return elementInfo;
            }

            @Override
            public List<TypedElementInfo> elementArgInfo() {
                return elementArgInfo;
            }

            @Override
            public List<Provider<Interceptor>> interceptors() {
                return interceptors;
            }

            @Override
            public Map<String, Object> contextData() {
                return contextData;
            }

            @Override
            public String toString() {
                return "InvocationContext{"
                        + "serviceProvider=" + serviceProvider + ","
                        + "serviceTypeName=" + serviceTypeName + ","
                        + "classAnnotations=" + classAnnotations + ","
                        + "elementInfo=" + elementInfo + ","
                        + "elementArgInfo=" + elementArgInfo + ","
                        + "interceptors=" + interceptors + ","
                        + "contextData=" + contextData
                        + "}";
            }

            @Override
            public boolean equals(Object o) {
                if (o == this) {
                    return true;
                }
                if (!(o instanceof InvocationContext other)) {
                    return false;
                }
                return Objects.equals(serviceProvider, other.serviceProvider())
                        && Objects.equals(serviceTypeName, other.serviceTypeName())
                        && Objects.equals(classAnnotations, other.classAnnotations())
                        && Objects.equals(elementInfo, other.elementInfo())
                        && Objects.equals(elementArgInfo, other.elementArgInfo())
                        && Objects.equals(interceptors, other.interceptors())
                        && Objects.equals(contextData, other.contextData());
            }

            @Override
            public int hashCode() {
                return Objects.hash(serviceProvider, serviceTypeName, classAnnotations, elementInfo, elementArgInfo, interceptors, contextData);
            }

        }

    }

    /**
     * Fluent API builder for {@link InvocationContext}.
     */
    class Builder extends InvocationContext.BuilderBase<InvocationContext.Builder, InvocationContext> implements io.helidon.common.Builder<InvocationContext.Builder, InvocationContext> {

        private Builder() {
        }

        @Override
        public InvocationContext buildPrototype() {
            preBuildPrototype();
            validatePrototype();
            return new InvocationContextImpl(this);
        }

        @Override
        public InvocationContext build() {
            return buildPrototype();
        }

    }

}
