/*
 * Copyright (c) 2023 Oracle and/or its affiliates.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package io.helidon.inject.api;

import java.time.Duration;
import java.util.Objects;
import java.util.Optional;

import io.helidon.builder.api.Prototype;
import io.helidon.common.Generated;
import io.helidon.common.config.Config;

/**
 * This is the configuration that the Injection service provider uses internally.
 * <p>
 * If left as-is a default configuration instance will be used with default values provided herein. Callers can
 * optionally configure values by providing a {@link Bootstrap#config()} prior to Injection startup. The configuration provided
 * will be used, and tunable configuration must be located under the key {@code inject} within the provided configuration
 * element.
 *
 * @see #builder()
 * @see #create()
 */
@Generated(value = "io.helidon.builder.processor.BlueprintProcessor", trigger = "io.helidon.inject.api.InjectionServicesConfigBlueprint")
public interface InjectionServicesConfig extends InjectionServicesConfigBlueprint, Prototype.Api {

    /**
     * Create a new fluent API builder to customize configuration.
     *
     * @return a new builder
     */
    static InjectionServicesConfig.Builder builder() {
        return new InjectionServicesConfig.Builder();
    }

    /**
     * Create a new fluent API builder from an existing instance.
     *
     * @param instance an existing instance used as a base for the builder
     * @return a builder based on an instance
     */
    static InjectionServicesConfig.Builder builder(InjectionServicesConfig instance) {
        return InjectionServicesConfig.builder().from(instance);
    }

    /**
     * Create a new instance from configuration.
     *
     * @param config used to configure the new instance
     * @return a new instance configured from configuration
     */
    static InjectionServicesConfig create(Config config) {
        return InjectionServicesConfig.builder().config(config).buildPrototype();
    }

    /**
     * Create a new instance with default values.
     *
     * @return a new instance
     */
    static InjectionServicesConfig create() {
        return InjectionServicesConfig.builder().buildPrototype();
    }

    /**
     * Fluent API builder base for {@link InjectionServicesConfig}.
     *
     * @param <BUILDER> type of the builder extending this abstract builder
     * @param <PROTOTYPE> type of the prototype interface that would be built by {@link #buildPrototype()}
     */
    abstract class BuilderBase<BUILDER extends InjectionServicesConfig.BuilderBase<BUILDER, PROTOTYPE>, PROTOTYPE extends InjectionServicesConfig> implements Prototype.ConfiguredBuilder<BUILDER, PROTOTYPE> {

        private Boolean debug;
        private boolean activationLogs = false;
        private boolean permitsDynamic = false;
        private boolean permitsReflection = false;
        private boolean serviceLookupCaching = false;
        private boolean supportsCompileTime = true;
        private boolean supportsContextualLookup = false;
        private boolean supportsDynamic = true;
        private boolean supportsJsr330 = true;
        private boolean supportsJsr330Privates = false;
        private boolean supportsJsr330Statics = false;
        private boolean supportsReflection = false;
        private boolean usesCompileTimeApplications = true;
        private boolean usesCompileTimeModules = true;
        private boolean usesJsr330 = false;
        private Config config;
        private Duration activationDeadlockDetectionTimeout = Duration.parse("PT10S");
        private Duration shutdownTimeout = Duration.parse("PT10S");
        private String providerName;
        private String providerVersion;

        /**
         * Protected to support extensibility.
         */
        protected BuilderBase() {
        }

        /**
         * Update this builder from an existing prototype instance.
         *
         * @param prototype existing prototype to update this builder from
         * @return updated builder instance
         */
        public BUILDER from(InjectionServicesConfig prototype) {
            providerName(prototype.providerName());
            providerVersion(prototype.providerVersion());
            activationDeadlockDetectionTimeout(prototype.activationDeadlockDetectionTimeout());
            shutdownTimeout(prototype.shutdownTimeout());
            activationLogs(prototype.activationLogs());
            serviceLookupCaching(prototype.serviceLookupCaching());
            permitsDynamic(prototype.permitsDynamic());
            supportsDynamic(prototype.supportsDynamic());
            permitsReflection(prototype.permitsReflection());
            supportsReflection(prototype.supportsReflection());
            usesCompileTimeApplications(prototype.usesCompileTimeApplications());
            usesCompileTimeModules(prototype.usesCompileTimeModules());
            supportsCompileTime(prototype.supportsCompileTime());
            usesJsr330(prototype.usesJsr330());
            supportsJsr330(prototype.supportsJsr330());
            supportsJsr330Statics(prototype.supportsJsr330Statics());
            supportsJsr330Privates(prototype.supportsJsr330Privates());
            supportsContextualLookup(prototype.supportsContextualLookup());
            debug(prototype.debug());
            return self();
        }

        /**
         * Update this builder from an existing prototype builder instance.
         *
         * @param builder existing builder prototype to update this builder from
         * @return updated builder instance
         */
        public BUILDER from(InjectionServicesConfig.BuilderBase<?, ?> builder) {
            builder.providerName().ifPresent(this::providerName);
            builder.providerVersion().ifPresent(this::providerVersion);
            activationDeadlockDetectionTimeout(builder.activationDeadlockDetectionTimeout());
            shutdownTimeout(builder.shutdownTimeout());
            activationLogs(builder.activationLogs());
            serviceLookupCaching(builder.serviceLookupCaching());
            permitsDynamic(builder.permitsDynamic());
            supportsDynamic(builder.supportsDynamic());
            permitsReflection(builder.permitsReflection());
            supportsReflection(builder.supportsReflection());
            usesCompileTimeApplications(builder.usesCompileTimeApplications());
            usesCompileTimeModules(builder.usesCompileTimeModules());
            supportsCompileTime(builder.supportsCompileTime());
            usesJsr330(builder.usesJsr330());
            supportsJsr330(builder.supportsJsr330());
            supportsJsr330Statics(builder.supportsJsr330Statics());
            supportsJsr330Privates(builder.supportsJsr330Privates());
            supportsContextualLookup(builder.supportsContextualLookup());
            builder.debug().ifPresent(this::debug);
            return self();
        }

        /**
         * Update builder from configuration (node of this type).
         * If a value is present in configuration, it would override currently configured values.
         *
         * @param config configuration instance used to obtain values to update this builder
         * @return updated builder instance
         */
        @Override
        public BUILDER config(Config config) {
            Objects.requireNonNull(config);
            this.config = config;
            config.get("provider-name").as(String.class).ifPresent(this::providerName);
            config.get("provider-version").as(String.class).ifPresent(this::providerVersion);
            config.get("activation-deadlock-detection-timeout").as(Duration.class).ifPresent(this::activationDeadlockDetectionTimeout);
            config.get("shutdown-timeout").as(Duration.class).ifPresent(this::shutdownTimeout);
            config.get("activation-logs").as(Boolean.class).ifPresent(this::activationLogs);
            config.get("service-lookup-caching").as(Boolean.class).ifPresent(this::serviceLookupCaching);
            config.get("permits-dynamic").as(Boolean.class).ifPresent(this::permitsDynamic);
            config.get("supports-dynamic").as(Boolean.class).ifPresent(this::supportsDynamic);
            config.get("permits-reflection").as(Boolean.class).ifPresent(this::permitsReflection);
            config.get("supports-reflection").as(Boolean.class).ifPresent(this::supportsReflection);
            config.get("uses-compile-time-applications").as(Boolean.class).ifPresent(this::usesCompileTimeApplications);
            config.get("uses-compile-time-modules").as(Boolean.class).ifPresent(this::usesCompileTimeModules);
            config.get("supports-compile-time").as(Boolean.class).ifPresent(this::supportsCompileTime);
            config.get("uses-jsr330").as(Boolean.class).ifPresent(this::usesJsr330);
            config.get("supports-jsr330").as(Boolean.class).ifPresent(this::supportsJsr330);
            config.get("supports-jsr330-statics").as(Boolean.class).ifPresent(this::supportsJsr330Statics);
            config.get("supports-jsr330-privates").as(Boolean.class).ifPresent(this::supportsJsr330Privates);
            config.get("supports-contextual-lookup").as(Boolean.class).ifPresent(this::supportsContextualLookup);
            config.get("debug").as(Boolean.class).ifPresent(this::debug);
            return self();
        }

        /**
         * Clear existing value of this property.
         *
         * @return updated builder instance
         * @see #providerName()
         */
        public BUILDER clearProviderName() {
            this.providerName = null;
            return self();
        }

        /**
         * The provider implementation name.
         *
         * @param providerName the provider implementation name
         * @return updated builder instance
         * @see #providerName()
         */
        public BUILDER providerName(String providerName) {
            Objects.requireNonNull(providerName);
            this.providerName = providerName;
            return self();
        }

        /**
         * Clear existing value of this property.
         *
         * @return updated builder instance
         * @see #providerVersion()
         */
        public BUILDER clearProviderVersion() {
            this.providerVersion = null;
            return self();
        }

        /**
         * The provider implementation version.
         *
         * @param providerVersion the provider implementation version
         * @return updated builder instance
         * @see #providerVersion()
         */
        public BUILDER providerVersion(String providerVersion) {
            Objects.requireNonNull(providerVersion);
            this.providerVersion = providerVersion;
            return self();
        }

        /**
         * The deadlock detection timeout in millis.
         *
         * @param activationDeadlockDetectionTimeout the deadlock detection timeout
         * @return updated builder instance
         * @see #activationDeadlockDetectionTimeout()
         */
        public BUILDER activationDeadlockDetectionTimeout(Duration activationDeadlockDetectionTimeout) {
            Objects.requireNonNull(activationDeadlockDetectionTimeout);
            this.activationDeadlockDetectionTimeout = activationDeadlockDetectionTimeout;
            return self();
        }

        /**
         * Shutdown timeout.
         *
         * @param shutdownTimeout shutdown timeout
         * @return updated builder instance
         * @see #shutdownTimeout()
         */
        public BUILDER shutdownTimeout(Duration shutdownTimeout) {
            Objects.requireNonNull(shutdownTimeout);
            this.shutdownTimeout = shutdownTimeout;
            return self();
        }

        /**
         * Flag indicating whether activation logs are captured, recorded, and retained.
         *
         * @param activationLogs the flag indicating whether activation logs are captured and retained
         * @return updated builder instance
         * @see #activationLogs()
         */
        public BUILDER activationLogs(boolean activationLogs) {
            this.activationLogs = activationLogs;
            return self();
        }

        /**
         * Flag indicating whether service lookups (i.e., via {@link Services#lookup}) are cached.
         *
         * @param serviceLookupCaching the flag indicating whether service lookups are cached
         * @return updated builder instance
         * @see #serviceLookupCaching()
         */
        public BUILDER serviceLookupCaching(boolean serviceLookupCaching) {
            this.serviceLookupCaching = serviceLookupCaching;
            return self();
        }

        /**
         * Flag indicating whether the services registry permits dynamic behavior. The default
         * implementation of Injection supports dynamic (see {@link #supportsDynamic()}), but does not permit it by default.
         *
         * @param permitsDynamic the flag indicating whether the services registry supports dynamic updates of the service registry
         * @return updated builder instance
         * @see #permitsDynamic()
         */
        public BUILDER permitsDynamic(boolean permitsDynamic) {
            this.permitsDynamic = permitsDynamic;
            return self();
        }

        /**
         * Flag indicating whether the services registry supports dynamic behavior. Note that
         * if the provider does not support this flag then permitting it via {@link #permitsDynamic()} will have no affect. The
         * default implementation of Injection supports dynamic, but does not permit it by default.
         *
         * @param supportsDynamic the flag indicating whether the services registry supports dynamic updates of the service registry post
         *                        startup
         * @return updated builder instance
         * @see #supportsDynamic()
         */
        public BUILDER supportsDynamic(boolean supportsDynamic) {
            this.supportsDynamic = supportsDynamic;
            return self();
        }

        /**
         * Flag indicating whether reflection is permitted. The default implementation of Injection
         * supports reflection at compile-time only, and is not controlled by this flag directly.
         *
         * @param permitsReflection the flag indicating whether the provider is permitted to use reflection for normal runtime usage
         * @return updated builder instance
         * @see #permitsReflection()
         */
        public BUILDER permitsReflection(boolean permitsReflection) {
            this.permitsReflection = permitsReflection;
            return self();
        }

        /**
         * Flag indicating whether the reflection is supported. Note that if the provider does not support this
         * flag then permitting it via {@link #permitsReflection()} will have no affect. The default implementation of Injection supports
         * reflection only during compile-time operations using the Injection <i>maven-plugin</i>.
         *
         * @param supportsReflection the flag indicating whether reflection is supported during runtime operations
         * @return updated builder instance
         * @see #supportsReflection()
         */
        public BUILDER supportsReflection(boolean supportsReflection) {
            this.supportsReflection = supportsReflection;
            return self();
        }

        /**
         * Flag indicating whether compile-time generated {@link Application}'s should be used at Injection's startup initialization.
         * Setting
         * this value to false will have no affect if the underlying provider does not support compile-time generation via
         * {@link #supportsCompileTime()}.
         *
         * @param usesCompileTimeApplications the flag indicating whether the provider is permitted to use Application generated code from compile-time
         * @return updated builder instance
         * @see #usesCompileTimeApplications()
         */
        public BUILDER usesCompileTimeApplications(boolean usesCompileTimeApplications) {
            this.usesCompileTimeApplications = usesCompileTimeApplications;
            return self();
        }

        /**
         * Flag indicating whether compile-time generated {@link ModuleComponent}'s should be used at Injection's startup
         * initialization. Setting this value to false will have no affect if the underlying provider does not support compile-time
         * generation via {@link #supportsCompileTime()}.
         *
         * @param usesCompileTimeModules the flag indicating whether the provider is permitted to use Application generated code from compile-time
         * @return updated builder instance
         * @see #usesCompileTimeModules()
         */
        public BUILDER usesCompileTimeModules(boolean usesCompileTimeModules) {
            this.usesCompileTimeModules = usesCompileTimeModules;
            return self();
        }

        /**
         * Flag indicating whether the dependency injection model for the {@link Application} and
         * {@link Activator} is capable for being produced at compile-time, and therefore used/loaded during runtime operations.
         *
         * @param supportsCompileTime the flag indicating whether the provider supports compile-time code generation of DI artifacts
         * @return updated builder instance
         * @see #supportsCompileTime()
         */
        public BUILDER supportsCompileTime(boolean supportsCompileTime) {
            this.supportsCompileTime = supportsCompileTime;
            return self();
        }

        /**
         * Flag indicating whether jsr330 specification will be used and enforced.
         *
         * @param usesJsr330 the flag indicating whether strict jsr330 specification will be enforced
         * @return updated builder instance
         * @see #usesJsr330()
         */
        public BUILDER usesJsr330(boolean usesJsr330) {
            this.usesJsr330 = usesJsr330;
            return self();
        }

        /**
         * Flag indicating whether jsr330 is supported by the provider implementation.
         *
         * @param supportsJsr330 the flag indicating whether the provider supports the jsr330 specification
         * @return updated builder instance
         * @see #supportsJsr330()
         */
        public BUILDER supportsJsr330(boolean supportsJsr330) {
            this.supportsJsr330 = supportsJsr330;
            return self();
        }

        /**
         * Flag indicating whether jsr330 is supported by the provider implementation for the use on static injection points.
         *
         * @param supportsJsr330Statics the flag indicating whether the provider supports the jsr330 specification for the use of static injection points
         * @return updated builder instance
         * @see #supportsJsr330Statics()
         */
        public BUILDER supportsJsr330Statics(boolean supportsJsr330Statics) {
            this.supportsJsr330Statics = supportsJsr330Statics;
            return self();
        }

        /**
         * Flag indicating whether jsr330 is supported by the provider implementation for the use on private injection points.
         *
         * @param supportsJsr330Privates the flag indicating whether the provider supports the jsr330 specification for the use of private injection points
         * @return updated builder instance
         * @see #supportsJsr330Privates()
         */
        public BUILDER supportsJsr330Privates(boolean supportsJsr330Privates) {
            this.supportsJsr330Privates = supportsJsr330Privates;
            return self();
        }

        /**
         * Flag indicating whether contextual lookup is supported via {@link Services#contextualServices(InjectionPointInfo)}.
         *
         * @param supportsContextualLookup the flag indicating whether the provider supports contextual lookup
         * @return updated builder instance
         * @see #supportsContextualLookup()
         */
        public BUILDER supportsContextualLookup(boolean supportsContextualLookup) {
            this.supportsContextualLookup = supportsContextualLookup;
            return self();
        }

        /**
         * Clear existing value of this property.
         *
         * @return updated builder instance
         * @see #debug()
         */
        public BUILDER clearDebug() {
            this.debug = null;
            return self();
        }

        /**
         * Whether debug is enabled.
         * Defaults to false, can be overridden using system property {@link InjectionServices#TAG_DEBUG}.
         *
         * @param debug if debug should be enabled
         * @return updated builder instance
         * @see #debug()
         */
        public BUILDER debug(boolean debug) {
            Objects.requireNonNull(debug);
            this.debug = debug;
            return self();
        }

        /**
         * The provider implementation name.
         *
         * @return the provider name
         */
        public Optional<String> providerName() {
            return Optional.ofNullable(providerName);
        }

        /**
         * The provider implementation version.
         *
         * @return the provider version
         */
        public Optional<String> providerVersion() {
            return Optional.ofNullable(providerVersion);
        }

        /**
         * The deadlock detection timeout in millis.
         *
         * @return the activation deadlock detection timeout
         */
        public Duration activationDeadlockDetectionTimeout() {
            return activationDeadlockDetectionTimeout;
        }

        /**
         * Shutdown timeout.
         *
         * @return the shutdown timeout
         */
        public Duration shutdownTimeout() {
            return shutdownTimeout;
        }

        /**
         * Flag indicating whether activation logs are captured, recorded, and retained.
         *
         * @return the activation logs
         */
        public boolean activationLogs() {
            return activationLogs;
        }

        /**
         * Flag indicating whether service lookups (i.e., via {@link Services#lookup}) are cached.
         *
         * @return the service lookup caching
         */
        public boolean serviceLookupCaching() {
            return serviceLookupCaching;
        }

        /**
         * Flag indicating whether the services registry permits dynamic behavior. The default
         * implementation of Injection supports dynamic (see {@link #supportsDynamic()}), but does not permit it by default.
         *
         * @return the permits dynamic
         */
        public boolean permitsDynamic() {
            return permitsDynamic;
        }

        /**
         * Flag indicating whether the services registry supports dynamic behavior. Note that
         * if the provider does not support this flag then permitting it via {@link #permitsDynamic()} will have no affect. The
         * default implementation of Injection supports dynamic, but does not permit it by default.
         *
         * @return the supports dynamic
         */
        public boolean supportsDynamic() {
            return supportsDynamic;
        }

        /**
         * Flag indicating whether reflection is permitted. The default implementation of Injection
         * supports reflection at compile-time only, and is not controlled by this flag directly.
         *
         * @return the permits reflection
         */
        public boolean permitsReflection() {
            return permitsReflection;
        }

        /**
         * Flag indicating whether the reflection is supported. Note that if the provider does not support this
         * flag then permitting it via {@link #permitsReflection()} will have no affect. The default implementation of Injection supports
         * reflection only during compile-time operations using the Injection <i>maven-plugin</i>.
         *
         * @return the supports reflection
         */
        public boolean supportsReflection() {
            return supportsReflection;
        }

        /**
         * Flag indicating whether compile-time generated {@link Application}'s should be used at Injection's startup initialization.
         * Setting
         * this value to false will have no affect if the underlying provider does not support compile-time generation via
         * {@link #supportsCompileTime()}.
         *
         * @return the uses compile time applications
         */
        public boolean usesCompileTimeApplications() {
            return usesCompileTimeApplications;
        }

        /**
         * Flag indicating whether compile-time generated {@link ModuleComponent}'s should be used at Injection's startup
         * initialization. Setting this value to false will have no affect if the underlying provider does not support compile-time
         * generation via {@link #supportsCompileTime()}.
         *
         * @return the uses compile time modules
         */
        public boolean usesCompileTimeModules() {
            return usesCompileTimeModules;
        }

        /**
         * Flag indicating whether the dependency injection model for the {@link Application} and
         * {@link Activator} is capable for being produced at compile-time, and therefore used/loaded during runtime operations.
         *
         * @return the supports compile time
         */
        public boolean supportsCompileTime() {
            return supportsCompileTime;
        }

        /**
         * Flag indicating whether jsr330 specification will be used and enforced.
         *
         * @return the uses jsr330
         */
        public boolean usesJsr330() {
            return usesJsr330;
        }

        /**
         * Flag indicating whether jsr330 is supported by the provider implementation.
         *
         * @return the supports jsr330
         */
        public boolean supportsJsr330() {
            return supportsJsr330;
        }

        /**
         * Flag indicating whether jsr330 is supported by the provider implementation for the use on static injection points.
         *
         * @return the supports jsr330 statics
         */
        public boolean supportsJsr330Statics() {
            return supportsJsr330Statics;
        }

        /**
         * Flag indicating whether jsr330 is supported by the provider implementation for the use on private injection points.
         *
         * @return the supports jsr330 privates
         */
        public boolean supportsJsr330Privates() {
            return supportsJsr330Privates;
        }

        /**
         * Flag indicating whether contextual lookup is supported via {@link Services#contextualServices(InjectionPointInfo)}.
         *
         * @return the supports contextual lookup
         */
        public boolean supportsContextualLookup() {
            return supportsContextualLookup;
        }

        /**
         * Whether debug is enabled.
         * Defaults to false, can be overridden using system property {@link InjectionServices#TAG_DEBUG}.
         *
         * @return the debug
         */
        public Optional<Boolean> debug() {
            return Optional.ofNullable(debug);
        }

        /**
         * If this instance was configured, this would be the config instance used.
         *
         * @return config node used to configure this builder, or empty if not configured
         */
        public Optional<Config> config() {
            return Optional.ofNullable(config);
        }

        @Override
        public String toString() {
            return "InjectionServicesConfigBuilder{"
                    + "providerName=" + providerName + ","
                    + "providerVersion=" + providerVersion + ","
                    + "activationDeadlockDetectionTimeout=" + activationDeadlockDetectionTimeout + ","
                    + "shutdownTimeout=" + shutdownTimeout + ","
                    + "activationLogs=" + activationLogs + ","
                    + "serviceLookupCaching=" + serviceLookupCaching + ","
                    + "permitsDynamic=" + permitsDynamic + ","
                    + "supportsDynamic=" + supportsDynamic + ","
                    + "permitsReflection=" + permitsReflection + ","
                    + "supportsReflection=" + supportsReflection + ","
                    + "usesCompileTimeApplications=" + usesCompileTimeApplications + ","
                    + "usesCompileTimeModules=" + usesCompileTimeModules + ","
                    + "supportsCompileTime=" + supportsCompileTime + ","
                    + "usesJsr330=" + usesJsr330 + ","
                    + "supportsJsr330=" + supportsJsr330 + ","
                    + "supportsJsr330Statics=" + supportsJsr330Statics + ","
                    + "supportsJsr330Privates=" + supportsJsr330Privates + ","
                    + "supportsContextualLookup=" + supportsContextualLookup + ","
                    + "debug=" + debug
                    + "}";
        }

        /**
         * Handles providers and decorators.
         */
        protected void preBuildPrototype() {
        }

        /**
         * Validates required properties.
         */
        protected void validatePrototype() {
        }

        /**
         * The provider implementation name.
         *
         * @param providerName the provider implementation name
         * @return updated builder instance
         * @see #providerName()
         */
        BUILDER providerName(Optional<String> providerName) {
            Objects.requireNonNull(providerName);
            this.providerName = providerName.map(java.lang.String.class::cast).orElse(this.providerName);
            return self();
        }

        /**
         * The provider implementation version.
         *
         * @param providerVersion the provider implementation version
         * @return updated builder instance
         * @see #providerVersion()
         */
        BUILDER providerVersion(Optional<String> providerVersion) {
            Objects.requireNonNull(providerVersion);
            this.providerVersion = providerVersion.map(java.lang.String.class::cast).orElse(this.providerVersion);
            return self();
        }

        /**
         * Whether debug is enabled.
         * Defaults to false, can be overridden using system property {@link InjectionServices#TAG_DEBUG}.
         *
         * @param debug if debug should be enabled
         * @return updated builder instance
         * @see #debug()
         */
        BUILDER debug(Optional<Boolean> debug) {
            Objects.requireNonNull(debug);
            this.debug = debug.map(java.lang.Boolean.class::cast).orElse(this.debug);
            return self();
        }

        /**
         * Generated implementation of the prototype, can be extended by descendant prototype implementations.
         */
        protected static class InjectionServicesConfigImpl implements InjectionServicesConfig {

            private final boolean activationLogs;
            private final boolean permitsDynamic;
            private final boolean permitsReflection;
            private final boolean serviceLookupCaching;
            private final boolean supportsCompileTime;
            private final boolean supportsContextualLookup;
            private final boolean supportsDynamic;
            private final boolean supportsJsr330;
            private final boolean supportsJsr330Privates;
            private final boolean supportsJsr330Statics;
            private final boolean supportsReflection;
            private final boolean usesCompileTimeApplications;
            private final boolean usesCompileTimeModules;
            private final boolean usesJsr330;
            private final Duration activationDeadlockDetectionTimeout;
            private final Duration shutdownTimeout;
            private final Optional<Boolean> debug;
            private final Optional<String> providerName;
            private final Optional<String> providerVersion;

            /**
             * Create an instance providing a builder.
             *
             * @param builder extending builder base of this prototype
             */
            protected InjectionServicesConfigImpl(InjectionServicesConfig.BuilderBase<?, ?> builder) {
                this.providerName = builder.providerName();
                this.providerVersion = builder.providerVersion();
                this.activationDeadlockDetectionTimeout = builder.activationDeadlockDetectionTimeout();
                this.shutdownTimeout = builder.shutdownTimeout();
                this.activationLogs = builder.activationLogs();
                this.serviceLookupCaching = builder.serviceLookupCaching();
                this.permitsDynamic = builder.permitsDynamic();
                this.supportsDynamic = builder.supportsDynamic();
                this.permitsReflection = builder.permitsReflection();
                this.supportsReflection = builder.supportsReflection();
                this.usesCompileTimeApplications = builder.usesCompileTimeApplications();
                this.usesCompileTimeModules = builder.usesCompileTimeModules();
                this.supportsCompileTime = builder.supportsCompileTime();
                this.usesJsr330 = builder.usesJsr330();
                this.supportsJsr330 = builder.supportsJsr330();
                this.supportsJsr330Statics = builder.supportsJsr330Statics();
                this.supportsJsr330Privates = builder.supportsJsr330Privates();
                this.supportsContextualLookup = builder.supportsContextualLookup();
                this.debug = builder.debug();
            }

            @Override
            public Optional<String> providerName() {
                return providerName;
            }

            @Override
            public Optional<String> providerVersion() {
                return providerVersion;
            }

            @Override
            public Duration activationDeadlockDetectionTimeout() {
                return activationDeadlockDetectionTimeout;
            }

            @Override
            public Duration shutdownTimeout() {
                return shutdownTimeout;
            }

            @Override
            public boolean activationLogs() {
                return activationLogs;
            }

            @Override
            public boolean serviceLookupCaching() {
                return serviceLookupCaching;
            }

            @Override
            public boolean permitsDynamic() {
                return permitsDynamic;
            }

            @Override
            public boolean supportsDynamic() {
                return supportsDynamic;
            }

            @Override
            public boolean permitsReflection() {
                return permitsReflection;
            }

            @Override
            public boolean supportsReflection() {
                return supportsReflection;
            }

            @Override
            public boolean usesCompileTimeApplications() {
                return usesCompileTimeApplications;
            }

            @Override
            public boolean usesCompileTimeModules() {
                return usesCompileTimeModules;
            }

            @Override
            public boolean supportsCompileTime() {
                return supportsCompileTime;
            }

            @Override
            public boolean usesJsr330() {
                return usesJsr330;
            }

            @Override
            public boolean supportsJsr330() {
                return supportsJsr330;
            }

            @Override
            public boolean supportsJsr330Statics() {
                return supportsJsr330Statics;
            }

            @Override
            public boolean supportsJsr330Privates() {
                return supportsJsr330Privates;
            }

            @Override
            public boolean supportsContextualLookup() {
                return supportsContextualLookup;
            }

            @Override
            public Optional<Boolean> debug() {
                return debug;
            }

            @Override
            public String toString() {
                return "InjectionServicesConfig{"
                        + "providerName=" + providerName + ","
                        + "providerVersion=" + providerVersion + ","
                        + "activationDeadlockDetectionTimeout=" + activationDeadlockDetectionTimeout + ","
                        + "shutdownTimeout=" + shutdownTimeout + ","
                        + "activationLogs=" + activationLogs + ","
                        + "serviceLookupCaching=" + serviceLookupCaching + ","
                        + "permitsDynamic=" + permitsDynamic + ","
                        + "supportsDynamic=" + supportsDynamic + ","
                        + "permitsReflection=" + permitsReflection + ","
                        + "supportsReflection=" + supportsReflection + ","
                        + "usesCompileTimeApplications=" + usesCompileTimeApplications + ","
                        + "usesCompileTimeModules=" + usesCompileTimeModules + ","
                        + "supportsCompileTime=" + supportsCompileTime + ","
                        + "usesJsr330=" + usesJsr330 + ","
                        + "supportsJsr330=" + supportsJsr330 + ","
                        + "supportsJsr330Statics=" + supportsJsr330Statics + ","
                        + "supportsJsr330Privates=" + supportsJsr330Privates + ","
                        + "supportsContextualLookup=" + supportsContextualLookup + ","
                        + "debug=" + debug
                        + "}";
            }

            @Override
            public boolean equals(Object o) {
                if (o == this) {
                    return true;
                }
                if (!(o instanceof InjectionServicesConfig other)) {
                    return false;
                }
                return Objects.equals(providerName, other.providerName())
                        && Objects.equals(providerVersion, other.providerVersion())
                        && Objects.equals(activationDeadlockDetectionTimeout, other.activationDeadlockDetectionTimeout())
                        && Objects.equals(shutdownTimeout, other.shutdownTimeout())
                        && activationLogs == other.activationLogs()
                        && serviceLookupCaching == other.serviceLookupCaching()
                        && permitsDynamic == other.permitsDynamic()
                        && supportsDynamic == other.supportsDynamic()
                        && permitsReflection == other.permitsReflection()
                        && supportsReflection == other.supportsReflection()
                        && usesCompileTimeApplications == other.usesCompileTimeApplications()
                        && usesCompileTimeModules == other.usesCompileTimeModules()
                        && supportsCompileTime == other.supportsCompileTime()
                        && usesJsr330 == other.usesJsr330()
                        && supportsJsr330 == other.supportsJsr330()
                        && supportsJsr330Statics == other.supportsJsr330Statics()
                        && supportsJsr330Privates == other.supportsJsr330Privates()
                        && supportsContextualLookup == other.supportsContextualLookup()
                        && Objects.equals(debug, other.debug());
            }

            @Override
            public int hashCode() {
                return Objects.hash(providerName, providerVersion, activationDeadlockDetectionTimeout, shutdownTimeout, activationLogs, serviceLookupCaching, permitsDynamic, supportsDynamic, permitsReflection, supportsReflection, usesCompileTimeApplications, usesCompileTimeModules, supportsCompileTime, usesJsr330, supportsJsr330, supportsJsr330Statics, supportsJsr330Privates, supportsContextualLookup, debug);
            }

        }

    }

    /**
     * Fluent API builder for {@link InjectionServicesConfig}.
     */
    class Builder extends InjectionServicesConfig.BuilderBase<InjectionServicesConfig.Builder, InjectionServicesConfig> implements io.helidon.common.Builder<InjectionServicesConfig.Builder, InjectionServicesConfig> {

        private Builder() {
        }

        @Override
        public InjectionServicesConfig buildPrototype() {
            preBuildPrototype();
            validatePrototype();
            return new InjectionServicesConfigImpl(this);
        }

        @Override
        public InjectionServicesConfig build() {
            return buildPrototype();
        }

    }

}
