/*
 * Copyright (c) 2023 Oracle and/or its affiliates.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package io.helidon.inject.api;

import java.util.Objects;
import java.util.Optional;
import java.util.function.Consumer;
import java.util.function.Supplier;

import io.helidon.builder.api.Prototype;
import io.helidon.common.Errors;
import io.helidon.common.Generated;

/**
 * Combines the {@link ServiceInfo} criteria along with the {@link InjectionPointInfo} context
 * that the query applies to.
 *
 * @see InjectionPointProvider
 * @see #builder()
 * @see #create()
 */
@Generated(value = "io.helidon.builder.processor.BlueprintProcessor", trigger = "io.helidon.inject.api.ContextualServiceQueryBlueprint")
public interface ContextualServiceQuery extends ContextualServiceQueryBlueprint, Prototype.Api {

    /**
     * Create a new fluent API builder to customize configuration.
     *
     * @return a new builder
     */
    static ContextualServiceQuery.Builder builder() {
        return new ContextualServiceQuery.Builder();
    }

    /**
     * Create a new fluent API builder from an existing instance.
     *
     * @param instance an existing instance used as a base for the builder
     * @return a builder based on an instance
     */
    static ContextualServiceQuery.Builder builder(ContextualServiceQuery instance) {
        return ContextualServiceQuery.builder().from(instance);
    }

    /**
     * Create a new instance with default values.
     *
     * @return a new instance
     */
    static ContextualServiceQuery create() {
        return ContextualServiceQuery.builder().buildPrototype();
    }

    /**
     * Creates a contextual service query given the injection point info.
     *
     * @param ipInfo    the injection point info
     * @param expected  true if the query is expected to at least have a single match
     * @return the query
     */
    static ContextualServiceQuery create(InjectionPointInfo ipInfo, boolean expected) {
        return ContextualServiceQueryBlueprint.CustomMethods.create(ipInfo, expected);
    }

    /**
     * Fluent API builder base for {@link ContextualServiceQuery}.
     *
     * @param <BUILDER> type of the builder extending this abstract builder
     * @param <PROTOTYPE> type of the prototype interface that would be built by {@link #buildPrototype()}
     */
    abstract class BuilderBase<BUILDER extends ContextualServiceQuery.BuilderBase<BUILDER, PROTOTYPE>, PROTOTYPE extends ContextualServiceQuery> implements Prototype.Builder<BUILDER, PROTOTYPE> {

        private boolean expected;
        private InjectionPointInfo injectionPointInfo;
        private ServiceInfoCriteria serviceInfoCriteria;

        /**
         * Protected to support extensibility.
         */
        protected BuilderBase() {
        }

        /**
         * Update this builder from an existing prototype instance.
         *
         * @param prototype existing prototype to update this builder from
         * @return updated builder instance
         */
        public BUILDER from(ContextualServiceQuery prototype) {
            serviceInfoCriteria(prototype.serviceInfoCriteria());
            injectionPointInfo(prototype.injectionPointInfo());
            expected(prototype.expected());
            return self();
        }

        /**
         * Update this builder from an existing prototype builder instance.
         *
         * @param builder existing builder prototype to update this builder from
         * @return updated builder instance
         */
        public BUILDER from(ContextualServiceQuery.BuilderBase<?, ?> builder) {
            builder.serviceInfoCriteria().ifPresent(this::serviceInfoCriteria);
            builder.injectionPointInfo().ifPresent(this::injectionPointInfo);
            expected(builder.expected());
            return self();
        }

        /**
         * The criteria to use for the lookup into {@link Services}.
         *
         * @param serviceInfoCriteria the service info criteria
         * @return updated builder instance
         * @see #serviceInfoCriteria()
         */
        public BUILDER serviceInfoCriteria(ServiceInfoCriteria serviceInfoCriteria) {
            Objects.requireNonNull(serviceInfoCriteria);
            this.serviceInfoCriteria = serviceInfoCriteria;
            return self();
        }

        /**
         * The criteria to use for the lookup into {@link Services}.
         *
         * @param consumer consumer of builder for
         *                 the service info criteria
         * @return updated builder instance
         * @see #serviceInfoCriteria()
         */
        public BUILDER serviceInfoCriteria(Consumer<ServiceInfoCriteria.Builder> consumer) {
            Objects.requireNonNull(consumer);
            var builder = ServiceInfoCriteria.builder();
            consumer.accept(builder);
            this.serviceInfoCriteria(builder.build());
            return self();
        }

        /**
         * The criteria to use for the lookup into {@link Services}.
         *
         * @param supplier supplier of
         *                 the service info criteria
         * @return updated builder instance
         * @see #serviceInfoCriteria()
         */
        public BUILDER serviceInfoCriteria(Supplier<? extends ServiceInfoCriteria> supplier) {
            Objects.requireNonNull(supplier);
            this.serviceInfoCriteria(supplier.get());
            return self();
        }

        /**
         * Clear existing value of this property.
         *
         * @return updated builder instance
         * @see #injectionPointInfo()
         */
        public BUILDER clearInjectionPointInfo() {
            this.injectionPointInfo = null;
            return self();
        }

        /**
         * Optionally, the injection point context this search applies to.
         *
         * @param injectionPointInfo the optional injection point context info
         * @return updated builder instance
         * @see #injectionPointInfo()
         */
        public BUILDER injectionPointInfo(InjectionPointInfo injectionPointInfo) {
            Objects.requireNonNull(injectionPointInfo);
            this.injectionPointInfo = injectionPointInfo;
            return self();
        }

        /**
         * Optionally, the injection point context this search applies to.
         *
         * @param consumer the optional injection point context info
         * @return updated builder instance
         * @see #injectionPointInfo()
         */
        public BUILDER injectionPointInfo(Consumer<InjectionPointInfo.Builder> consumer) {
            Objects.requireNonNull(consumer);
            var builder = InjectionPointInfo.builder();
            consumer.accept(builder);
            this.injectionPointInfo(builder.build());
            return self();
        }

        /**
         * Set to true if there is an expectation that there is at least one match result from the search.
         *
         * @param expected true if it is expected there is at least a single match result
         * @return updated builder instance
         * @see #expected()
         */
        public BUILDER expected(boolean expected) {
            this.expected = expected;
            return self();
        }

        /**
         * The criteria to use for the lookup into {@link Services}.
         *
         * @return the service info criteria
         */
        public Optional<ServiceInfoCriteria> serviceInfoCriteria() {
            return Optional.ofNullable(serviceInfoCriteria);
        }

        /**
         * Optionally, the injection point context this search applies to.
         *
         * @return the injection point info
         */
        public Optional<InjectionPointInfo> injectionPointInfo() {
            return Optional.ofNullable(injectionPointInfo);
        }

        /**
         * Set to true if there is an expectation that there is at least one match result from the search.
         *
         * @return the expected
         */
        public boolean expected() {
            return expected;
        }

        @Override
        public String toString() {
            return "ContextualServiceQueryBuilder{"
                    + "serviceInfoCriteria=" + serviceInfoCriteria + ","
                    + "injectionPointInfo=" + injectionPointInfo + ","
                    + "expected=" + expected
                    + "}";
        }

        /**
         * Handles providers and decorators.
         */
        protected void preBuildPrototype() {
        }

        /**
         * Validates required properties.
         */
        protected void validatePrototype() {
            Errors.Collector collector = Errors.collector();
            if (serviceInfoCriteria == null) {
                collector.fatal(getClass(), "Property \"serviceInfoCriteria\" must not be null, but not set");
            }
            collector.collect().checkValid();
        }

        /**
         * Optionally, the injection point context this search applies to.
         *
         * @param injectionPointInfo the optional injection point context info
         * @return updated builder instance
         * @see #injectionPointInfo()
         */
        BUILDER injectionPointInfo(Optional<? extends InjectionPointInfo> injectionPointInfo) {
            Objects.requireNonNull(injectionPointInfo);
            this.injectionPointInfo = injectionPointInfo.map(InjectionPointInfo.class::cast).orElse(this.injectionPointInfo);
            return self();
        }

        /**
         * Generated implementation of the prototype, can be extended by descendant prototype implementations.
         */
        protected static class ContextualServiceQueryImpl implements ContextualServiceQuery {

            private final boolean expected;
            private final Optional<InjectionPointInfo> injectionPointInfo;
            private final ServiceInfoCriteria serviceInfoCriteria;

            /**
             * Create an instance providing a builder.
             *
             * @param builder extending builder base of this prototype
             */
            protected ContextualServiceQueryImpl(ContextualServiceQuery.BuilderBase<?, ?> builder) {
                this.serviceInfoCriteria = builder.serviceInfoCriteria().get();
                this.injectionPointInfo = builder.injectionPointInfo();
                this.expected = builder.expected();
            }

            @Override
            public ServiceInfoCriteria serviceInfoCriteria() {
                return serviceInfoCriteria;
            }

            @Override
            public Optional<InjectionPointInfo> injectionPointInfo() {
                return injectionPointInfo;
            }

            @Override
            public boolean expected() {
                return expected;
            }

            @Override
            public String toString() {
                return "ContextualServiceQuery{"
                        + "serviceInfoCriteria=" + serviceInfoCriteria + ","
                        + "injectionPointInfo=" + injectionPointInfo + ","
                        + "expected=" + expected
                        + "}";
            }

            @Override
            public boolean equals(Object o) {
                if (o == this) {
                    return true;
                }
                if (!(o instanceof ContextualServiceQuery other)) {
                    return false;
                }
                return Objects.equals(serviceInfoCriteria, other.serviceInfoCriteria())
                        && Objects.equals(injectionPointInfo, other.injectionPointInfo())
                        && expected == other.expected();
            }

            @Override
            public int hashCode() {
                return Objects.hash(serviceInfoCriteria, injectionPointInfo, expected);
            }

        }

    }

    /**
     * Fluent API builder for {@link ContextualServiceQuery}.
     */
    class Builder extends ContextualServiceQuery.BuilderBase<ContextualServiceQuery.Builder, ContextualServiceQuery> implements io.helidon.common.Builder<ContextualServiceQuery.Builder, ContextualServiceQuery> {

        private Builder() {
        }

        @Override
        public ContextualServiceQuery buildPrototype() {
            preBuildPrototype();
            validatePrototype();
            return new ContextualServiceQueryImpl(this);
        }

        @Override
        public ContextualServiceQuery build() {
            return buildPrototype();
        }

    }

}
