<doc-view>

<h2 id="_contents">Contents</h2>
<div class="section">
<ul class="ulist">
<li>
<p><router-link to="#_overview" @click.native="this.scrollFix('#_overview')">Overview</router-link></p>

</li>
<li>
<p><router-link to="#maven-coordinates" @click.native="this.scrollFix('#maven-coordinates')">Maven Coordinates</router-link></p>

</li>
<li>
<p><router-link to="#_example" @click.native="this.scrollFix('#_example')">Example</router-link></p>

</li>
<li>
<p><router-link to="#_reference" @click.native="this.scrollFix('#_reference')">Reference</router-link></p>

</li>
</ul>

</div>


<h2 id="_overview">Overview</h2>
<div class="section">
<p>Helidon integrates with <a target="_blank" href="https://projects.eclipse.org/projects/ee4j.tyrus">Tyrus</a> to provide support for the
<a target="_blank" href="https://jakarta.ee/specifications/websocket/2.0/websocket-spec-2.0.html">Jakarta WebSocket API</a>.
The WebSocket API enables Java applications to participate in WebSocket interactions
as both servers and clients. The server API supports two flavors: annotated and
programmatic endpoints.</p>

<p>Annotated endpoints, as suggested by their name, use Java annotations to provide
the necessary meta-data to define WebSocket handlers; programmatic endpoints
implement API interfaces and are annotation free. Annotated endpoints tend to be
more flexible since they allow different method signatures depending on the
application needs, whereas programmatic endpoints must implement an interface
and are, therefore, bounded to its definition.</p>

<p>Helidon SE support is based on the <code>WebSocketRouting</code> class which enables Helidon application to
configure routing for both annotated and programmatic WebSocket endpoints.</p>

</div>


<h2 id="maven-coordinates">Maven Coordinates</h2>
<div class="section">
<p>To enable {feature-name}
add the following dependency to your project&#8217;s <code>pom.xml</code> (see
 <router-link to="/about/managing-dependencies">Managing Dependencies</router-link>).</p>

<markup
lang="xml"

>&lt;dependency&gt;
    &lt;groupId&gt;io.helidon.reactive.webserver&lt;/groupId&gt;
    &lt;artifactId&gt;helidon-reactive-webserver-websocket&lt;/artifactId&gt;
&lt;/dependency&gt;</markup>

</div>


<h2 id="_example">Example</h2>
<div class="section">
<p>This section describes the implementation of a simple application
that uses a REST resource to push messages into a shared queue and a
programmatic WebSocket endpoint to download messages from the queue,
one at a time, over a connection.
The example will show how REST and WebSocket connections can
be seamlessly combined into a Helidon application.</p>

<p>The complete Helidon SE example is available <a target="_blank" href="https://github.com/oracle/helidon/tree/master/examples/webserver/websocket">here</a>. Let us start by
looking at <code>MessageQueueService</code>:</p>

<markup
lang="java"

>public class MessageQueueService implements Service {

    private final MessageQueue messageQueue = MessageQueue.instance();

    @Override
    public void update(Routing.Rules routingRules) {
        routingRules.post("/board", this::handlePost);
    }

    private void handlePost(ServerRequest request, ServerResponse response) {
        request.content()
                .as(String.class)
                .thenAccept(it -&gt; {
                    messageQueue.push(it);
                    response.status(204).send();
                });
    }
}</markup>

<p>This class exposes a REST resource where messages can be posted. Upon
receiving a message, it simply pushes it into a shared queue and
returns 204 (No Content).</p>

<p>Messages pushed into the queue can be obtained by opening a WebSocket
connection served by <code>MessageBoardEndpoint</code>:</p>

<markup
lang="java"

>public class MessageBoardEndpoint extends Endpoint {

    private final MessageQueue messageQueue = MessageQueue.instance();

    @Override
    public void onOpen(Session session, EndpointConfig endpointConfig) {
        session.addMessageHandler(new MessageHandler.Whole&lt;String&gt;() {
            @Override
            public void onMessage(String message) {
                try {
                    // Send all messages in the queue
                    if (message.equals("SEND")) {
                        while (!messageQueue.isEmpty()) {
                            session.getBasicRemote().sendObject(messageQueue.pop());
                        }
                    }
                } catch (Exception e) {
                    // handle exception
                }
            }
        });
    }
}</markup>

<p>This is an example of a programmatic endpoint that extends <code>Endpoint</code>. The method
<code>onOpen</code> will be invoked for every new connection. In this example, the application
registers a message handler for strings, and when the special <code>SEND</code> message
is received, it empties the shared queue sending messages one at a time over
the WebSocket connection.</p>

<p>In Helidon SE, REST and WebSocket classes need to be manually registered into
the web server. This is accomplished via a <code>Routing</code> builder:</p>

<markup
lang="java"

>List&lt;Class&lt;? extends Encoder&gt;&gt; encoders =
    Collections.singletonList(UppercaseEncoder.class);

WebServer server = WebServer.builder()
        .port(8080)
        .routing(r -&gt; r
                .register("/web", StaticContentSupport.builder("/WEB")
                        .welcomeFileName("index.html")
                        .build())
                .register("/rest", new MessageQueueService())
        )
        .addRouting(WebSocketRouting.builder()
                .endpoint("/websocket", ServerEndpointConfig.Builder.create(MessageBoardEndpoint.class, "/board")
                        .encoders(encoders)
                        .build())
                .build()
        )
        .build()</markup>

<p>This code snippet uses multiple builders for <code>Routing</code>, <code>WebSocketRouting</code> and <code>ServerEndpointConfig</code>.
In particular, it registers <code>MessageBoardEndpoint.class</code> at <code>"/websocket/board"</code> and associates
with it a <em>message encoder</em>. For more information on message encoders and decoders the
reader see the <a target="_blank" href="https://jakarta.ee/specifications/websocket/2.0/websocket-spec-2.0.html">websocket specification</a>; in this
 <a target="_blank" href="https://github.com/oracle/helidon/tree/master/examples/webserver/websocket">example</a>, <code>UppercaseEncoder.class</code> simply uppercases every
 message sent from the server.</p>

<p>Endpoint methods in Helidon SE are executed in Netty&#8217;s worker thread pool. Threads in this
pool are intended to be <em>non-blocking</em>, thus it is recommended for any blocking or
long-running operation triggered by an endpoint method to be executed using a separate
thread pool. See the documentation for <code>io.helidon.common.configurable.ThreadPoolSupplier</code>.</p>

</div>


<h2 id="_reference">Reference</h2>
<div class="section">
<ul class="ulist">
<li>
<p><a target="_blank" href="./apidocs/io.helidon.reactive.webserver.websocket/module-summary.html">Helidon WebSocket JavaDoc</a></p>

</li>
<li>
<p><a target="_blank" href="https://jakarta.ee/specifications/websocket/2.0/websocket-spec-2.0.html">Jakarta WebSocket Specification</a></p>

</li>
</ul>

</div>

</doc-view>
