<doc-view>

<v-layout row wrap>
<v-flex xs12 sm10 lg10>
<v-card class="section-def" v-bind:color="$store.state.currentColor">
<v-card-text class="pa-3">
<v-card class="section-def__card">
<v-card-text>
<dl>
<dt slot=title>Helidon SE DB Client Guide</dt>
<dd slot="desc"><p>This guide describes Helidon DB Client features and how to create a sample Helidon SE project
that can be used to run some basic example using helidon db client.</p>
</dd>
</dl>
</v-card-text>
</v-card>
</v-card-text>
</v-card>
</v-flex>
</v-layout>


<h2 id="_what_you_need">What You Need</h2>
<div class="section">
<p>For this 15 minute tutorial, you will need the following:</p>


<div class="table__overflow elevation-1  flex sm7
">
<table class="datatable table">
<colgroup>
<col style="width: 50%;">
<col style="width: 50%;">
</colgroup>
<thead>
</thead>
<tbody>
<tr>
<td class="">A Helidon SE Application</td>
<td class="">You can use your own application or use the
 <router-link to="/se/guides/quickstart">Helidon SE Quickstart</router-link> to create a sample application.</td>
</tr>
<tr>
<td class=""><a target="_blank" href="https://www.oracle.com/technetwork/java/javase/downloads">Java&#160;SE&#160;17</a> (<a target="_blank" href="http://jdk.java.net">Open&#160;JDK&#160;17</a>)</td>
<td class="">Helidon requires Java 17+.</td>
</tr>
<tr>
<td class=""><a target="_blank" href="https://maven.apache.org/download.cgi">Maven 3.6.1+</a></td>
<td class="">Helidon requires Maven 3.6.1+.</td>
</tr>
<tr>
<td class=""><a target="_blank" href="https://docs.docker.com/install/">Docker 18.09+</a></td>
<td class="">You need Docker if you
want to build and deploy Docker containers.</td>
</tr>
<tr>
<td class=""><a target="_blank" href="https://kubernetes.io/docs/tasks/tools/install-kubectl/">Kubectl 1.16.5+</a></td>
<td class="">If you want to
deploy to Kubernetes, you need <code>kubectl</code> and a Kubernetes cluster (you can
<router-link to="/about/kubernetes">install one on your desktop</router-link>.</td>
</tr>
</tbody>
</table>
</div>

<markup
lang="bash"
title="Verify Prerequisites"
>java -version
mvn --version
docker --version
kubectl version --short</markup>

<markup
lang="bash"
title="Setting JAVA_HOME"
># On Mac
export JAVA_HOME=`/usr/libexec/java_home -v 17`

# On Linux
# Use the appropriate path to your JDK
export JAVA_HOME=/usr/lib/jvm/jdk-17</markup>

</div>


<h2 id="_introduction">Introduction</h2>
<div class="section">
<p>The Helidon DB Client offers a simple and easy way to work with databases in reactive application.
In Helidon 2.0.0 we introduced the pre-release version of Helidon DB Client — a unified,
reactive API for working with databases in non-blocking way. Helidon team constantly updates this API in order
to make it more user-friendly and efficient. For this matter, it is recommended to use the latest helidon version.</p>


<h3 id="_main_features">Main Features</h3>
<div class="section">
<p>The main features of  Helidon DB Client are:</p>

<ul class="ulist">
<li>
<p><strong>Unified API for data access and query</strong>:
The API was implemented as a layer above JDBC or MongoDB Reactive Streams Java Driver, so any relational databases
with JDBC driver or MongoDB are supported.</p>

</li>
<li>
<p><strong>Reactive database access with non-reactive drivers</strong>:
Most JDBC drivers are blocking. Using them in a reactive application is problematic. Helidon DB Client allows the use
of blocking JDBC drivers in your reactive application by wrapping a blocking driver in an executor service.</p>

</li>
<li>
<p><strong>Observability</strong>:
Support for health checks, metrics and tracing was added to all Helidon SE APIs and the Helidon DB Client
is no exception.</p>

</li>
<li>
<p><strong>Backpressure</strong>:
Helidon DB Client performs database operations only when it is requested by the consumer.
This is propagated all the way to the TCP layer.</p>

</li>
<li>
<p><strong>Portability between relational database drivers</strong>:
The DB Client works with native database statements that can be used inline in the code or defined as named statements
in database configuration. By moving the native query code to configuration files, the Helidon DB Client allows you to
switch to another database by changing the configuration files, not the code.</p>

</li>
</ul>

</div>

</div>


<h2 id="_getting_started_with_helidon_db_client">Getting Started with Helidon DB Client</h2>
<div class="section">
<p>This section describes how to configure and use the key features of the Helidon DB Client.</p>


<h3 id="_set_up_h2_database">Set up H2 database</h3>
<div class="section">

<h4 id="_from_docker">From Docker</h4>
<div class="section">
<p>Create a new file in <code>helidon-quickstart-se</code> named <code>Dockerfile.h2</code>. It will be used to create the H2 docker image
to run H2 in a container.</p>

<markup
lang="dockerfile"
title="Write the following content into the new file created"
>FROM openjdk:11-jre-slim

ENV H2_VERSION "1.4.199"

ADD "https://repo1.maven.org/maven2/com/h2database/h2/${H2_VERSION}/h2-${H2_VERSION}.jar" /opt/h2.jar

COPY h2.server.properties /root/.h2.server.properties

EXPOSE 8082
EXPOSE 9092

CMD java \
       -cp /opt/h2.jar \
       org.h2.tools.Server \
       -web -webDaemon -webAllowOthers -webPort 8082 \
       -tcp -tcpAllowOthers -tcpPort 9092 \
       -ifNotExists</markup>

<p>Create a new file <code>h2.server.properties</code> in the current directory.</p>

<markup
lang="properties"
title="Copy the properties into the properties file."
>webSSL=false
webAllowOthers=true
webPort=8082
0=Generic H2 (Server)|org.h2.Driver|jdbc\:h2\:tcp\://localhost\:9092/~/test|sa</markup>

<markup
lang="bash"
title="Build the H2 docker image"
>docker build -f Dockerfile.h2 . -t h2db</markup>

<markup
lang="bash"
title="Run the H2 docker image"
>docker run --rm -p 8082:8082 -p 9092:9092 --name=h2 h2db</markup>

</div>


<h4 id="_from_command_line">From Command Line</h4>
<div class="section">
<p>A database stores the books from the library. H2 is a java SQL database easy to use and light.
If H2 is not installed on your machine, here are few steps to quickly download and set it up:</p>

<ul class="ulist">
<li>
<p>Download the latest H2 version from the official website: <a target="_blank" href="https://www.h2database.com/html/main.html" class="bare">https://www.h2database.com/html/main.html</a></p>
<ul class="ulist">
<li>
<p>On windows, let guides yourself by the H2 installer.</p>

</li>
<li>
<p>On another platform, unzip the downloaded file into your wished directory.</p>

</li>
<li>
<p>Only the h2-{latest-version}.jar, located into h2/bin folder, will be needed.</p>

</li>
</ul>

</li>
<li>
<p>Run the following command to start H2, do not forget to replace <code>{latest-version}</code> by your current version.</p>

</li>
</ul>

<markup
lang="bash"
title="Open a terminal and enter the following:"
>java -jar h2-\{latest-version}.jar -webAllowOthers -tcpAllowOthers</markup>

</div>


<h4 id="_run_h2">Run H2</h4>
<div class="section">
<p>This output should be printed:</p>

<markup
lang="bash"
title="Terminal output"
>Web Console server running at http://127.0.1.1:8082 (others can connect)
Opening in existing browser session.
TCP server running at tcp://127.0.1.1:9092 (others can connect)
PG server running at pg://127.0.1.1:5435 (only local connections)</markup>

<p>Open the console at <a target="_blank" href="http://127.0.1.1:8082" class="bare">http://127.0.1.1:8082</a> in your favorite browser. It displays a login window.
Select <code>Generic H2 (Server)</code> from <code>Saved Settings</code>. Here is the list of setting, all of them should be set by default.</p>

<ul class="ulist">
<li>
<p>Driver Class: org.h2.Driver</p>

</li>
<li>
<p>JDBC URL: jdbc:h2:tcp://localhost:9092/~/test</p>

</li>
<li>
<p>User Name: sa</p>

</li>
<li>
<p>Password:</p>

</li>
</ul>

<p>Password must stay empty. Press connect, the browser displays a web page. The database is correctly set and running.</p>

</div>

</div>


<h3 id="_create_a_sample_se_project">Create a sample SE project</h3>
<div class="section">

<h4 id="_using_maven_archetype">Using maven archetype</h4>
<div class="section">
<p>Generate the project sources using the Helidon SE Maven archetype.
The result is a simple project that can be used for the examples in this guide.</p>

<markup
lang="bash"
title="Run the Maven archetype:"
>mvn -U archetype:generate -DinteractiveMode=false \
    -DarchetypeGroupId=io.helidon.archetypes \
    -DarchetypeArtifactId=helidon-quickstart-se \
    -DarchetypeVersion=4.0.0-ALPHA6 \
    -DgroupId=io.helidon.examples \
    -DartifactId=helidon-quickstart-se \
    -Dpackage=io.helidon.examples.quickstart.se</markup>

<p>A new directory named <code>helidon-quickstart-se</code> is created.</p>

<markup
lang="bash"
title="Enter into this directory:"
>cd helidon-quickstart-se</markup>

</div>

</div>


<h3 id="_add_new_dependency">Add new dependency</h3>
<div class="section">
<p>First thing to do is to open the pom.xml file and add helidon dependencies to use the DB Client.</p>

<markup
lang="xml"
title="Copy those dependencies:"
>&lt;dependencies&gt;
    &lt;dependency&gt;
        &lt;groupId&gt;io.helidon.reactive.dbclient&lt;/groupId&gt;       <span class="conum" data-value="1" />
        &lt;artifactId&gt;helidon-reactive-dbclient&lt;/artifactId&gt;
    &lt;/dependency&gt;
    &lt;dependency&gt;
        &lt;groupId&gt;io.helidon.reactive.dbclient&lt;/groupId&gt;      <span class="conum" data-value="2" />
        &lt;artifactId&gt;helidon-reactive-dbclient-jdbc&lt;/artifactId&gt;
    &lt;/dependency&gt;
    &lt;dependency&gt;
        &lt;groupId&gt;io.helidon.integrations.db&lt;/groupId&gt;   <span class="conum" data-value="3" />
        &lt;artifactId&gt;h2&lt;/artifactId&gt;
    &lt;/dependency&gt;
    &lt;dependency&gt;
        &lt;groupId&gt;io.helidon.reactive.dbclient&lt;/groupId&gt;      <span class="conum" data-value="4" />
        &lt;artifactId&gt;helidon-reactive-dbclient-health&lt;/artifactId&gt;
    &lt;/dependency&gt;
    &lt;dependency&gt;
        &lt;groupId&gt;io.helidon.reactive.dbclient&lt;/groupId&gt;      <span class="conum" data-value="5" />
        &lt;artifactId&gt;helidon-reactive-dbclient-metrics&lt;/artifactId&gt;
    &lt;/dependency&gt;
    &lt;dependency&gt;
        &lt;groupId&gt;io.helidon.reactive.dbclient&lt;/groupId&gt;      <span class="conum" data-value="6" />
        &lt;artifactId&gt;helidon-reactive-dbclient-jsonp&lt;/artifactId&gt;
    &lt;/dependency&gt;
&lt;/dependencies&gt;</markup>

<ul class="colist">
<li data-value="1">DB Client API dependency.</li>
<li data-value="2">Using JDBC driver for this example.</li>
<li data-value="3">H2 driver dependency.</li>
<li data-value="4">Support for health check.</li>
<li data-value="5">Support for metrics.</li>
<li data-value="6">Support for Jsonp.</li>
</ul>

</div>


<h3 id="_configure_the_db_client">Configure the DB Client</h3>
<div class="section">
<p>To configure the application, helidon uses by default the application.yaml. The DB client configuration can be joined
in the same file. It is located here: <code>src/main/resources</code>.</p>

<markup
lang="yaml"
title="Copy those properties into application.yaml"
>db:
  source: jdbc  <span class="conum" data-value="1" />
  connection:   <span class="conum" data-value="2" />
    url: "jdbc:h2:tcp://localhost:9092/~/test"
    username: "sa"
    password:
  statements:   <span class="conum" data-value="3" />
    create-table: "CREATE TABLE IF NOT EXISTS LIBRARY (NAME VARCHAR NOT NULL, INFO VARCHAR NOT NULL)"
    insert-book: "INSERT INTO LIBRARY (NAME, INFO) VALUES (:name, :info)"
    select-book: "SELECT INFO FROM LIBRARY WHERE NAME = ?"
    delete-book: "DELETE FROM LIBRARY WHERE NAME = ?"</markup>

<ul class="colist">
<li data-value="1">Source property support two values: jdbc and mongo.</li>
<li data-value="2">Connection detail we used to set up H2.</li>
<li data-value="3">SQL statements to manage the database.</li>
</ul>

</div>


<h3 id="_build_and_set_up_helidon_db_client">Build and set up Helidon DB Client</h3>
<div class="section">
<p>In the application Main.class, an instance of DbClient is created based on the configuration from
application.yaml.</p>

<markup
lang="java"
title="Create a DbClient in the Main.startServer method:"
>import io.helidon.reactive.dbclient.metrics.DbClientMetrics; <span class="conum" data-value="1" />
import io.helidon.reactive.dbclient.DbClient;

Config config = Config.create(); // Landmark to add DB client

DbClient dbClient = DbClient.builder()
                .config(config.get("db"))       <span class="conum" data-value="2" />
                .addService(DbClientMetrics.counter().statementNames("select-book"))   <span class="conum" data-value="3" />
                .build();</markup>

<ul class="colist">
<li data-value="1">Add import statements</li>
<li data-value="2">Configure the DB Client with the "db" section of application.yaml.</li>
<li data-value="3">Add a counter for metrics.</li>
</ul>

<p>The DB Client metric counter will be executed only for <code>select-book</code> statement, and check how many times it was invoked.
At this point, the database is empty, and needs to be initialised. To achieve that, the DB Client can be used
to create a table in the database.</p>

<markup
lang="java"
title="Insert a createTable method below the dbClient:"
>DbClient dbClient = DbClient.builder()
                .config(config.get("db"))
                .addService(DbClientMetrics.counter().statementNames("select-book"))
                .build();

createTable(dbClient);</markup>

<markup
lang="java"
title="Use the DbClient to build a table:"
>private static void createTable(DbClient dbClient) {
    dbClient.execute(exec -&gt; exec.namedDml("create-table"))     <span class="conum" data-value="1" />
            .await();
}</markup>

<ul class="colist">
<li data-value="1">Use the "create-table" script to build a table with book name and information.</li>
</ul>

<p>The <code>createTable</code> is invoked only once and create an empty table with two columns: name and info. It is used to boostrap
the server, so the <code>await</code> method is called in this particular case because the table must be created before the server
starts. A new service can manage request to interact with this table which represent our library.The services are
registered in the <code>createRouting</code> method.</p>

<markup
lang="java"
title="Modify the createRouting method:"
>import io.helidon.reactive.dbclient.health.DbClientHealthCheck;

WebServer server = WebServer.builder(createRouting(config, dbClient))   <span class="conum" data-value="1" />
                .config(config.get("server"))
                .addMediaSupport(JsonpSupport.create())
                .build();

private static Routing createRouting(Config config, DbClient dbClient) {
    HealthSupport health = HealthSupport.builder()
            .addLiveness(DbClientHealthCheck.create(dbClient))  <span class="conum" data-value="2" />
            .build();

    return Routing.builder()
            .register(health)                   // Health at "/health"
            .register(MetricsSupport.create())                  // Metrics at "/metrics"
            .register("/greet", new GreetService(config))
            .register("/library", new LibraryService(dbClient))  <span class="conum" data-value="3" />
            .build();
}</markup>

<ul class="colist">
<li data-value="1">Add dbClient as a parameter of createRouting method.</li>
<li data-value="2">Add Health check to control the application behavior.</li>
<li data-value="3">Register the LibraryService to the Routing.</li>
</ul>

<p>The library service does not exist yet but that is the next step of the guide. It has a constructor with the
DB Client as a parameter because it will manage the library. The DB Client health check uses the <code>select-book</code> statement
from the configuration. As shown above, to create a DB Client health check, call the <code>DbClientHealthCheck.create</code> method
and pass the concerned DbClient. Then add it to the health support builder and register it to the routing.</p>

</div>


<h3 id="_create_the_library_service">Create the Library service</h3>
<div class="section">
<p>Create LibraryService class into <code>io.helidon.examples.quickstart.se</code> package.</p>

<markup
lang="java"
title="LibraryService class looks like this:"
>package io.helidon.examples.quickstart.se;

import io.helidon.common.http.Http;     <span class="conum" data-value="1" />
import io.helidon.reactive.dbclient.DbClient;
import io.helidon.reactive.webserver.Routing;
import io.helidon.reactive.webserver.ServerRequest;
import io.helidon.reactive.webserver.ServerResponse;
import io.helidon.reactive.webserver.Service;

public class LibraryService implements Service {

    private final DbClient dbClient;    <span class="conum" data-value="2" />

    LibraryService(DbClient pDbClient){
        this.dbClient = pDbClient;      <span class="conum" data-value="3" />
    }
}</markup>

<ul class="colist">
<li data-value="1">Add new import statement</li>
<li data-value="2">Declare the Helidon DB Client</li>
<li data-value="3">A DB Client instance is provided when LibraryService is instantiated.</li>
</ul>

<p>As the LibraryService implements <code>io.helidon.reactive.webserver.Service</code>, the <code>update(Routing)</code> method has to be implemented.
It defines application endpoints and Http request which can be reached by clients.</p>

<markup
lang="java"
title="Add update method to LibraryService"
>@Override
public void update(Routing.Rules rules) {
    rules
        .get("/{name}", this::getBook)      <span class="conum" data-value="1" />
        .put("/{name}", this::addBook)      <span class="conum" data-value="2" />
        .delete("/{name}", this::deleteBook)   <span class="conum" data-value="3" />
        .get("/json/{name}", this::getJsonBook); <span class="conum" data-value="4" />
}</markup>

<ul class="colist">
<li data-value="1">Return information about the required book from the database.</li>
<li data-value="2">Add a book to the library.</li>
<li data-value="3">Remove a book from the library.</li>
<li data-value="4">Return the book information in Json format.</li>
</ul>

<p>To summarise, there is one endpoint which can manipulate books. The number of endpoints and application
features can be changed from these rules by creating or modifying methods. <code>{name}</code> is a path parameter for the book
name. The architecture of the application is defined, so next step is to create these features.</p>

<markup
lang="java"
title="Add getBook to the LibraryService:"
>private void getBook(ServerRequest serverRequest, ServerResponse serverResponse) {
    String bookName = serverRequest.path().param("name");   <span class="conum" data-value="1" />

    dbClient.execute(exec -&gt; exec.namedGet("select-book", bookName))    <span class="conum" data-value="2" />
            .thenAccept(row -&gt; {
                if (row.isPresent()) {
                    serverResponse.send(row.get().column("INFO").as(String.class)); <span class="conum" data-value="3" />
                } else {
                    serverResponse.status(Http.Status.NOT_FOUND_404)    <span class="conum" data-value="4" />
                            .send();
                }
            })
            .exceptionally(serverResponse::send);   <span class="conum" data-value="5" />
}</markup>

<ul class="colist">
<li data-value="1">Get the book name from the path in the url.</li>
<li data-value="2">Helidon db Client execute the <code>select-book</code> SQL script from application.yaml.</li>
<li data-value="3">Send book information to the client.</li>
<li data-value="4">Send 404 Http status if no book was found for the given name.</li>
<li data-value="5">If an exception occurred during the process, it is sent to the client.</li>
</ul>

<p>The <code>getBook</code> method reach the book from the database and send the information to the client. The name of the book is
located into the url path. If the book is not present in the database, a HTTP 404 is sent back.
The <code>execute(Function&lt;DbExecute, T&gt; executor)</code> method is called on the dbClient instance to execute one statement.
Nevertheless, it is possible to execute a set of tasks into a single execution unit by using
 <code>inTransaction (Function&lt;DbTransaction, T&gt; executor)</code> method.</p>

<p>DbExecute class provides many builders to create statements such as, DML, insert, update, delete, query and get
statements. For each statement there are two builders which can be regrouped in 2 categories. Builders with methods
containing <code>Named</code> keyword, they use a statement defined in the configuration file.</p>

<p>And builders without <code>Named</code> keyword, they use a statement passed as an argument. More information on the Helidon DB
Client <router-link to="/se/dbclient">here</router-link>.</p>

<markup
lang="java"
title="Add getJsonBook to the LibraryService:"
>private void getJsonBook(ServerRequest serverRequest, ServerResponse serverResponse) {
    String bookName = serverRequest.path().param("name");

    dbClient.execute(exec -&gt; exec.namedGet("select-book", bookName))
            .thenAccept(row -&gt; {
                if (row.isPresent()) {
                    serverResponse.send(row.get().as(JsonObject.class));
                } else {
                    serverResponse.status(Http.Status.NOT_FOUND_404)
                            .send();
                }
            })
            .exceptionally(serverResponse::send);
}</markup>

<p>Instead of sending the <code>INFO</code> content of the targeted book, the <code>getJsonBook</code> method send the whole row of the
database as a <code>JsonObject</code>.</p>

<markup
lang="java"
title="Add addBook to the LibraryService:"
>private void addBook(ServerRequest serverRequest, ServerResponse serverResponse) {
    String bookName = serverRequest.path().param("name");

    serverRequest.content()
            .as(String.class)
            .thenAccept(newValue -&gt; {
                dbClient.execute(exec -&gt; exec.createNamedInsert("insert-book")
                        .addParam("name", bookName)     <span class="conum" data-value="1" />
                        .addParam("info", newValue)
                        .execute())
                        .thenAccept(count -&gt; serverResponse.status(Http.Status.CREATED_201).send())     <span class="conum" data-value="2" />
                        .exceptionally(serverResponse::send);
            });
}</markup>

<ul class="colist">
<li data-value="1">The SQL statement requires the book name and its information. They are provided with <code>addParam</code> method.</li>
<li data-value="2">A new book was added to library, so a HTTP 201 code is returned.</li>
</ul>

<p>When a user adds a new book, it uses HTTP PUT method where the book name is in the URL and the information in the
request content. To catch this content, the information is retrieved as a string and then the DB Client execute the
<code>insert-book</code> script to add the book to the library. It requires two parameters, the book name and information which are
passed to the dbClient thanks to <code>addParam</code> method. A HTTP 201 is sent back as a confirmation if no exception is thrown.</p>

<markup
lang="java"
title="Add deleteBook to LibraryService:"
>private void deleteBook(ServerRequest serverRequest, ServerResponse serverResponse) {
    String bookName = serverRequest.path().param("name");

    dbClient.execute(exec -&gt;  exec.namedDelete("delete-book", bookName))     <span class="conum" data-value="1" />
                        .thenAccept(count -&gt; serverResponse.status(Http.Status.NO_CONTENT_204).send())   <span class="conum" data-value="2" />
                        .exceptionally(serverResponse::send);
}</markup>

<ul class="colist">
<li data-value="1">Execute SQL script from application.yaml to remove a book from the library by its name.</li>
<li data-value="2">The required book was removed, so a HTTP 204 is sent.</li>
</ul>

<p>To remove a book from the library, use the "delete-book" script in the way than previously. If the book is removed
successfully, a HTTP 204 is sent back.</p>

</div>

</div>


<h2 id="_build_and_run_the_library_application">Build and Run the Library application</h2>
<div class="section">
<p>The application is ready to be built and ran.</p>

<markup
lang="bash"
title="Run the following to build the application:"
>mvn package</markup>

<p>Note that the tests are passing as the GreetService process was not modified. For the purposes of this demonstration,
we only added independent new content to the existing application.
Make sure H2 is running and start the Helidon quickstart with this command:</p>

<markup
lang="bash"
title="Run the application"
>java -jar target/helidon-quickstart-se.jar</markup>

<p>Once the application starts, check the table LIBRARY is created in the H2 database. To do so, go to the
H2 Server console and LIBRARY table should be present in the left column under <code>jdbc:h2:tcp://localhost:9092/~/test</code>.
If it is not, try to refresh the page, and it should appear.</p>

<p>Use <code>curl</code> to send request to the application:</p>

<markup
lang="bash"
title="Get a book from the library"
>curl -i http://localhost:8080/library/SomeBook</markup>

<markup
lang="listing"
title="HTTP response"
>HTTP/1.1 404 Not Found
Date: Tue, 12 Jan 2021 14:00:48 +0100
transfer-encoding: chunked
connection: keep-alive</markup>

<p>There is currently no book inside the library, so the application returns a 404. Yet the application created an empty
library table. Try to add a new book.</p>

<markup
lang="bash"
title="Add a book from the library"
>curl -i -X PUT -d "Fantasy" http://localhost:8080/library/HarryPotter</markup>

<markup
lang="listing"
title="HTTP response"
>HTTP/1.1 201 Created
Date: Tue, 12 Jan 2021 14:01:08 +0100
transfer-encoding: chunked
connection: keep-alive</markup>

<p>This command creates an HTTP PUT request with the genre <code>Fantasy</code> content at the address
<a target="_blank" href="http://localhost:8080/library/{book-name}" class="bare">http://localhost:8080/library/{book-name}</a>. The 201 code means that Harry Potter book was successfully added to
the library. You can now try to get it !</p>

<markup
lang="bash"
title="Get Harry Potter from the library"
>curl -i http://localhost:8080/library/HarryPotter</markup>

<markup
lang="listing"
title="HTTP response"
>HTTP/1.1 200 OK
Content-Type: text/plain
Date: Tue, 12 Jan 2021 14:01:14 +0100
connection: keep-alive
content-length: 6

Fantasy</markup>

<p>The application accepted the request and returned an HTTP 200 OK with the book genre that was added earlier.</p>

<markup
lang="bash"
title="Get Harry Potter from the library in Json"
>curl -i http://localhost:8080/library/json/HarryPotter</markup>

<markup
lang="listing"
title="HTTP response"
>HTTP/1.1 200 OK
Content-Type: text/plain
Date: Tue, 12 Jan 2021 14:01:14 +0100
connection: keep-alive
content-length: 6

{"INFO":"Fantasy"}</markup>

<p>It returns the database row in a Json format for Harry Potter book.
Harry Potter can be removed from the library with the following:</p>

<markup
lang="bash"
title="Remove Harry Potter from the library"
>curl -i -X DELETE http://localhost:8080/library/HarryPotter</markup>

<markup
lang="listing"
title="HTTP response"
>HTTP/1.1 204 No Content
Date: Tue, 12 Jan 2021 14:01:22 +0100
connection: keep-alive</markup>

<p>The book had been removed from the library and confirmed by the 204 HTTP status. To check that the book was correctly
deleted, try to get it again.</p>

<markup
lang="bash"
title="Get Harry Potter from the library"
>curl -i http://localhost:8080/library/HarryPotter</markup>

<markup
lang="listing"
title="HTTP response"
>HTTP/1.1 404 Not Found
Date: Tue, 12 Jan 2021 14:00:48 +0100
transfer-encoding: chunked
connection: keep-alive</markup>

<p>The book is not found. We quickly checked, thanks to this suite of command, the application behavior.</p>

<markup
lang="bash"
title="Check the health of your application:"
>curl http://localhost:8080/health</markup>

<markup
lang="json"
title="Response body"
>{
  "state" : "UP",
  "status" : "UP",
  "name" : "jdbc:h2"
}</markup>

<p>It confirms that the database is UP.</p>

<markup
lang="bash"
title="Check the metrics of your application:"
>curl -H "Accept: application/json" http://localhost:8080/metrics/application</markup>

<markup
lang="json"
title="Response body"
>{
  "db.counter.select-book" : 4
}</markup>

<p>The select-book statement was invoked four times.</p>

<p>This guide provided an introduction to the Helidon DB Client&#8217;s key features. If you want to learn more, see the
Helidon DB Client samples in <a target="_blank" href="https://medium.com/helidon/helidon-db-client-e12bbdc85b7" class="bare">https://medium.com/helidon/helidon-db-client-e12bbdc85b7</a>.</p>

</div>

</doc-view>
