<doc-view>

<h2 id="_contents">Contents</h2>
<div class="section">
<ul class="ulist">
<li>
<p><router-link to="#_overview" @click.native="this.scrollFix('#_overview')">Overview</router-link></p>

</li>
<li>
<p><router-link to="#maven-coordinates" @click.native="this.scrollFix('#maven-coordinates')">Maven Coordinates</router-link></p>

</li>
<li>
<p><router-link to="#_usage" @click.native="this.scrollFix('#_usage')">Usage</router-link></p>

</li>
<li>
<p><router-link to="#_configuration" @click.native="this.scrollFix('#_configuration')">Configuration</router-link></p>

</li>
<li>
<p><router-link to="#_examples" @click.native="this.scrollFix('#_examples')">Examples</router-link></p>

</li>
</ul>

</div>


<h2 id="_overview">Overview</h2>
<div class="section">
<p>The Helidon gRPC server provides a framework for creating <a target="_blank" href="http://grpc.io/">gRPC</a> applications.
While it allows you to deploy any standard gRPC service that
implements <code>io.grpc.BindableService</code> interface, including services generated
from the Protobuf IDL files (and even allows you to customize them to a certain
extent), using Helidon gRPC framework to implement your services has a number of
benefits:</p>

<ul class="ulist">
<li>
<p>It allows you to define both HTTP and gRPC services using a similar programming
model, simplifying the learning curve for developers.</p>

</li>
<li>
<p>It provides a number of helper methods that make service implementation
significantly simpler.</p>

</li>
<li>
<p>It allows you to configure some of the Helidon value-added features, such
as <router-link to="#_security" @click.native="this.scrollFix('#_security')">security</router-link> and <router-link to="#_service_metrics" @click.native="this.scrollFix('#_service_metrics')">metrics collection</router-link>
down to the method level.</p>

</li>
<li>
<p>It allows you to easily specify custom marshallers for requests and
responses if Protobuf does not satisfy your needs.</p>

</li>
<li>
<p>It provides built-in support for <router-link to="#_service_health_checks" @click.native="this.scrollFix('#_service_health_checks')">health checks</router-link>.</p>

</li>
</ul>

</div>


<h2 id="maven-coordinates">Maven Coordinates</h2>
<div class="section">
<p>To enable gRPC Server
add the following dependency to your project&#8217;s <code>pom.xml</code> (see
 <router-link to="/about/managing-dependencies">Managing Dependencies</router-link>).</p>

<markup
lang="xml"

>&lt;dependency&gt;
    &lt;groupId&gt;io.helidon.grpc&lt;/groupId&gt;
    &lt;artifactId&gt;helidon-grpc-server&lt;/artifactId&gt;
&lt;/dependency&gt;</markup>

<p id="security_maven_coordinartes">If <code>gRPC server security</code> is required as described in the <router-link to="#_security" @click.native="this.scrollFix('#_security')"></router-link> section, add the following dependency to your project’s pom.xml:</p>

<markup
lang="xml"

>&lt;dependency&gt;
    &lt;groupId&gt;io.helidon.security.integration&lt;/groupId&gt;
    &lt;artifactId&gt;helidon-security-integration-grpc&lt;/artifactId&gt;
&lt;/dependency&gt;</markup>

</div>


<h2 id="_usage">Usage</h2>
<div class="section">

<h3 id="_grpc_server_routing">gRPC Server Routing</h3>
<div class="section">
<p>Unlike the webserver, which allows you to route requests based on path expression
and the HTTP verb, the gRPC server always routes requests based on the service and
method name. This makes routing configuration somewhat simpler&#8201;&#8212;&#8201;all you need
to do is register your services:</p>

<markup
lang="java"

>private static GrpcRouting createRouting(Config config) {
    return GrpcRouting.builder()
            .register(new GreetService(config)) <span class="conum" data-value="1" />
            .register(new EchoService())        <span class="conum" data-value="2" />
            .register(new MathService())        <span class="conum" data-value="3" />
            .build();
}</markup>

<ul class="colist">
<li data-value="1">Register <code>GreetService</code> instance.</li>
<li data-value="2">Register <code>EchoService</code> instance.</li>
<li data-value="3">Register <code>MathService</code> instance.</li>
</ul>

<p>Both "standard" gRPC services that implement <code>io.grpc.BindableService</code> interface
(typically implemented by extending the generated server-side stub and overriding
its methods), and Helidon gRPC services that implement
<code>io.helidon.grpc.server.GrpcService</code> interface can be registered.
The difference is that Helidon gRPC services allow you to customize behavior
down to the method level, and provide a number of useful helper methods that
make service implementation easier, as we&#8217;ll see in a moment.</p>


<h4 id="_customizing_service_definitions">Customizing Service Definitions</h4>
<div class="section">
<p>When registering a service, regardless of its type, you can customize its
descriptor by providing a configuration consumer as a second argument to the
<code>register</code> method.</p>

<p>This is particularly useful when registering standard <code>BindableService</code>
instances, as it allows you to add certain Helidon-specific behaviors, such as
<router-link to="#_service_health_checks" @click.native="this.scrollFix('#_service_health_checks')">health checks</router-link> and <router-link to="#_service_metrics" @click.native="this.scrollFix('#_service_metrics')">metrics</router-link> to them:</p>

<markup
lang="java"

>private static GrpcRouting createRouting(Config config) {
    return GrpcRouting.builder()
            .register(new GreetService(config))
            .register(new EchoService(), service -&gt; {
                service.healthCheck(CustomHealthChecks::echoHealthCheck)  <span class="conum" data-value="1" />
                       .metered();                                        <span class="conum" data-value="2" />
            })
            .build();
}</markup>

<ul class="colist">
<li data-value="1">Add custom health check to the service.</li>
<li data-value="2">Specify that all the calls to service methods should be metered.</li>
</ul>

</div>


<h4 id="_specifying_global_interceptors">Specifying Global Interceptors</h4>
<div class="section">
<p><code>GrpcRouting</code> also allows you to specify <router-link to="#_interceptors" @click.native="this.scrollFix('#_interceptors')">custom interceptors</router-link> that will be applied to all registered services.</p>

<p>This is useful to configure features such as tracing, security and metrics collection,
and we provide built-in interceptors for those purposes that you can simply register
with the routing definition:</p>

<markup
lang="java"

>private static GrpcRouting createRouting(Config config) {
    return GrpcRouting.builder()
            .intercept(GrpcMetrics.timed())     <span class="conum" data-value="1" />
            .register(new GreetService(config))
            .register(new EchoService())
            .register(new MathService())
            .build();
}</markup>

<ul class="colist">
<li data-value="1">Register <code>GrpcMetrics</code> interceptor that will collect timers for all methods of all services (but can be overridden at the individual service or even method level).</li>
</ul>

</div>

</div>


<h3 id="_service_implementation">Service Implementation</h3>
<div class="section">
<p>At the very basic level, all you need to do in order to implement a Helidon
gRPC service is create a class that implements the <code>io.helidon.grpc.server.GrpcService</code>
interface and define one or more methods for the service:</p>

<markup
lang="java"

>class EchoService implements GrpcService {

    @Override
    public void update(ServiceDescriptor.Rules rules) {
        rules.marshallerSupplier(new JsonbMarshaller.Supplier()) <span class="conum" data-value="1" />
             .unary("Echo", this::echo); <span class="conum" data-value="2" />
    }

    /**
     * Echo the message back to the caller.
     *
     * @param request   the echo request containing the message to echo
     * @param observer  the response observer
     */
    public void echo(String request, StreamObserver&lt;String&gt; observer) {  <span class="conum" data-value="3" />
        complete(observer, request);  <span class="conum" data-value="4" />
    }
}</markup>

<ul class="colist">
<li data-value="1">Specify a custom marshaller to marshall requests and responses.</li>
<li data-value="2">Define unary method <code>Echo</code> and map it to the <code>this::echo</code> handler.</li>
<li data-value="3">Create a handler for the <code>Echo</code> method.</li>
<li data-value="4">Send the request string back to the client by completing response observer.</li>
</ul>

<div class="admonition note">
<p class="admonition-inline">The <code>complete</code> method shown in the example above is just one of many helper
methods available in the <code>GrpcService</code> class. See the full list
<a target="_blank" href="./apidocs/io.helidon.grpc.server/io/helidon/grpc/server/GrpcService.html">here</a>.</p>
</div>

<p>The example above implements a service with a single unary method which will be
exposed at the `EchoService/Echo' endpoint. The service explicitly defines
a marshaller for requests and responses, so this implies that you will have to
implement clients by hand and configure them to use the same marshaller as the
server. Obviously, one of the major selling points of gRPC is that it makes it easy to
generate clients for a number of languages (as long as you use Protobuf for marshalling),
so let&#8217;s see how we would implement Protobuf enabled Helidon gRPC service.</p>


<h4 id="_implementing_protobuf_services">Implementing Protobuf Services</h4>
<div class="section">
<p>In order to implement Protobuf-based service, you would follow the official
<a target="_blank" href="https://grpc.io/docs/quickstart/java.html">instructions</a> on the gRPC
web site, which boil down to the following:</p>


<h5 id="_define_the_service_idl">Define the Service IDL</h5>
<div class="section">
<p>For this example, we will re-implement the <code>EchoService</code> above as a Protobuf
service in <code>echo.proto</code> file.</p>

<markup
lang="proto"

>syntax = "proto3";
option java_package = "org.example.services.echo";

service EchoService {
  rpc Echo (EchoRequest) returns (EchoResponse) {}
}

message EchoRequest {
  string message = 1;
}

message EchoResponse {
  string message = 1;
}</markup>

<p>Based on this IDL, the gRPC compiler will generate message classes (<code>EchoRequest</code>
and <code>EchoResponse</code>), client stubs that can be used to make RPC calls to the server,
as well as the base class for the server-side service implementation.</p>

<p>We can ignore the last one, and implement the service using Helidon gRPC framework
instead.</p>

</div>


<h5 id="_implement_the_service">Implement the Service</h5>
<div class="section">
<p>The service implementation will be very similar to our original implementation:</p>

<markup
lang="java"

>class EchoService implements GrpcService {

    @Override
    public void update(ServiceDescriptor.Rules rules) {
        rules.proto(Echo.getDescriptor())  <span class="conum" data-value="1" />
             .unary("Echo", this::echo);   <span class="conum" data-value="2" />
    }

    /**
     * Echo the message back to the caller.
     *
     * @param request   the echo request containing the message to echo
     * @param observer  the response observer
     */
    public void echo(Echo.EchoRequest request, StreamObserver&lt;Echo.EchoResponse&gt; observer) {  <span class="conum" data-value="3" />
        String message = request.getMessage();  <span class="conum" data-value="4" />
        Echo.EchoResponse response = Echo.EchoResponse.newBuilder().setMessage(message).build();  <span class="conum" data-value="5" />
        complete(observer, response);  <span class="conum" data-value="6" />
    }
}</markup>

<ul class="colist">
<li data-value="1">Specify the proto descriptor in order to provide necessary type information and
enable Protobuf marshalling.</li>
<li data-value="2">Define unary method <code>Echo</code> and map it to the <code>this::echo</code> handler.</li>
<li data-value="3">Create a handler for the <code>Echo</code> method, using Protobuf message types for request and response.</li>
<li data-value="4">Extract message string from the request.</li>
<li data-value="5">Create the response containing extracted message.</li>
<li data-value="6">Send the response back to the client by completing response observer.</li>
</ul>

</div>

</div>

</div>


<h3 id="_interceptors">Interceptors</h3>
<div class="section">
<p>Helidon gRPC allows you to configure standard interceptors using <code>io.grpc.ServerInterceptor</code>.</p>

<p>For example, you could implement an interceptor that logs each RPC call:</p>

<markup
lang="java"

>class LoggingInterceptor implements ServerInterceptor {   <span class="conum" data-value="1" />

    private static final Logger LOG = Logger.getLogger(LoggingInterceptor.class.getName());

    @Override
    public &lt;ReqT, ResT&gt; ServerCall.Listener&lt;ReqT&gt; interceptCall(ServerCall&lt;ReqT, ResT&gt; call,
                                                                 Metadata metadata,
                                                                 ServerCallHandler&lt;ReqT, ResT&gt; handler) {

        LOG.info(() -&gt; "CALL: " + call.getMethodDescriptor());  <span class="conum" data-value="2" />
        return handler.startCall(call, metadata);               <span class="conum" data-value="3" />
    }
}</markup>

<ul class="colist">
<li data-value="1">Implement the interceptor class using <code>io.grpc.ServerInterceptor</code>.</li>
<li data-value="2">Implement the logging logic.</li>
<li data-value="3">The intercepted call is started.</li>
</ul>


<h4 id="_registering_interceptors">Registering Interceptors</h4>
<div class="section">
<p>You can register interceptors globally, in which case they will be applied to all
methods of all services, by simply adding them to the <code>GrpcRouting</code> instance:</p>

<markup
lang="java"

>private static GrpcRouting createRouting(Config config) {
    return GrpcRouting.builder()
            .intercept(new LoggingInterceptor())  <span class="conum" data-value="1" />
            .register(new GreetService(config))
            .register(new EchoService())
            .build();
}</markup>

<ul class="colist">
<li data-value="1">Adds <code>LoggingInterceptor</code> to all methods of <code>GreetService</code> and <code>EchoService</code>.</li>
</ul>

<p>You can also register an interceptor for a specific service, either by implementing
<code>GrpcService.update</code> method:</p>

<markup
lang="java"

>public class MyService implements GrpcService {

    @Override
    public void update(ServiceDescriptor.Rules rules) {
        rules.intercept(new LoggingInterceptor())   <span class="conum" data-value="1" />
                .unary("MyMethod", this::myMethod);
    }

    private &lt;ReqT, ResT&gt; void myMethod(ReqT request, StreamObserver&lt;ResT&gt; observer) {
        // do something
    }
}</markup>

<ul class="colist">
<li data-value="1">Adds <code>LoggingInterceptor</code> to all methods of <code>MyService</code>.</li>
</ul>

<p>Or by configuring <code>ServiceDescriptor</code> externally, when creating <code>GrpcRouting</code>, which
allows you to add interceptors to plain <code>io.grpc.BindableService</code> services as well:</p>

<markup
lang="java"

>private static GrpcRouting createRouting(Config config) {
    return GrpcRouting.builder()
            .register(new GreetService(config), cfg -&gt; cfg.intercept(new LoggingInterceptor()))  <span class="conum" data-value="1" />
            .register(new EchoService())
            .build();
}</markup>

<ul class="colist">
<li data-value="1">Adds <code>LoggingInterceptor</code> to all methods of <code>GreetService</code> only.</li>
</ul>

<p>Finally, you can also register an interceptor at the method level:</p>

<markup
lang="java"

>public class MyService implements GrpcService {

    @Override
    public void update(ServiceDescriptor.Rules rules) {
        rules.unary("MyMethod",
                     this::myMethod,
                     cfg -&gt; cfg.intercept(new LoggingInterceptor()));  <span class="conum" data-value="1" />
    }

    private &lt;ReqT, ResT&gt; void myMethod(ReqT request, StreamObserver&lt;ResT&gt; observer) {
        // do something
    }
}</markup>

<ul class="colist">
<li data-value="1">Adds <code>LoggingInterceptor</code> to <code>MyService::MyMethod</code> only.</li>
</ul>

</div>

</div>


<h3 id="_service_health_checks">Service Health Checks</h3>
<div class="section">
<p>Helidon gRPC services provide built-in support for Helidon Health Checks.</p>

<p>Unless a custom health check is implemented by the service developer, each service
deployed to the gRPC server will be provisioned with a default health check, which
always returns status of <code>UP</code>.</p>

<p>This allows all services, including the ones that don&#8217;t have a meaningful health check,
to show up in the health report (or to be queried for health) without service developer
having to do anything.</p>

<p>However, services that do need custom health checks can easily define one,
directly within <code>GrpcService</code> implementation:</p>

<markup
lang="java"

>public class MyService implements GrpcService {

    @Override
    public void update(ServiceDescriptor.Rules rules) {
        rules.unary("MyMethod", this::myMethod)
                .healthCheck(this::healthCheck);  <span class="conum" data-value="1" />
    }

    private HealthC
heckResponse healthCheck() {
        boolean fUp = isMyServiceUp();            <span class="conum" data-value="2" />
        return HealthCheckResponse
                .named(name())                    <span class="conum" data-value="3" />
                .state(fUp)                       <span class="conum" data-value="4" />
                .withData("ts", System.currentTimeMillis())  <span class="conum" data-value="5" />
                .build();
    }

    private &lt;ReqT, ResT&gt; void myMethod(ReqT request, StreamObserver&lt;ResT&gt; observer) {
        // do something
    }
}</markup>

<ul class="colist">
<li data-value="1">Configure a custom health check for the service.</li>
<li data-value="2">Determine the service status.</li>
<li data-value="3">Use service name as a health check name for consistency.</li>
<li data-value="4">Set the determined service status.</li>
<li data-value="5">Optionally provide additional metadata.</li>
</ul>

<p>You can also define custom health checks for an existing service, including plain
<code>io.grpc.BindableService</code> implementations, using a service configurer inside the
<code>GrpcRouting</code> definition:</p>

<markup
lang="java"

>private static GrpcRouting createRouting() {
    return GrpcRouting.builder()
            .register(new EchoService(), cfg -&gt; cfg.healthCheck(MyCustomHealthChecks::echoHealthCheck))  <span class="conum" data-value="1" />
            .build();
}</markup>

<ul class="colist">
<li data-value="1">Configure custom health check for an existing or legacy service.</li>
</ul>


<h4 id="_exposing_health_checks">Exposing Health Checks</h4>
<div class="section">
<p>All gRPC service health checks are managed by the Helidon gRPC server, and are
automatically exposed to the gRPC clients using a custom implementation of the
standard gRPC <code>HealthService</code> API.</p>

<p>However, they can also be exposed to REST clients via the standard Helidon/Microprofile
<code>/health</code> endpoint:</p>

<markup
lang="java"

>        GrpcServer grpcServer = GrpcServer.create(grpcServerConfig(), createRouting(config));  <span class="conum" data-value="1" />
        grpcServer.start();                                                                    <span class="conum" data-value="2" />

        HealthSupport health = HealthSupport.builder()
                .add(grpcServer.healthChecks())     <span class="conum" data-value="3" />
                .build();

        Routing routing = Routing.builder()
                .register(health)                   <span class="conum" data-value="4" />
                .build();

        WebServer.create(webServerConfig(), routing).start();   <span class="conum" data-value="5" /></markup>

<ul class="colist">
<li data-value="1">Create the <code>GrpcServer</code> instance.</li>
<li data-value="2">Start the gRPC server which will deploy all the services and register default and custom health checks.</li>
<li data-value="3">Add gRPC server managed health checks to <code>HealthSupport</code> instance.</li>
<li data-value="4">Add <code>HealthSupport</code> to the web server routing definition.</li>
<li data-value="5">Create and start the web server.</li>
</ul>

<p>All gRPC health checks will now be available via the <code>/health</code> REST endpoint, in
addition to the standard gRPC <code>HealthService</code></p>

</div>

</div>


<h3 id="_service_metrics">Service Metrics</h3>
<div class="section">
<p>The Helidon gRPC server has built-in support for metrics capture, which allows
service developers to easily enable application-level metrics for their services.</p>


<h4 id="_enabling_metrics_capture">Enabling Metrics Capture</h4>
<div class="section">
<p>By default, the gRPC server only captures two vendor-level metrics: <code>grpc.request.count</code>
and <code>grpc.request.meter</code>. These metrics provide an aggregate view of requests across
all services, and serve as an indication of the overall server load.</p>

<p>However, users can enable more fine-grained metrics by simply configuring a built-in
<code>GrpcMetrics</code> interceptor within the routing:</p>

<markup
lang="java"

>private static GrpcRouting createRouting(Config config) {
    return GrpcRouting.builder()
            .intercept(GrpcMetrics.timed())       <span class="conum" data-value="1" />
            .register(new GreetService(config))
            .register(new EchoService())
            .build();
}</markup>

<ul class="colist">
<li data-value="1">Capture the metrics for all methods of all services as a <code>timer</code>.</li>
</ul>

<p>In the example above we have chosen to create and keep a <code>timer</code> metric type for
each method of each service. Alternatively, we could&#8217;ve chosen to use a
<code>counter</code>, <code>meter</code> or a <code>histogram</code> instead.</p>

</div>


<h4 id="_overriding_metrics_capture">Overriding Metrics Capture</h4>
<div class="section">
<p>While global metrics capture is certainly useful, it is not always sufficient.
Keeping a separate <code>timer</code> for each gRPC method may be excessive, so the user
could decide to use a lighter-weight metric type, such as a <code>counter</code> or a <code>meter</code>.</p>

<p>However, the user may still want to enable a <code>histogram</code> or a <code>timer</code> for some services,
or even only some methods of some services.</p>

<p>This can be easily accomplished by overriding the type of the captured metric at
either the service or the method level:</p>

<markup
lang="java"

>private static GrpcRouting createRouting(Config config) {
    return GrpcRouting.builder()
            .intercept(GrpcMetrics.counted())  <span class="conum" data-value="1" />
            .register(new MyService())
            .build();
}

public static class MyService implements GrpcService {

    @Override
    public void update(ServiceDescriptor.Rules rules) {
        rules
            .intercept(GrpcMetrics.metered())  <span class="conum" data-value="2" />
            .unary("MyMethod", this::myMethod,
                       cfg -&gt; cfg.intercept(GrpcMetrics.timer())); <span class="conum" data-value="3" />
    }

    private &lt;ReqT, ResT&gt; void myMethod(ReqT request, StreamObserver&lt;ResT&gt; observer) {
        // do something
    }
}</markup>

<ul class="colist">
<li data-value="1">Use <code>counter</code> for all methods of all services, unless overridden.</li>
<li data-value="2">Use <code>meter</code> for all methods of <code>MyService</code>.</li>
<li data-value="3">Use <code>timer</code> for <code>MyService::MyMethod</code>.</li>
</ul>

</div>


<h4 id="_exposing_metrics_externally">Exposing Metrics Externally</h4>
<div class="section">
<p>Collected metrics are stored in the standard Helidon metric registries, such as the vendor and
application registries, and can be exposed via the standard <code>/metrics</code> REST API.</p>

<markup
lang="java"

>Routing routing = Routing.builder()
        .register(MetricsSupport.create())    <span class="conum" data-value="1" />
        .build();

WebServer.create(webServerConfig(), routing)  <span class="conum" data-value="2" />
         .start()</markup>

<ul class="colist">
<li data-value="1">Add the <code>MetricsSupport</code> instance to web server routing.</li>
<li data-value="2">Create and start the Helidon web server.</li>
</ul>

<p>See <router-link to="/se/metrics/metrics">Helidon Metrics</router-link> documentation for more details.</p>

</div>


<h4 id="_specifying_metric_metadata">Specifying Metric Metadata</h4>
<div class="section">
<p>Helidon metrics contain metadata such as tags, a description, units etc. It is possible to
add this additional metadata when specifying the metrics.</p>


<h5 id="_adding_tags">Adding Tags</h5>
<div class="section">
<p>To add tags to a metric, a <code>Map</code> of key/value tags can be supplied.</p>

<markup
lang="java"

>Map&lt;String, String&gt; tagMap = new HashMap&lt;&gt;();
tagMap.put("keyOne", "valueOne");
tagMap.put("keyTwo", "valueTwo");

GrpcRouting routing = GrpcRouting.builder()
        .intercept(GrpcMetrics.counted().tags(tagMap))   <span class="conum" data-value="1" />
        .register(new MyService())
        .build();</markup>

<ul class="colist">
<li data-value="1">The <code>tags()</code> method is used to add the <code>Map</code> of tags to the metric.</li>
</ul>

</div>


<h5 id="_adding_a_description">Adding a Description</h5>
<div class="section">
<p>A meaningful description can be added to a metric.</p>

<markup
lang="java"

>GrpcRouting routing = GrpcRouting.builder()
        .intercept(GrpcMetrics.counted().description("Something useful")) <span class="conum" data-value="1" />
        .register(new MyService())
        .build();</markup>

<ul class="colist">
<li data-value="1">The <code>description()</code> method is used to add the description to the metric.</li>
</ul>

</div>


<h5 id="_adding_metric_units">Adding Metric Units</h5>
<div class="section">
<p>A <code>units</code> value can be added to a metric.</p>

<markup
lang="java"

>GrpcRouting routing = GrpcRouting.builder()
        .intercept(GrpcMetrics.timed().units(MetricUnits.SECONDS)) <span class="conum" data-value="1" />
        .register(new MyService())
        .build();</markup>

<ul class="colist">
<li data-value="1">The <code>units()</code> method is used to specify the metric units,
the value of which is one of the constants from the <code>org.eclipse.microprofile.metrics.MetricUnits</code> class.</li>
</ul>

</div>

</div>


<h4 id="_overriding_the_metric_name">Overriding the Metric Name</h4>
<div class="section">
<p>By default, the metric name is the gRPC service name followed by a dot ('.') followed by the method name.
It is possible to supply a function that can be used to override the default behaviour.</p>

<p>The function should implement the <code>io.helidon.grpc.metrics.GrpcMetrics.NamingFunction</code> interface.</p>

<markup
lang="java"

>@FunctionalInterface
public interface NamingFunction {
    /**
     * Create a metric name.
     *
     * @param service    the service descriptor
     * @param methodName the method name
     * @param metricType the metric type
     * @return the metric name
     */
    String createName(ServiceDescriptor service, String methodName, MetricType metricType);
}</markup>

<p>This is a functional interface so a lambda expression can be used too.</p>

<markup
lang="java"

>GrpcRouting routing = GrpcRouting.builder()
        .intercept(GrpcMetrics.counted()
                .nameFunction((svc, method, metric) -&gt; "grpc." + service.name() + '.' + method) <span class="conum" data-value="1" /></markup>

<ul class="colist">
<li data-value="1">The <code>NamingFunction</code> is just a lambda that returns the concatenated service name and method name
with the prefix <code>grpc.</code>. So for a service "Foo" and method "bar", the above example would produce a name "grpc.Foo.bar".</li>
</ul>

</div>

</div>


<h3 id="_security">Security</h3>
<div class="section">
<p>To enable server security, refer to the earlier section about <router-link to="#security_maven_coordinartes" @click.native="this.scrollFix('#security_maven_coordinartes')">Security maven coordinates</router-link> for guidance on what dependency to add in the project&#8217;s pom.xml.</p>


<h4 id="_bootstrapping">Bootstrapping</h4>
<div class="section">
<p>There are two steps to configure security with the gRPC server:</p>

<ol style="margin-left: 15px;">
<li>
Create the security instance and register it the with server.

</li>
<li>
Protect the gRPC services of the server with various security features.

</li>
</ol>

<markup
lang="java"
title="Example using builders"
>// gRPC server's routing
GrpcRouting.builder()
    // This is step 1 - register security instance with gRPC server processing
    // security - instance of security either from config or from a builder
    // securityDefaults - default enforcement for each service that has a security definition
    .intercept(GrpcSecurity.create(security).securityDefaults(GrpcSecurity.authenticate()))
    // this is step 2 - protect a service
    // register and protect this service with authentication (from defaults) and role "user"
    .register(greetService, GrpcSecurity.rolesAllowed("user"))
    .build();</markup>

<markup
lang="java"
title="Example using builders for more fine grained method level security"
>// create the service descriptor
ServiceDescriptor greetService = ServiceDescriptor.builder(new GreetService())
        // Add an instance of gRPC security that will apply to all methods of
        // the service - in this case require the "user" role
        .intercept(GrpcSecurity.rolesAllowed("user"))
        // Add an instance of gRPC security that will apply to the "SetGreeting"
        // method of the service - in this case require the "admin" role
        .intercept("SetGreeting", GrpcSecurity.rolesAllowed("admin"))
        .build();

// Create the gRPC server's routing
GrpcRouting.builder()
    // This is step 1 - register security instance with gRPC server processing
    // security - instance of security either from config or from a builder
    // securityDefaults - default enforcement for each service that has a security definition
    .intercept(GrpcSecurity.create(security).securityDefaults(GrpcSecurity.authenticate()))
    // this is step 2 - add the service descriptor
    .register(greetService)
    .build();</markup>

<markup
lang="java"
title="Example using configuration"
>GrpcRouting.builder()
    // helper method to load both security and gRPC server security from configuration
    .intercept(GrpcSecurity.create(config))
    // continue with gRPC server route configuration...
    .register(new GreetService())
    .build();</markup>

<markup
lang="conf"
title="Example using configuration - configuration (HOCON)"
># This may change in the future - to align with gRPC server configuration,
# once it is supported
security
  grpc-server:
    # Configuration of integration with gRPC server
    defaults:
        authenticate: true
    # Configuration security for individual services
    services:
    - name: "GreetService"
      defaults:
      roles-allowed: ["user"]
      # Configuration security for individual methods of the service
      methods:
      - name: "SetGreeting"
        roles-allowed: ["admin"]</markup>


<h5 id="_client_security">Client security</h5>
<div class="section">
<p>When using the Helidon SE gRPC client, API security can be configured for a gRPC service
or at the individual method level. The client API has a custom <code>CallCredentials</code> implementation that
integrates with the Helidon security APIs.</p>

<markup
lang="java"
title="Example configuring client security for a service"
>Security security = Security.builder()  <span class="conum" data-value="1" />
        .addProvider(HttpBasicAuthProvider.create(config.get("http-basic-auth")))
        .build();

GrpcClientSecurity clientSecurity = GrpcClientSecurity.builder(security.createContext("test.client")) <span class="conum" data-value="2" />
        .property(HttpBasicAuthProvider.EP_PROPERTY_OUTBOUND_USER, user)
        .property(HttpBasicAuthProvider.EP_PROPERTY_OUTBOUND_PASSWORD, password)
        .build();

ClientServiceDescriptor descriptor = ClientServiceDescriptor <span class="conum" data-value="3" />
        .builder(StringService.class)
        .unary("Lower")
        .callCredentials(clientSecurity)                     <span class="conum" data-value="4" />
        .build();

GrpcServiceClient client = GrpcServiceClient.create(channel, descriptor); <span class="conum" data-value="5" />

String response = client.blockingUnary("Lower", "ABCD"); <span class="conum" data-value="6" /></markup>

<ul class="colist">
<li data-value="1">Create the Helidon <code>Security</code> instance which, in this case, will use the basic auth provider.</li>
<li data-value="2">Create the <code>GrpcClientSecurity</code> gRPC <code>CallCredentials</code> adding the user and password
property expected by the basic auth provider.</li>
<li data-value="3">Create the gRPC <code>ClientServiceDescriptor</code> for the <code>StringService</code> gRPC service.</li>
<li data-value="4">Set the <code>GrpcClientSecurity</code> instance as the call credentials for all methods of the service.</li>
<li data-value="5">Create a <code>GrpcServiceClient</code> that will allow methods to be called on the service.</li>
<li data-value="6">Call the "Lower" method which will use the configured basic auth credentials.</li>
</ul>

<markup
lang="java"
title="Example configuring client security for a specific method"
>GrpcClientSecurity clientSecurity = GrpcClientSecurity.builder(security.createContext("test.client")) <span class="conum" data-value="1" />
        .property(HttpBasicAuthProvider.EP_PROPERTY_OUTBOUND_USER, user)
        .property(HttpBasicAuthProvider.EP_PROPERTY_OUTBOUND_PASSWORD, password)
        .build();

ClientServiceDescriptor descriptor = ClientServiceDescriptor <span class="conum" data-value="2" />
        .builder(StringService.class)
        .unary("Lower")
        .unary("Upper", rules -&gt; rules.callCredentials(clientSecurity)) <span class="conum" data-value="3" />
        .build();</markup>

<ul class="colist">
<li data-value="1">Create the <code>GrpcClientSecurity</code> call credentials in the same way as above.</li>
<li data-value="2">Create the <code>ClientServiceDescriptor</code>, this time with two unary methods, "Lower" and "Upper".</li>
<li data-value="3">The "Upper" method is configured to use the <code>GrpcClientSecurity</code> call credentials, the "Lower" method
will be called without any credentials.</li>
</ul>

</div>


<h5 id="_outbound_security">Outbound security</h5>
<div class="section">
<p>Outbound security covers three scenarios:</p>

<ul class="ulist">
<li>
<p>Calling a secure gRPC service from inside a gRPC service method handler.</p>

</li>
<li>
<p>Calling a secure gRPC service from inside a web server method handler.</p>

</li>
<li>
<p>Calling a secure web endpoint from inside a gRPC service method handler.</p>

</li>
</ul>

<p>Within each scenario, credentials can be propagated if the gRPC/http method
handler is executing within a security context or credentials can be overridden
to provide a different set of credentials to use for calling the outbound endpoint.</p>

<markup
lang="java"
title="Example calling a secure gRPC service from inside a gRPC service method handler"
>// Obtain the SecurityContext from the current gRPC call Context
SecurityContext securityContext = GrpcSecurity.SECURITY_CONTEXT.get();

// Create a gRPC CallCredentials that will use the current request's
// security context to configure outbound credentials
GrpcClientSecurity clientSecurity = GrpcClientSecurity.create(securityContext);

// Create the gRPC stub using the CallCredentials
EchoServiceGrpc.EchoServiceBlockingStub stub = noCredsEchoStub.withCallCredentials(clientSecurity);</markup>

<markup
lang="java"
title="Example calling a secure gRPC service from inside a web server method handler"
>private static void propagateCredentialsWebRequest(ServerRequest req, ServerResponse res) {
    try {
        // Create a gRPC CallCredentials that will use the current request's
        // security context to configure outbound credentials
        GrpcClientSecurity clientSecurity = GrpcClientSecurity.create(req);

        // Create the gRPC stub using the CallCredentials
        EchoServiceGrpc.EchoServiceBlockingStub stub = noCredsEchoStub.withCallCredentials(clientSecurity);

        String message = req.queryParams().first("message").orElse(null);
        Echo.EchoResponse echoResponse = stub.echo(Echo.EchoRequest.newBuilder().setMessage(message).build());
        res.send(echoResponse.getMessage());
    } catch (StatusRuntimeException e) {
        res.status(GrpcHelper.toHttpResponseStatus(e)).send();
    } catch (Throwable thrown) {
        res.status(Http.ResponseStatus.create(500, thrown.getMessage())).send();
    }
}</markup>

<markup
lang="java"
title="Example calling a secure web endpoint from inside a gRPC service method handler"
>// Obtain the SecurityContext from the gRPC call Context
SecurityContext securityContext = GrpcSecurity.SECURITY_CONTEXT.get();

// Use the SecurityContext as normal to make a http request
Response webResponse = client.target(url)
        .path("/test")
        .request()
        .property(ClientSecurity.PROPERTY_CONTEXT, securityContext)
        .get();</markup>

</div>

</div>

</div>


<h3 id="_marshalling">Marshalling</h3>
<div class="section">

<h4 id="_default_marshalling_support">Default Marshalling Support</h4>
<div class="section">
<p>Helidon gRPC supports Protobuf out of the box. The Protobuf marshaller will be used by default for any request and response classes that extend <code>com.google.protobuf.MessageLite</code>, which is the case for all classes generated from a <code>proto</code> file using <code>protoc</code> compiler.</p>

<p>That means that you don&#8217;t need any special handling or configuration in order to support Protobuf serialization of requests and responses.</p>

</div>


<h4 id="_custom_marshalling">Custom Marshalling</h4>
<div class="section">
<p>Helidon makes the use of custom marshallers trivial and provides one custom implementation, <a target="_blank" href="https://github.com/oracle/helidon/tree/master/grpc/core/src/main/java/io/helidon/grpc/core/JsonbMarshaller.java">JsonbMarshaller</a>, out of the box.</p>

<p>You can also easily implement your own marshaller to support serialization formats that are not supported natively
by Helidon, by implementing <code>Marshaller</code> and <code>MarshallerSupplier</code> interfaces. As an example, check out
the source code of the built-in marshaller:
<a target="_blank" href="https://github.com/oracle/helidon/tree/master/grpc/core/src/main/java/io/helidon/grpc/core/JsonbMarshaller.java">JsonbMarshaller.java</a>.</p>

<p>Furthermore, <a target="_blank" href="https://coherence.community/">Oracle Coherence CE</a> provides a marshaller for a highly optimized, binary, platform independent Portable Object Format (POF). You can find more information about POF in <a target="_blank" href="https://coherence.community/20.12/docs/#/docs/core/04_portable_types">Coherence documentation</a></p>


<h5 id="_setting_the_custom_marshaller">Setting the custom marshaller</h5>
<div class="section">
<p>You can implement the <code>update</code> method on your service&#8217;s class and set the custom marshaller supplier via the
<code>ServiceDescriptor.Rules.marshallerSupplier()</code> method:</p>

<markup
lang="java"
title="Sample code for setting the marshaller on the gRPC service"
>public class GreetServiceJava
        implements GrpcService {
    private String greeting;


    public GreetServiceJava(Config config) {
        this.greeting = config.get("app.greeting").asString().orElse("Ciao");
    }

    @Override
    public void update(ServiceDescriptor.Rules rules) {
        rules.marshallerSupplier(new JsonbMarshaller.Supplier())  <span class="conum" data-value="1" />
                .unary("Greet", this::greet)
                .unary("SetGreeting", this::setGreeting);
    }

    // Implement Service methods
}</markup>

<ul class="colist">
<li data-value="1">Specify the custom marshaller to use.</li>
</ul>

</div>

</div>

</div>

</div>


<h2 id="_configuration">Configuration</h2>
<div class="section">
<p>Configure the gRPC server using the Helidon configuration framework, either programmatically
or via a configuration file.</p>


<h3 id="_configuring_the_grpc_server_in_your_code">Configuring the gRPC server in your code</h3>
<div class="section">
<p>The easiest way to configure the gRPC server is in your application code.</p>

<markup
lang="java"

>GrpcServerConfiguration configuration = GrpcServerConfiguration.builder()
                                                       .port(8080)
                                                       .build();
GrpcServer grpcServer = GrpcServer.create(configuration, routing);</markup>

<p>See all configuration options
<a target="_blank" href="./apidocs/io.helidon.grpc/GrpcServerConfiguration.html">here</a>.</p>

</div>


<h3 id="_configuring_the_grpc_server_in_a_configuration_file">Configuring the gRPC server in a configuration file</h3>
<div class="section">
<p>You can also define the gRPC server configuration in a file.</p>

<p>Type: <a target="_blank" href="./apidocs/io.helidon.grpc.server/io/helidon/grpc/server/GrpcServerConfiguration.html">io.helidon.grpc.server.GrpcServerConfiguration</a></p>

</div>


<h3 id="_configuration_options">Configuration options</h3>
<div class="section">
<div class="block-title"><span>Optional configuration options</span></div>
<div class="table__overflow elevation-1  ">
<table class="datatable table">
<colgroup>
<col style="width: 23.077%;">
<col style="width: 23.077%;">
<col style="width: 15.385%;">
<col style="width: 38.462%;">
</colgroup>
<thead>
<tr>
<th>key</th>
<th>type</th>
<th>default value</th>
<th>description</th>
</tr>
</thead>
<tbody>
<tr>
<td class=""><code>name</code></td>
<td class=""><doc-view>
<p>string</p>

</doc-view>
</td>
<td class=""><code>grpc.server</code></td>
<td class=""><doc-view>
<p>Set the name of the gRPC server.</p>

<pre>Configuration key: `name`</pre>
</doc-view>
</td>
</tr>
<tr>
<td class=""><code>native</code></td>
<td class=""><doc-view>
<p>boolean</p>

</doc-view>
</td>
<td class=""><code>false</code></td>
<td class=""><doc-view>
<p>Specify if native transport should be used.</p>

</doc-view>
</td>
</tr>
<tr>
<td class=""><code>port</code></td>
<td class=""><doc-view>
<p>int</p>

</doc-view>
</td>
<td class=""><code>1408</code></td>
<td class=""><doc-view>
<p>Sets server port. If port is <code>0</code> or less then any available ephemeral port will be used.</p>

<pre>Configuration key: `port`</pre>
</doc-view>
</td>
</tr>
<tr>
<td class=""><code>workers</code></td>
<td class=""><doc-view>
<p>int</p>

</doc-view>
</td>
<td class=""><code>Number of processors available to the JVM</code></td>
<td class=""><doc-view>
<p>Sets a count of threads in pool used to process HTTP requests.
 Default value is <code>CPU_COUNT * 2</code>.</p>

<pre>Configuration key: `workers`</pre>
</doc-view>
</td>
</tr>
</tbody>
</table>
</div>

<markup
lang="yaml"
title="GrpcServer configuration file example using  <code>application.yaml</code>"
>grpc:
  port: 3333</markup>

<p>Then, in your application code, load the configuration from that file.</p>

<markup
lang="java"
title="GrpcServer initialization using the <code>application.conf</code> file located on the classpath"
>GrpcServerConfiguration configuration = GrpcServerConfiguration.create(
        Config.builder()
              .sources(classpath("application.conf"))
              .build());

GrpcServer grpcServer = GrpcServer.create(configuration, routing);</markup>

</div>

</div>


<h2 id="_examples">Examples</h2>
<div class="section">

<h3 id="_quick_start">Quick Start</h3>
<div class="section">
<p>Here is the code for a minimalist gRPC application that runs on a default port (1408):</p>

<markup
lang="java"

>public static void main(String[] args) throws Exception {
    GrpcServer grpcServer = GrpcServer
            .create(GrpcRouting.builder()
                            .register(new HelloService()) <span class="conum" data-value="1" />
                            .build())
            .start() <span class="conum" data-value="2" />
            .toCompletableFuture()
            .get(10, TimeUnit.SECONDS); // Implement the simplest possible gRPC service. <span class="conum" data-value="3" />

    System.out.println("gRPC server started at: http://localhost:" + grpcServer.port()); <span class="conum" data-value="4" />
}

static class HelloService implements GrpcService { <span class="conum" data-value="5" />
    @Override
    public void update(ServiceDescriptor.Rules rules) {
        rules.marshallerSupplier(new JsonbMarshaller.Supplier()) <span class="conum" data-value="6" />
             .unary("SayHello", ((request, responseObserver) -&gt; complete(responseObserver, "Hello " + request))); <span class="conum" data-value="7" />
    }
}</markup>

<ul class="colist">
<li data-value="1">Register the gRPC service.</li>
<li data-value="2">Start the server.</li>
<li data-value="3">Wait for the server to start while throwing possible errors as exceptions.</li>
<li data-value="4">The server is bound to a default port (1408).</li>
<li data-value="5">Implement the simplest possible gRPC service.</li>
<li data-value="6">Specify a custom marshaller using the built-in JsonB marshaller to marshall requests and responses.</li>
<li data-value="7">Add unary method <code>HelloService/SayHello</code> to the service definition.</li>
</ul>

</div>


<h3 id="_additional_grpc_server_examples">Additional gRPC server examples</h3>
<div class="section">
<p>A set of gRPC server examples for Helidon SE can be found in the following links:</p>

<ul class="ulist">
<li>
<p><a target="_blank" href="https://github.com/oracle/helidon/tree/master/examples/grpc/basics">Basic gRPC Server</a></p>

</li>
<li>
<p><a target="_blank" href="https://github.com/oracle/helidon/tree/master/examples/grpc/metrics">gRPC Server Metrics</a></p>

</li>
<li>
<p><a target="_blank" href="https://github.com/oracle/helidon/tree/master/examples/grpc/opentracing">OpenTracing on a gRPC Server</a></p>

</li>
<li>
<p><a target="_blank" href="https://github.com/oracle/helidon/tree/master/examples/grpc/security">Basic Auth Security on a gRPC Server</a></p>

</li>
<li>
<p><a target="_blank" href="https://github.com/oracle/helidon/tree/master/examples/grpc/security-abac">Attribute-Based Access Control (ABAC) security on a gRPC Server</a></p>

</li>
<li>
<p><a target="_blank" href="https://github.com/oracle/helidon/tree/master/examples/grpc/security-outbound">Outbound Security on a gRPC Server</a></p>

</li>
</ul>

</div>

</div>

</doc-view>
