<doc-view>

<h2 id="_contents">Contents</h2>
<div class="section">
<ul class="ulist">
<li>
<p><router-link to="#_overview" @click.native="this.scrollFix('#_overview')">Overview</router-link></p>

</li>
<li>
<p><router-link to="#maven-coordinates" @click.native="this.scrollFix('#maven-coordinates')">Maven Coordinates</router-link></p>

</li>
<li>
<p><router-link to="#_usage" @click.native="this.scrollFix('#_usage')">Usage</router-link></p>

</li>
<li>
<p><router-link to="#_configuration" @click.native="this.scrollFix('#_configuration')">Configuration</router-link></p>

</li>
<li>
<p><router-link to="#_examples" @click.native="this.scrollFix('#_examples')">Examples</router-link></p>

</li>
</ul>

</div>


<h2 id="_overview">Overview</h2>
<div class="section">
<p>Helidon gRPC client provides a framework for creating <a target="_blank" href="http://grpc.io/">gRPC</a> client applications. The client framework
allows a uniform way to access gRPC services that use either Protobuf or some custom serialization format. The benefits of using Helidon gRPC client Framework include:</p>

<ul class="ulist">
<li>
<p>It provides a number of helper methods that make client implementation
significantly simpler.</p>

</li>
<li>
<p>It allows you to configure some of the Helidon value-added features, such
as <router-link :to="{path: '/se/grpc/server', hash: '#_security'}">security</router-link>,  <router-link :to="{path: '/se/grpc/server', hash: '#_service_metrics'}">metrics collection</router-link> and <router-link :to="{path: '/se/grpc/server', hash: '#_interceptors'}">interceptors</router-link> down to the method level.</p>

</li>
<li>
<p>It allows you to easily specify custom marshallers for requests and
responses if <code>protobuf</code> does not satisfy your needs.</p>

</li>
</ul>

<p>The class <code>GrpcServiceClient</code> acts as the client object for accessing a gRPC service. Creating a <code>GrpcServiceClient</code> involves:</p>

<ol style="margin-left: 15px;">
<li>
Creating a <code>ClientServiceDescriptor</code> which describes the methods in the service that this client can invoke.

</li>
<li>
Creating a gRPC <code>Channel</code> through which the client communicates with the server.

</li>
</ol>

<p>In later sections in this document, you will see how to customize both <code>ClientServiceDescriptor</code> and the <code>Channel</code>.</p>

</div>


<h2 id="maven-coordinates">Maven Coordinates</h2>
<div class="section">
<p>To enable gRPC Client
add the following dependency to your project&#8217;s <code>pom.xml</code> (see
 <router-link to="/about/managing-dependencies">Managing Dependencies</router-link>).</p>

<markup
lang="xml"

>&lt;dependency&gt;
    &lt;groupId&gt;io.helidon.grpc&lt;/groupId&gt;
    &lt;artifactId&gt;helidon-grpc-client&lt;/artifactId&gt;
&lt;/dependency&gt;</markup>

</div>


<h2 id="_usage">Usage</h2>
<div class="section">

<h3 id="_client_implementation_basics">Client Implementation Basics</h3>
<div class="section">
<ol style="margin-left: 15px;">
<li>
The first step to create a Helidon gRPC client application is to describe the set of methods in the gRPC service. Helidon
gRPC client Framework (simply called the "Client framework" in the remainder of the document) provides a class called
<code>ClientServiceDescriptor</code> to describe the set of methods of a service that the client may invoke. There are several ways to build and initialize a <code>ClientServiceDescriptor</code>.
<ul class="ulist">
<li>
<p>The first option is to initialize <code>ClientServiceDescriptor</code> using <code>protoc</code> generated artifacts like
<code>BindableService</code> or <code>io.grpc.ServiceDescriptor</code>. This option is possible if the gRPC service
was built using <code>.proto</code> file. In this case, the set of gRPC methods, their types and
the appropriate marshallers are detected automatically. This is certainly the easiest way to initialize
a <code>ClientServiceDescriptor</code>.</p>

</li>
<li>
<p>The other option is to programmatically build the <code>ClientServiceDescriptor</code>. This option should be
taken if the service was <strong>not</strong> built from protobuf files or if the <code>protoc</code> generated artifacts are not
available to the client.</p>

</li>
</ul>

</li>
<li>
The next step is to create a gRPC <code>Channel</code> to use to communicate with the server.

</li>
<li>
Finally, we create an instance of <code>GrpcServiceClient</code> passing the <code>ClientMethodDescriptor</code> and the <code>Channel</code> instances.

</li>
</ol>

</div>


<h3 id="_creating_grpc_clients_from_protoc_generated_artifacts">Creating gRPC clients from <code>protoc</code> generated artifacts</h3>
<div class="section">
<p>As mentioned above, the easiest way to create a <code>ClientServiceDescriptor</code> is to create it from an <code>io.grpc.ServiceDescriptor</code> or
from a <code>io.grpc.BindableService</code>. It is fairly trivial to obtain these from a service generated from artifacts generated
from protobuf IDL file.</p>

<p>For this section we will assume the following proto file:</p>

<markup
lang="proto"

>syntax = "proto3";
option java_package = "io.helidon.grpc.client.test";

service StringService {
  rpc Upper (StringMessage) returns (StringMessage) {}                  // (Unary)
  rpc Lower (StringMessage) returns (StringMessage) {}                  // (Unary)
  rpc Split (StringMessage) returns (stream StringMessage) {}           // (Server Streaming)
  rpc Join (stream StringMessage) returns (StringMessage) {}            // (Client Streaming)
  rpc Echo (stream StringMessage) returns (stream StringMessage) {}     // (Bidirectional Streaming)
}

message StringMessage {
  string text = 1;
}</markup>

<p>If you run it through <code>protoc</code>, it will generate a class (among other things) called <code>StringService</code>.
Assuming that the <code>StringService</code> server is running on port 1408, here is how you can create a Helidon gRPC
Client that uses the Client Framework to invoke various types of gRPC methods.</p>


<h4 id="_creating_and_initializing_a_clientservicedescriptor_for_stringservice_generated_from_protoc">Creating and initializing a ClientServiceDescriptor for StringService (generated from <code>protoc</code>)</h4>
<div class="section">
<p>Let&#8217;s build a class called <code>ProtoBasedStringServiceClient</code> that invokes the various types of
gRPC methods that our <code>StringService</code> offers.</p>

<markup
lang="java"

>public class ProtoBasedStringServiceClient {

    private GrpcServiceClient client;

    public ProtoBasedStringServiceClient() {
        ClientServiceDescriptor desc = ClientServiceDescriptor
                .builder(StringService.getServiceDescriptor())  <span class="conum" data-value="1" />
                .build();

        Channel channel = ManagedChannelBuilder.forAddress("localhost", 1408)  <span class="conum" data-value="2" />
                .usePlaintext().build();

        this.client = GrpcServiceClient.create(channel, desc);  <span class="conum" data-value="3" />
    }

    /**
     * Many gRPC methods take a {@link io.grpc.StreamObserver} as an argument. Lets
     * build a helper class that can be used in our example.
     */
    public static class StringMessageStream&lt;T&gt; implements StreamObserver&lt;T&gt; {  <span class="conum" data-value="4" />
        @Override
        public void onNext(T value) {
            System.out.println("Received : " + value);
        }

        @Override
        public void onError(Throwable t) {
          t.printStracktrace();
        }

        @Override
        public void onCompleted() {
          System.out.println("DONE");
        }
    }
}</markup>

<ul class="colist">
<li data-value="1">Initialize the builder by specifying the <code>StringService&#8217;s</code> proto <code>ServiceDescriptor</code>. From
the <code>ServiceDescriptor</code>, the builder detects the service name, the set of method names, the type of
each method (like Unary, ServerStreaming, etc.), the request and response types (and
hence their corresponding Marshallers), etc.</li>
<li data-value="2">We create a <code>Channel</code> to the service that is running on <code>localhost:1408</code>.</li>
<li data-value="3">Finally, we create our <code>GrpcServiceClient</code> by using the above mentioned <code>ClientServiceDescriptor</code>.
and <code>Channel</code>. This <code>client</code> reference will be used to invoke various gRPC methods in our
<code>StringService</code>.</li>
<li data-value="4">We define a static inner class that implements the <code>io.grpc.StreamObserver</code> interface. An instance
of this class can be used wherever a <code>io.grpc.StreamObserver</code> is required (like server streaming,
bi-directional streaming methods).</li>
</ul>

</div>


<h4 id="_invoking_a_unary_method_on_the_stringservice">Invoking a unary method on the StringService</h4>
<div class="section">
<p>The Client Framework provides many helper methods to invoke gRPC unary methods.</p>

<markup
lang="java"

>public class ProtoBasedStringServiceClient {

    private GrpcServiceClient client;

    public ProtoBasedStringServiceClient() { /* code omitted */ }

    public void invokeUnaryMethod() throws Exception {
        StringMessage input = StringMessage.newBuilder().setText("ABC").build();

        CompletableFuture&lt;String&gt; result = client.unary("Lower", input);  <span class="conum" data-value="1" />

        String lcase = client.blockingUnary("Lower", input);  <span class="conum" data-value="2" />

        StringMessageStream stream = new StringMessageStream&lt;StringMessage&gt;();
        client.blockingUnary("Lower", stream);  <span class="conum" data-value="3" />
    }

    public static class StringMessageStream&lt;T&gt; { /* code omitted */ }
}</markup>

<ul class="colist">
<li data-value="1">This variant of the <code>unary</code> API takes the method name and a request object and returns
a <code>CompletableFuture&lt;Response&gt;</code> where <code>&lt;Response&gt;</code> is the response type. Here we invoke the
<code>Lower</code> method passing the input <code>StringMessage</code>. This method returns a <code>CompletableFuture&lt;StringMessage&gt;</code>
as its response thus allowing the client to obtain the result asynchronously.</li>
<li data-value="2">This is simply a wrapper around the above method. This method blocks until the result is available.</li>
<li data-value="3">Here, we invoke the <code>unary</code> method by passing the <code>StringMessageStream</code> whose <code>onNext</code> method
will be called (once) when the result is available.</li>
</ul>

</div>


<h4 id="_invoking_a_client_streaming_method_on_the_stringservice">Invoking a client streaming method on the StringService</h4>
<div class="section">
<p>Let&#8217;s invoke the <code>Join</code> method which causes the server to return a single result <strong>after</strong> the client
has streamed the request values to the server. The gRPC API expects the client application to provide
an instance of <code>io.grpc.StreamObserver</code> as an argument during the invocation of the client
streaming method.</p>

<p>In order to simplify the task of invoking Client Streaming methods, the Helidon Client Framework provides
two methods to invoke gRPC client Streaming methods. The first variant takes an <code>Iterable</code> as
argument which in turn is converted into a <code>io.grpc.StreamObserver</code>. The second variant takes a
<code>io.grpc.StreamObserver</code> as argument. The first variant can be used if the number of values to be
streamed in small and known a priori.</p>

<markup
lang="java"

>public class ProtoBasedStringServiceClient {

    private GrpcServiceClient client;

    public ProtoBasedStringServiceClient() { /* code omitted */ }

    public void invokeClientStreamingWithIterable() throws Exception {

        String sentence = "A simple invocation of a client streaming method";
        Collection&lt;StringMessage&gt; input = Arrays.stream(sentence.split(" "))  <span class="conum" data-value="1" />
                  .map(w -&gt; StringMessage.newBuilder().setText(w).build())
                  .collect(Collectors.toList());

        CompletableFuture&lt;StringMessage&gt; result =
                  grpcClient.clientStreaming("Join", input);  <span class="conum" data-value="2" />
    }

    public void invokeClientStreaming() throws Exception {
        String sentence = "A simple invocation of a client streaming method";
        StringMessageStream responseStream = new StringMessageStream&lt;StringMessage&gt;();
        StreamObserver&lt;StringMessage&gt; clientStream =
                  grpcClient.clientStreaming("Join", responseStream);  <span class="conum" data-value="3" />

        for (String word : sentence.split(" ")) {
            clientStream.onNext(StringMessage.newBuilder().setText(word).build());  <span class="conum" data-value="4" />
        }
        clientStream.onCompleted();  <span class="conum" data-value="5" />
    }

    public static class StringMessageStream&lt;T&gt; { /* code is omitted */ }

}</markup>

<ul class="colist">
<li data-value="1">We prepare the collection that contains the values to be streamed.</li>
<li data-value="2">We call the first variant of the <code>clientStreaming()</code> method that takes the
method name and the collection of values to be streamed from the client.
Note: The above helper method is useful if the values to be streamed is fixed and small in number.</li>
<li data-value="3">If the number of values to be streamed is large (or unknown), then it is better to use this
variant of the <code>clientStreaming()</code> method that takes a <code>io.grpc.StreamObserver</code> as an argument. This
method returns a client stream through which the client can stream (potentially a large number of)
value to the server.</li>
<li data-value="4">Once the client stream is obtained, the client streams the values using the <code>onNext()</code> method on the
stream.</li>
<li data-value="5">When all values have been stream, the client invokes the <code>onCompleted()</code> method signal that all values
have been streamed from the client.</li>
</ul>

</div>

</div>


<h3 id="_invoking_a_server_streaming_method_on_the_stringservice_generated_from_protoc">Invoking a server streaming method on the StringService (generated from <code>protoc</code>)</h3>
<div class="section">
<p>Let&#8217;s invoke the "Split" method which causes the server to stream the results back.</p>

<markup
lang="java"

>public class ProtoBasedStringServiceClient {

    private GrpcServiceClient client;

    public ProtoBasedStringServiceClient() { /* code omitted */ }

    public void invokeServerStreaming() throws Exception {
        String sentence = "This sentence will be split into words and sent back to client";
        StringMessage input = StringMessage.newBuilder().setText(sentence).build();  <span class="conum" data-value="1" />

        StringMessageStream&lt;StringMessage&gt; observer = new StringMessageStream&lt;&gt;();  <span class="conum" data-value="2" />
        grpcClient.serverStreaming("Split", input, observer);  <span class="conum" data-value="3" />
    }

    public static class StringMessageStream&lt;T&gt; { /* code is omitted */ }

}</markup>

<ul class="colist">
<li data-value="1">We prepare the input <code>StringMessage</code> that needs to be  split.</li>
<li data-value="2">We create a <code>StringMessageStream</code> which will receive the results streamed from the server.</li>
<li data-value="3">We call the <code>serverStreaming()</code> passing the input and the <code>StringMessageStream</code> as arguments.
The server sends a stream of words by calling the <code>onNext()</code> method on the <code>StringMessageStream</code> for
each word.</li>
</ul>

</div>


<h3 id="_invoking_a_bi_directional_streaming_method_on_the_stringservice_generated_from_protoc">Invoking a bi-directional streaming method on the StringService (generated from <code>protoc</code>)</h3>
<div class="section">
<p>Now let&#8217;s invoke the <code>Echo</code> method in which both the client and the server have to stream
the request and response.</p>

<markup
lang="java"

>public class ProtoBasedStringServiceClient {

    private GrpcServiceClient client;

    public ProtoBasedStringServiceClient() { /* code omitted */ }

    public void invokeBidiStreaming() throws Exception {

        StringMessageStream&lt;StringMessage&gt; observer = new StringMessageStream&lt;&gt;();  <span class="conum" data-value="1" />
        StringMessageStream&lt;StringMessage&gt; clientStream = grpcClient
                                .bidiStreaming("Echo", observer);  <span class="conum" data-value="2" />

        String sentence = "Each word will be echoed back to the client by the server";
        for (String word : sentence.split(" ")) {
            clientStream.onNext(StringMessage.newBuilder().setText(word).build());  <span class="conum" data-value="3" />
        }
        clientStream.onCompleted();  <span class="conum" data-value="4" />
    }

    public static class StringMessageStream&lt;T&gt; { /* code is omitted */ }

}</markup>

<ul class="colist">
<li data-value="1">We create a <code>StringMessageStream</code> which will receive the results streamed from the server.</li>
<li data-value="2">We call the <code>bidiStreaming()</code> passing the <code>observer</code> as argument. The server will
send its results through this stream (basically by calling the <code>onNext()</code> on the <code>observer</code>).
The method returns a (client) stream which should be used by the client to stream values to the
server.</li>
<li data-value="3">We stream each word in our sentence to the server by calling the <code>onNext()</code> method on the
<code>clientStream</code>.</li>
<li data-value="4">We call the <code>onCompleted()</code> method on the <code>clientStream</code> to signal that the client has
streamed all its values.</li>
</ul>

</div>


<h3 id="_programmatically_creating_clientservicedescriptor_for_stringservice">Programmatically creating ClientServiceDescriptor for StringService</h3>
<div class="section">
<p>Assuming that the service is still running on port 1408, let&#8217;s see how to create our Client
without using the <code>StringService</code>'s proto <code>ServiceDescriptor</code>.</p>

<p>Since we are <strong>not</strong> going to use the <code>StringService</code>'s proto <code>ServiceDescriptor</code>, we need to
describe the methods that the client needs to invoke. The Helidon client framework provides
several methods to easily describe gRPC methods.</p>

<p>For example, to register a unary method, we need to use the <code>unary</code> method and configure it to
specify the request and response types.</p>

<p>Other than describing the methods that our client will invoke, the rest of the code should be
very similar to (or the same as) the previous section!!</p>

<markup
lang="java"

>public class StringServiceClient {

    public static void main(String[] args) {
        ClientMethodDescriptor lower = ClientMethodDescriptor
                    .unary("StringService", "Lower")  <span class="conum" data-value="1" />
                    .requestType(StringMessage.class)  <span class="conum" data-value="2" />
                    .responseType(StringMessage.class)   <span class="conum" data-value="3" />
                    .build();  <span class="conum" data-value="4" />

        ClientMethodDescriptor join = ClientMethodDescriptor
                    .clientStreaming("StringService", "Join")   <span class="conum" data-value="5" />
                    .requestType(StringMessage.class)
                    .responseType(StringMessage.class)
                    .build();

        ClientMethodDescriptor split = ClientMethodDescriptor
                    .serverStreaming("StringService", "Split")  <span class="conum" data-value="6" />
                    .requestType(StringMessage.class)
                    .responseType(StringMessage.class)
                    .build();

        ClientMethodDescriptor echo = ClientMethodDescriptor
                    .bidirectional("StringService", "Echo")  <span class="conum" data-value="7" />
                    .requestType(StringMessage.class)
                    .responseType(StringMessage.class)
                    .build();

        ClientServiceDescriptor serviceDesc = ClientServiceDescriptor  <span class="conum" data-value="8" />
                    .builder(StringService.class)
                    .unary(lower)
                    .clientStreaming(join)
                    .serverStreaming(split)
                    .bidirectional(echo)
                    .build();


        Channel channel = ManagedChannelBuilder.forAddress("localhost", 1408)  <span class="conum" data-value="9" />
                .usePlaintext().build();

        GrpcServiceClient client = GrpcServiceClient.create(channel, serviceDesc);  // (<span class="conum" data-value="10" />

    }

}</markup>

<ul class="colist">
<li data-value="1">Use the <code>unary()</code> method on <code>ClientMethodDescriptor</code> to create a builder for a gRPC unary method.
The service name and the method name ("Lower") are specified.</li>
<li data-value="2">Set the request type of the method to be <code>StringMessage</code> (since the <code>Lower</code> method takes <code>StringMessage</code> as a parameter).</li>
<li data-value="3">Set the response type of the method to be <code>StringMessage</code> (since the <code>Lower</code> method returns a <code>StringMessage</code> as a parameter).</li>
<li data-value="4">Build the <code>ClientMethodDescriptor</code>. Note that the return value is a <code>ClientMethodDescriptor</code> that contains
the correct marshaller for the request &amp; response types.</li>
<li data-value="5">Use the <code>clientStreaming()</code> method on <code>ClientMethodDescriptor</code> to create a builder for a gRPC client streaming method.
The service name and the method name ("Join") are specified.</li>
<li data-value="6">Use the <code>serverStreaming()</code> method on <code>ClientMethodDescriptor</code> to create a builder for a gRPC server streaming method.
The service name and the method name ("Split") are specified.</li>
<li data-value="7">Use the <code>bidirectional()</code> method on <code>ClientMethodDescriptor</code> to create a builder for a gRPC Bidi streaming method.
The service name and the method name ("Echo") are specified.</li>
<li data-value="8">Create a <code>ClientServiceDescriptor</code> for a service named <code>StringService</code> and add all the defined  <code>ClientMethodDescriptor</code>s.</li>
<li data-value="9">We create a <code>Channel</code> to the service that is running on <code>localhost:1408</code>.</li>
<li data-value="10">Finally, we create our <code>GrpcServiceClient</code> by using the above-mentioned <code>ClientServiceDescriptor</code>
and <code>Channel</code>.</li>
</ul>

<p>At this point the <code>client</code> object can be used to invoke any of the four types of methods we have seen in the
earlier sections.</p>

</div>


<h3 id="_creating_grpc_clients_for_non_protobuf_services">Creating gRPC clients for non-protobuf services</h3>
<div class="section">
<p>If your service is <strong>not</strong> using protobuf for serialization, then the client framework allows
you to programmatically initialize <code>ClientMethodDescriptor</code> and create clients to invoke
methods on the service.</p>

<p>All you have to do is create the set of <code>ClientMethodDescriptor</code>s and the <code>ClientServiceDescriptor</code> as
described in the previous section. Just <strong>do not</strong> set the request and response types
in the <code>ClientMethodDescriptor</code> anymore. Furthermore, there is an API in the <code>ClientServiceDescriptor</code>
that makes this even simpler where you can simply pass the method name. For example, to create a client streaming
method called <code>"JoinString"</code> that uses some custom marshalling, simply call the <code>clientStreaming("JoinString")</code>.</p>

<markup
lang="java"

>public static void main(String[] args) throws Exception {
    ClientServiceDescriptor descriptor = ClientServiceDescriptor.builder(HelloService.class)  <span class="conum" data-value="1" />
                                                                .marshallerSupplier(new JsonbMarshaller.Supplier())  <span class="conum" data-value="2" />
                                                                .clientStreaming("JoinString")  <span class="conum" data-value="3" />
                                                                .build();

    Channel channel = ManagedChannelBuilder.forAddress("localhost", 1408)
                                           .usePlaintext()
                                           .build();

    GrpcServiceClient client = GrpcServiceClient.create(channel, descriptor);

    String sentence = "A simple invocation of a client streaming method";
    Collection&lt;StringMessage&gt; input = Arrays.stream(sentence.split(" "))
                                        .map(w -&gt; StringMessage.newBuilder().setText(w).build())
                                        .collect(Collectors.toList());

    CompletableFuture&lt;StringMessage&gt; result = grpcClient.clientStreaming("Join", input);
}</markup>

<ul class="colist">
<li data-value="1">Create a <code>ClientServiceDescriptor</code> for the <code>HelloService</code>.</li>
<li data-value="2">Specify a custom marshaller using the built-in JSON-B marshaller to serialize/deserialize requests and responses.</li>
<li data-value="3">Add the "JoinString" client streaming method to the <code>ClientServiceDescriptor</code>.</li>
</ul>

<p>Since we didn&#8217;t set the request or response type (like we did in the previous sections), the custom marshaller will be
used for Marshalling and Unmarshalling the request and response values.</p>

<p>Note that whether a <code>ClientServiceDescriptor</code> is built using protobuf artifacts or is built programmatically,
the same set of APIs provided by the Client Framework can be used to invoke gRPC methods.</p>

</div>


<h3 id="_marshalling">Marshalling</h3>
<div class="section">

<h4 id="_default_marshalling_support">Default Marshalling Support</h4>
<div class="section">
<p>Helidon gRPC supports Protobuf out of the box. The Protobuf marshaller will be used by default for any request and response classes that extend <code>com.google.protobuf.MessageLite</code>, which is the case for all classes generated from a <code>proto</code> file using <code>protoc</code> compiler.</p>

<p>That means that you don&#8217;t need any special handling or configuration in order to support Protobuf serialization of requests and responses.</p>

</div>


<h4 id="_custom_marshalling">Custom Marshalling</h4>
<div class="section">
<p>Helidon makes the use of custom marshallers trivial and provides one custom implementation, <a target="_blank" href="https://github.com/oracle/helidon/tree/master/grpc/core/src/main/java/io/helidon/grpc/core/JsonbMarshaller.java">JsonbMarshaller</a>, out of the box.</p>

<p>You can also easily implement your own marshaller to support serialization formats that are not supported natively
by Helidon, by implementing <code>Marshaller</code> and <code>MarshallerSupplier</code> interfaces. As an example, check out
the source code of the built-in marshaller:
<a target="_blank" href="https://github.com/oracle/helidon/tree/master/grpc/core/src/main/java/io/helidon/grpc/core/JsonbMarshaller.java">JsonbMarshaller.java</a>.</p>

<p>Furthermore, <a target="_blank" href="https://coherence.community/">Oracle Coherence CE</a> provides a marshaller for a highly optimized, binary, platform independent Portable Object Format (POF). You can find more information about POF in <a target="_blank" href="https://coherence.community/20.12/docs/#/docs/core/04_portable_types">Coherence documentation</a></p>


<h5 id="_setting_the_custom_marshaller">Setting the custom marshaller</h5>
<div class="section">
<p>You can set the custom marshaller supplier via the <code>ClientServiceDescriptor.builder.marshallerSupplier()</code> method:</p>

<markup
lang="java"
title="Sample code for setting the marshaller on the ClientServiceDescriptor"
>ClientServiceDescriptor descriptor = ClientServiceDescriptor
        .builder(HelloService.class)
        .marshallerSupplier(new JsonbMarshaller.Supplier())  <span class="conum" data-value="1" />
        .clientStreaming("JoinString")
        .build();</markup>

<ul class="colist">
<li data-value="1">Specify the custom marshaller to use.</li>
</ul>

</div>

</div>

</div>

</div>


<h2 id="_configuration">Configuration</h2>
<div class="section">
<p>Configure the gRPC client using the Helidon configuration framework, either programmatically or via a configuration file.
As mentioned earlier, creating a <code>GrpcServiceClient</code> involves:</p>

<ol style="margin-left: 15px;">
<li>
Creating a <code>ClientServiceDescriptor</code> which describes the methods in the service that this client can invoke.

</li>
<li>
Creating a gRPC <code>Channel</code> through which the client communicates with the server.

</li>
</ol>


<h3 id="_configuring_the_clientservicedescriptor">Configuring the ClientServiceDescriptor</h3>
<div class="section">
<p>The only way to configure the <code>ClientServiceDescriptor</code> is in your application code.</p>

<markup
lang="java"

>ClientServiceDescriptor descriptor = ClientServiceDescriptor
        .builder(HelloService.class)  <span class="conum" data-value="1" />
        .unary("SayHello")  <span class="conum" data-value="2" />
        .build();  <span class="conum" data-value="3" /></markup>

<ul class="colist">
<li data-value="1">Create a  builder for a <code>ClientServiceDescriptor</code> for the <code>HelloService</code>.</li>
<li data-value="2">Specify that the <code>HelloService</code> has a unary method named <code>SayHello</code>. There are many other methods in this class that allow you
to define <code>ClientStreaming</code>, <code>ServerStreaming</code> and <code>Bidirectional</code> methods.</li>
<li data-value="3">Build the <code>ClientServiceDescriptor</code>.</li>
</ul>

</div>


<h3 id="_configuring_the_grpc_channel">Configuring the gRPC Channel</h3>
<div class="section">
<p>gRPC allows various channel configurations (deadlines, retries, interceptors etc.)</p>

<p>Please refer to gRPC documentation: <a target="_blank" href="https://grpc.io/grpc-java/javadoc/io/grpc/ManagedChannelBuilder.html" class="bare">https://grpc.io/grpc-java/javadoc/io/grpc/ManagedChannelBuilder.html</a>.</p>

</div>

</div>


<h2 id="_examples">Examples</h2>
<div class="section">

<h3 id="_quick_start">Quick Start</h3>
<div class="section">
<p>First, create and run a minimalist <code>HelloService</code> gRPC server application as described in the
<router-link :to="{path: '/se/grpc/server', hash: '#_quick_start'}">gRPC server quick start example</router-link>.</p>

<p>Assuming that the server is running on port 1408, create a client as follows:</p>

<markup
lang="java"

>public static void main(String[] args) throws Exception {
    ClientServiceDescriptor descriptor = ClientServiceDescriptor.builder(HelloService.class)  <span class="conum" data-value="1" />
                                                                .marshallerSupplier(new JsonbMarshaller.Supplier()) <span class="conum" data-value="2" />
                                                                .unary("SayHello")  <span class="conum" data-value="3" />
                                                                .build();

    Channel channel = ManagedChannelBuilder.forAddress("localhost", 1408)  <span class="conum" data-value="4" />
                                           .usePlaintext()
                                           .build();

    GrpcServiceClient client = GrpcServiceClient.create(channel, descriptor);  <span class="conum" data-value="5" />

    CompletionStage&lt;String&gt; future = client.unary("SayHello", "Helidon gRPC!!");  <span class="conum" data-value="6" />
    System.out.println(future.get());  <span class="conum" data-value="7" />

}</markup>

<ul class="colist">
<li data-value="1">Create a <code>ClientServiceDescriptor</code> for the <code>HelloService</code>.</li>
<li data-value="2">Specify a custom marshaller using the built-in JSON-B marshaller to serialize/deserialize request and response values.</li>
<li data-value="3">Add the <code>SayHello</code> unary method to the <code>ClientServiceDescriptor</code> which will use the specified custom marshaller.</li>
<li data-value="4">Create a gRPC <code>Channel</code> that will communicate with the server running in localhost and on port 1408 (using plaintext).</li>
<li data-value="5">Create the <code>GrpcServiceClient</code> that uses the above <code>Channel</code> and <code>ClientServiceDescriptor</code>. <code>GrpcClientService</code> represents
a client that can be used to define the set of methods described by the specified <code>ClientServiceDescriptor</code>. In our case, the
<code>ClientServiceDescriptor</code> defines one unary method called <code>SayHello</code>.</li>
<li data-value="6">Invoke the <code>SayHello</code> method which returns a <code>CompletionStage&lt;String&gt;</code>.</li>
<li data-value="7">Print the result.</li>
</ul>

</div>


<h3 id="_additional_grpc_client_examples">Additional gRPC client examples</h3>
<div class="section">
<p>A set of gRPC client examples for Helidon SE can be found in the following links:</p>

<ul class="ulist">
<li>
<p><a target="_blank" href="https://github.com/oracle/helidon/tree/master/examples/grpc/client-standalone">Basic gRPC Standalone Client</a></p>

</li>
<li>
<p><a target="_blank" href="https://github.com/oracle/helidon/tree/master/examples/grpc/metrics">gRPC Server Metrics</a></p>

</li>
<li>
<p><a target="_blank" href="https://github.com/oracle/helidon/tree/master/examples/grpc/opentracing">OpenTracing on a gRPC Server</a></p>

</li>
<li>
<p><a target="_blank" href="https://github.com/oracle/helidon/tree/master/examples/grpc/security">Basic Auth Security on a gRPC Server</a></p>

</li>
<li>
<p><a target="_blank" href="https://github.com/oracle/helidon/tree/master/examples/grpc/security-abac">Attribute-Based Access Control (ABAC) security on a gRPC Server</a></p>

</li>
<li>
<p><a target="_blank" href="https://github.com/oracle/helidon/tree/master/examples/grpc/security-outbound">Outbound Security on a gRPC Server</a></p>

</li>
</ul>

</div>

</div>

</doc-view>
