<doc-view>

<h2 id="_contents">Contents</h2>
<div class="section">
<ul class="ulist">
<li>
<p><router-link to="#_overview" @click.native="this.scrollFix('#_overview')">Overview</router-link></p>

</li>
<li>
<p><router-link to="#maven-coordinates" @click.native="this.scrollFix('#maven-coordinates')">Maven Coordinates</router-link></p>

</li>
<li>
<p><router-link to="#_usage" @click.native="this.scrollFix('#_usage')">Usage</router-link></p>

</li>
<li>
<p><router-link to="#_api" @click.native="this.scrollFix('#_api')">API</router-link></p>

</li>
<li>
<p><router-link to="#_examples" @click.native="this.scrollFix('#_examples')">Examples</router-link></p>

</li>
<li>
<p><router-link to="#_reference" @click.native="this.scrollFix('#_reference')">Reference</router-link></p>

</li>
</ul>

</div>


<h2 id="_overview">Overview</h2>
<div class="section">
<p>Helidon integrates with <a target="_blank" href="https://projects.eclipse.org/projects/ee4j.tyrus">Tyrus</a> to provide support for the
<a target="_blank" href="https://jakarta.ee/specifications/websocket/2.1/jakarta-websocket-spec-2.1.html">Jakarta WebSocket API</a>.</p>

</div>


<h2 id="maven-coordinates">Maven Coordinates</h2>
<div class="section">
<p>To enable Jakarta Websocket
add the following dependency to your project&#8217;s <code>pom.xml</code> (see
 <router-link to="/about/managing-dependencies">Managing Dependencies</router-link>).</p>

<markup
lang="xml"

>&lt;dependency&gt;
    &lt;groupId&gt;io.helidon.microprofile.websocket&lt;/groupId&gt;
    &lt;artifactId&gt;helidon-microprofile-websocket&lt;/artifactId&gt;
&lt;/dependency&gt;</markup>

</div>


<h2 id="_usage">Usage</h2>
<div class="section">
<p>The WebSocket API enables Java applications to participate in WebSocket interactions
as both servers and clients. The server API supports two flavors: annotated and
programmatic endpoints.</p>

<p>Annotated endpoints, as suggested by their name, use Java annotations to provide
the necessary meta-data to define WebSocket handlers; programmatic endpoints
implement API interfaces and are annotation free. Annotated endpoints tend to be
more flexible since they allow different method signatures depending on the
application needs, whereas programmatic endpoints must implement an interface
and are, therefore, bounded to its definition.</p>

<p>Helidon MP support is centered around annotations and bean discovery using
CDI. Developers can choose between annotated and programmatic endpoints or use
any combination of them. Using annotated endpoints is recommended in MP as
they usually result in more succinct and easier-to-read code.</p>

</div>


<h2 id="_api">API</h2>
<div class="section">

<div class="table__overflow elevation-1  flex sm10
">
<table class="datatable table">
<colgroup>
<col style="width: 25%;">
<col style="width: 75%;">
</colgroup>
<thead>
<tr>
<th>Annotation</th>
<th>Description</th>
</tr>
</thead>
<tbody>
<tr>
<td class="">@ServerEndpoint</td>
<td class="">This class level annotation declares that the class it decorates is a web socket endpoint that will be deployed and made available in the URI-space of a web socket server. The annotation allows the developer to define the URL (or URI template) which this endpoint will be published, and other important properties of the endpoint to the websocket runtime, such as the encoders it uses to send messages.</td>
</tr>
<tr>
<td class="">@ClientEndpoint</td>
<td class="">The ClientEndpoint annotation, a class level annotation, is used to denote that a POJO is a web socket client and can be deployed as such. Similar to <code>@ServerEndpoint</code>, POJOs that are annotated with this annotation can have methods that, using the web socket method level annotations, are web socket lifecycle methods.</td>
</tr>
<tr>
<td class="">@OnOpen</td>
<td class="">This method level annotation can be used to decorate a Java method that will be called when a new web socket session is open.</td>
</tr>
<tr>
<td class="">@OnMessage</td>
<td class="">This method level annotation can be used to make a Java method receive incoming web socket messages. Each websocket endpoint may only have one message handling method for each of the native websocket message formats: text, binary and pong.</td>
</tr>
<tr>
<td class="">@OnError</td>
<td class="">This method level annotation can be used to decorate a Java method that will be called in order to handle errors.</td>
</tr>
<tr>
<td class="">@OnClose</td>
<td class="">This method level annotation can be used to decorate a Java method that will be called when a web socket session is closing.</td>
</tr>
</tbody>
</table>
</div>

</div>


<h2 id="_examples">Examples</h2>
<div class="section">
<p>This section describes the implementation of a simple application that uses a REST resource to push messages into a shared queue and a WebSocket endpoint to download messages from the queue, one at a time, over a connection. The example will show how REST and WebSocket connections can be seamlessly combined into a Helidon application.</p>

<p>The Helidon MP application shown here takes full advantage of CDI and class scanning and does not require any additional code given that the necessary information is available from the  code annotations.</p>

<p>The REST endpoint is implemented as a JAX-RS resource, and the shared queue (in application scope) is directly injected:</p>

<markup
lang="java"

>@Path("rest")
public class MessageQueueResource {

    @Inject
    private MessageQueue messageQueue;

    @POST
    @Consumes("text/plain")
    public void push(String s) {
        messageQueue.push(s);
    }
}</markup>

<p>Here we opt for the use of an annotated WebSocket endpoint decorated by <code>@ServerEndpoint</code> that provides all the meta-data necessary for Helidon to create the endpoint.</p>

<markup
lang="java"

>@ServerEndpoint(
        value = "/websocket",
        encoders = {
                UppercaseEncoder.class
        })
public class MessageBoardEndpoint {

    @Inject
    private MessageQueue messageQueue;

    @OnMessage
    public void onMessage(Session session, String message)
            throws EncodeException, IOException {

        if (message.equals("SEND")) {
            while (!messageQueue.isEmpty()) {
                session.getBasicRemote()
                        .sendObject(messageQueue.pop());
            }
        }
    }
}</markup>

<p>Since <code>MessageBoardEndpoint</code> is just a POJO, it uses additional annotations for event handlers such as <code>@OnMessage</code>. One advantage of this approach, much like in the JAX-RS API, is that method signatures are not fixed. In the snipped above, the parameters (which could be specified in any order!) include the WebSocket session and the message received that triggered the call.</p>

<p>So what else is needed to run this Helidon MP app? Nothing else other than the supporting classes <code>MessageQueue</code> and <code>UppercaseEncoder</code>. Helidon MP declares both <code>@Path</code> and <code>@ServerEndpoint</code> as bean defining annotation, so all that is needed is for CDI discovery to be enabled --typically in your <code>beans.xml</code> file.</p>

<p>By default, both JAX-RS resources and WebSocket endpoints will be available under the <em>root path</em> <code>"/"</code>. This default value can be overridden by providing subclasses/implementations for <code>jakarta.ws.rs.Application</code> and <code>jakarta.websocket.server.ServerApplicationConfig</code>, respectively. JAX-RS uses <code>@ApplicationPath</code> on application subclasses to provide this root path, but since there is no equivalent in the WebSocket API, Helidon MP uses its own annotation <code>@RoutingPath</code> on <code>jakarta.websocket.server.ServerApplicationConfig</code> implementations.</p>

<p>For instance, if in our example we include the following class:</p>

<markup
lang="java"

>@ApplicationScoped
@RoutingPath("/web")
public class MessageBoardApplication implements ServerApplicationConfig {
    @Override
    public Set&lt;ServerEndpointConfig&gt; getEndpointConfigs(
            Set&lt;Class&lt;? extends Endpoint&gt;&gt; endpoints) {
        return Set.of(); // No programmatic endpoints
    }

    @Override
    public Set&lt;Class&lt;?&gt;&gt; getAnnotatedEndpointClasses(Set&lt;Class&lt;?&gt;&gt; endpoints) {
        return endpoints; // Returned scanned endpoints
    }
}</markup>

<p>the root path for WebSocket endpoints will be  <code>"/web"</code> instead of the default <code>"/"</code>. Note that <code>@RoutingPath</code> is <em>not</em> a bean defining annotation, thus the need to use <code>@ApplicationScoped</code> --which, as before, requires CDI bean discovery mode to be <code>annotated</code>. In addition to <code>@RoutingPath</code>, these classes can be annotated with <code>@RoutingName</code> to associate an endpoint with a Helidon named socket. Please refer to the Javadoc of that annotation for additional information.</p>

<p>All endpoint methods in Helidon MP are executed in a separate thread pool, independently of Netty. Therefore, there is no need to create additional threads for blocking or long-running operations as these will not affect Netty&#8217;s ability to process networking data.</p>

<p>For more information see the <a target="_blank" href="https://github.com/oracle/helidon/tree/4.0.10/examples/webserver/websocket">example</a>.</p>


<h3 id="_websocket_endpoints_on_different_ports">WebSocket Endpoints on Different Ports</h3>
<div class="section">
<p>The Helidon WebServer can listen on multiple ports or sockets. This can be useful when APIs for different type of users need to be exposed (such as admin vs. non-admin users). Just like for REST resources, it is possible to expose  WebSocket applications on different ports <em>provided that the routing paths are different</em> --this is due to a constraint in Tyrus, given that it is simply unaware that the endpoints are bound to different ports in the Helidon WebServer.</p>

<div class="admonition note">
<p class="admonition-inline">In practice, this implies that the value of <code>@RoutingPath</code>, or the equivalent entry in config, must be different across sockets to satisfy the restriction in Tyrus. An attempt to register two or more endpoints on the same path, even if they belong to applications registered on different ports, shall result in a <code>jakarta.websocket.DeploymentException</code> being thrown.</p>
</div>

<p>We can modify the <code>MessageBoardApplication</code> above and bind it to a non-default socket as follows:</p>

<markup
lang="java"

>@ApplicationScoped
@RoutingPath("/web")
@RoutingName(value = "admin", required = true)
public class MessageBoardApplication implements ServerApplicationConfig {
    @Override
    public Set&lt;ServerEndpointConfig&gt; getEndpointConfigs(
            Set&lt;Class&lt;? extends Endpoint&gt;&gt; endpoints) {
        return Set.of(); // No programmatic endpoints
    }

    @Override
    public Set&lt;Class&lt;?&gt;&gt; getAnnotatedEndpointClasses(Set&lt;Class&lt;?&gt;&gt; endpoints) {
        return endpoints; // Returned scanned endpoints
    }
}</markup>

<p>The value of the <code>@RoutingName</code> annotation must match that of a configured application socket as shown in the following <code>application.yaml</code> file:</p>

<markup


>server:
  port: 8080
  host: 0.0.0.0
  sockets:
    - name: admin
      port: 8888</markup>

<p>This example assumes that port 8888 is reserved for admin users and binds the <code>MessageBoardApplication</code> to it.</p>

</div>

</div>


<h2 id="_reference">Reference</h2>
<div class="section">
<ul class="ulist">
<li>
<p><a target="_blank" href="https://projects.eclipse.org/proposals/eclipse-tyrus">Eclipse Tyrus</a></p>

</li>
<li>
<p><a target="_blank" href="https://datatracker.ietf.org/doc/html/rfc6455">WebSocket RFC 6455</a></p>

</li>
<li>
<p><a target="_blank" href="/apidocs/io.helidon.microprofile.tyrus/module-summary.html">Helidon MicroProfile Tyrus Javadoc</a></p>

</li>
</ul>

</div>

</doc-view>
