/*
 * Copyright (c) 2020, 2023 Oracle and/or its affiliates.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package io.helidon.dbclient;

import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.ServiceLoader;

import io.helidon.common.HelidonServiceLoader;
import io.helidon.dbclient.spi.DbClientProvider;

/**
 * Loads database client providers from Java Service loader.
 */
final class DbClientProviderLoader {

    private static final Map<String, DbClientProvider> PROVIDERS;
    private static final String[] NAMES;
    private static final DbClientProvider FIRST;

    static {
        HelidonServiceLoader<DbClientProvider> serviceLoader = HelidonServiceLoader
                .builder(ServiceLoader.load(DbClientProvider.class))
                .build();
        List<DbClientProvider> providers = serviceLoader.asList();
        Map<String, DbClientProvider> providersMap = new HashMap<>(providers.size());
        providers.forEach(dbProvider -> providersMap.put(dbProvider.name(), dbProvider));

        FIRST = providers.isEmpty() ? null : providers.get(0);
        PROVIDERS = Map.copyOf(providersMap);
        NAMES = PROVIDERS.keySet().toArray(new String[PROVIDERS.size()]);
    }

    private DbClientProviderLoader() {
        throw new UnsupportedOperationException("Instances of DbClientProviderLoader are not allowed");
    }

    /**
     * Get the first provider.
     *
     * @return first provider
     */
    static DbClientProvider first() {
        return FIRST;
    }

    /**
     * Lookup a provider by name.
     *
     * @param name provider name
     * @return optional provider
     */
    static Optional<DbClientProvider> get(String name) {
        return Optional.ofNullable(PROVIDERS.get(name));
    }

    /**
     * Get the discovered provider names.
     *
     * @return provider names
     */
    static String[] names() {
        return NAMES;
    }

}
