/*
 * Decompiled with CFR 0.152.
 */
package io.helidon.config.git;

import io.helidon.common.OptionalHelper;
import io.helidon.config.Config;
import io.helidon.config.ConfigException;
import io.helidon.config.ConfigHelper;
import io.helidon.config.git.GitConfigSourceBuilder;
import io.helidon.config.internal.FileSourceHelper;
import io.helidon.config.spi.AbstractParsableConfigSource;
import io.helidon.config.spi.ConfigParser;
import java.io.File;
import java.io.IOException;
import java.io.StringReader;
import java.net.URI;
import java.nio.file.DirectoryStream;
import java.nio.file.FileVisitResult;
import java.nio.file.FileVisitor;
import java.nio.file.Files;
import java.nio.file.Path;
import java.nio.file.SimpleFileVisitor;
import java.nio.file.attribute.BasicFileAttributes;
import java.nio.file.attribute.FileAttribute;
import java.time.Instant;
import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import java.util.Optional;
import java.util.logging.Level;
import java.util.logging.Logger;
import org.eclipse.jgit.api.CloneCommand;
import org.eclipse.jgit.api.Git;
import org.eclipse.jgit.api.PullCommand;
import org.eclipse.jgit.api.PullResult;
import org.eclipse.jgit.api.Status;
import org.eclipse.jgit.api.errors.GitAPIException;
import org.eclipse.jgit.api.errors.JGitInternalException;
import org.eclipse.jgit.lib.Repository;
import org.eclipse.jgit.storage.file.FileRepositoryBuilder;

public class GitConfigSource
extends AbstractParsableConfigSource<byte[]> {
    private static final Logger LOGGER = Logger.getLogger(GitConfigSource.class.getName());
    private final URI uri;
    private final String branch;
    private Path directory;
    private Path targetPath;
    private Repository repository;
    private GitConfigSourceBuilder.GitEndpoint endpoint;
    private boolean isTempDirectory = false;
    private boolean isClosed = false;
    private final List<Git> gits = Collections.synchronizedList(new ArrayList());

    public static GitConfigSource create(Config config) {
        return GitConfigSourceBuilder.create(config).build();
    }

    GitConfigSource(GitConfigSourceBuilder builder, GitConfigSourceBuilder.GitEndpoint endpoint) {
        super((AbstractParsableConfigSource.Builder)builder);
        this.endpoint = endpoint;
        this.uri = endpoint.uri();
        this.branch = endpoint.branch();
        if (endpoint.directory() == null) {
            if (this.uri == null) {
                throw new ConfigException("Directory or Uri must be set.");
            }
            try {
                this.directory = Files.createTempDirectory("helidon-config-git-source-", new FileAttribute[0]);
                this.isTempDirectory = true;
            }
            catch (IOException e) {
                throw new ConfigException("Cannot create temporary directory.", (Throwable)e);
            }
        } else {
            this.directory = endpoint.directory();
        }
        try {
            this.init();
            this.targetPath = this.directory.resolve(endpoint.path());
        }
        catch (IOException | GitAPIException | JGitInternalException e) {
            throw new ConfigException(String.format("Cannot initialize repository '%s' in local temp dir %s", this.uri.toASCIIString(), this.directory.toString()), e);
        }
    }

    private void init() throws IOException, GitAPIException {
        block13: {
            if (!this.directory.toFile().exists()) {
                throw new ConfigException(String.format("Directory '%s' does not exist.", this.directory.toString()));
            }
            if (!this.directory.toFile().isDirectory()) {
                throw new ConfigException(String.format("'%s' is not a directory.", this.directory.toString()));
            }
            if (!this.directory.toFile().canRead() || !this.directory.toFile().canWrite()) {
                throw new ConfigException(String.format("Directory '%s' is not accessible.", this.directory.toString()));
            }
            try (DirectoryStream<Path> dirStream = Files.newDirectoryStream(this.directory);){
                if (dirStream.iterator().hasNext()) {
                    try {
                        this.recordGit(Git.open((File)this.directory.toFile()));
                        break block13;
                    }
                    catch (IOException e) {
                        throw new ConfigException(String.format("Directory '%s' is not empty and it is not a valid repository.", this.directory.toString()));
                    }
                }
                if (this.uri != null) {
                    CloneCommand cloneCommand = ((CloneCommand)Git.cloneRepository().setCredentialsProvider(this.endpoint.credentialsProvider())).setURI(this.uri.toASCIIString()).setBranchesToClone(Collections.singleton("refs/heads/" + this.branch)).setBranch("refs/heads/" + this.branch).setDirectory(this.directory.toFile());
                    Git cloneResult = this.recordGit(cloneCommand.call());
                    LOGGER.log(Level.CONFIG, () -> String.format("git clone result: %s", cloneResult.toString()));
                }
            }
        }
        this.repository = ((FileRepositoryBuilder)new FileRepositoryBuilder().setGitDir(this.directory.resolve(".git").toFile())).build();
    }

    private void pull() throws GitAPIException {
        Git git = this.recordGit(Git.wrap((Repository)this.repository));
        PullCommand pull = ((PullCommand)git.pull().setCredentialsProvider(this.endpoint.credentialsProvider())).setRebase(true);
        PullResult result = pull.call();
        if (!result.isSuccessful()) {
            LOGGER.log(Level.WARNING, () -> String.format("Cannot pull from git '%s', branch '%s'", this.uri.toASCIIString(), this.branch));
            if (LOGGER.isLoggable(Level.FINEST)) {
                Status status = git.status().call();
                LOGGER.finest(() -> "git status cleanliness: " + status.isClean());
                if (!status.isClean()) {
                    LOGGER.finest(() -> "git status uncommitted changes: " + status.getUncommittedChanges());
                    LOGGER.finest(() -> "git status untracked: " + status.getUntracked());
                }
            }
        } else {
            LOGGER.fine("Pull was successful.");
        }
        LOGGER.finest(() -> "git rebase result: " + result.getRebaseResult().getStatus().name());
        LOGGER.finest(() -> "git fetch result: " + result.getFetchResult().getMessages());
    }

    protected String uid() {
        StringBuilder sb = new StringBuilder();
        if (this.endpoint.directory() != null) {
            sb.append(this.endpoint.directory());
        }
        if (this.endpoint.uri() != null && this.endpoint.directory() != null) {
            sb.append('|');
        }
        if (this.endpoint.uri() != null) {
            sb.append(this.endpoint.uri().toASCIIString());
        }
        sb.append('#');
        sb.append(this.endpoint.path());
        return sb.toString();
    }

    protected String mediaType() {
        return OptionalHelper.from(Optional.ofNullable(super.mediaType())).or(this::probeContentType).asOptional().orElse(null);
    }

    private Optional<String> probeContentType() {
        return Optional.ofNullable(ConfigHelper.detectContentType((Path)this.targetPath));
    }

    protected Optional<byte[]> dataStamp() {
        try {
            this.pull();
        }
        catch (GitAPIException e) {
            LOGGER.log(Level.WARNING, "Pull failed.", e);
        }
        return Optional.ofNullable(FileSourceHelper.digest((Path)this.targetPath));
    }

    private Instant lastModifiedTime(Path path) {
        return FileSourceHelper.lastModifiedTime((Path)path);
    }

    protected ConfigParser.Content<byte[]> content() throws ConfigException {
        Instant lastModifiedTime = this.lastModifiedTime(this.targetPath);
        LOGGER.log(Level.FINE, String.format("Getting content from '%s'. Last stamp is %s.", this.targetPath, lastModifiedTime));
        LOGGER.finest(FileSourceHelper.safeReadContent((Path)this.targetPath));
        Optional<byte[]> stamp = this.dataStamp();
        return ConfigParser.Content.create((Readable)new StringReader(FileSourceHelper.safeReadContent((Path)this.targetPath)), (String)this.mediaType(), stamp);
    }

    GitConfigSourceBuilder.GitEndpoint gitEndpoint() {
        return this.endpoint;
    }

    private Git recordGit(Git git) {
        this.gits.add(git);
        return git;
    }

    public void close() throws IOException {
        if (!this.isClosed) {
            try {
                if (this.repository != null) {
                    this.repository.close();
                }
                this.closeGits();
                if (this.isTempDirectory) {
                    this.deleteTempDirectory();
                }
            }
            finally {
                this.isClosed = true;
            }
        }
    }

    private void closeGits() {
        this.gits.forEach(Git::close);
    }

    private void deleteTempDirectory() throws IOException {
        LOGGER.log(Level.FINE, () -> String.format("GitConfigSource deleting temp directory %s", this.directory.toString()));
        Files.walkFileTree(this.directory, (FileVisitor<? super Path>)new SimpleFileVisitor<Path>(){

            @Override
            public FileVisitResult visitFile(Path file, BasicFileAttributes attrs) throws IOException {
                if (!Files.isWritable(file)) {
                    file.toFile().setWritable(true);
                }
                Files.delete(file);
                return FileVisitResult.CONTINUE;
            }

            @Override
            public FileVisitResult postVisitDirectory(Path dir, IOException exc) throws IOException {
                Files.delete(dir);
                return FileVisitResult.CONTINUE;
            }
        });
    }
}

