/*
 * Copyright (c) 2023 Oracle and/or its affiliates.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package io.helidon.common.socket;

import java.net.SocketOption;
import java.time.Duration;
import java.util.Collections;
import java.util.LinkedHashMap;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;

import io.helidon.builder.api.Prototype;
import io.helidon.common.Generated;
import io.helidon.common.config.Config;

/**
 * Socket options.
 *
 * @see #builder()
 * @see #create()
 */
@Generated(value = "io.helidon.builder.processor.BlueprintProcessor", trigger = "io.helidon.common.socket.SocketOptionsBlueprint")
public interface SocketOptions extends SocketOptionsBlueprint, Prototype.Api {

    /**
     * Create a new fluent API builder to customize configuration.
     *
     * @return a new builder
     */
    static SocketOptions.Builder builder() {
        return new SocketOptions.Builder();
    }

    /**
     * Create a new fluent API builder from an existing instance.
     *
     * @param instance an existing instance used as a base for the builder
     * @return a builder based on an instance
     */
    static SocketOptions.Builder builder(SocketOptions instance) {
        return SocketOptions.builder().from(instance);
    }

    /**
     * Create a new instance from configuration.
     *
     * @param config used to configure the new instance
     * @return a new instance configured from configuration
     */
    static SocketOptions create(Config config) {
        return SocketOptions.builder().config(config).buildPrototype();
    }

    /**
     * Create a new instance with default values.
     *
     * @return a new instance
     */
    static SocketOptions create() {
        return SocketOptions.builder().buildPrototype();
    }

    /**
     * Fluent API builder base for {@link SocketOptions}.
     *
     * @param <BUILDER> type of the builder extending this abstract builder
     * @param <PROTOTYPE> type of the prototype interface that would be built by {@link #buildPrototype()}
     */
    abstract class BuilderBase<BUILDER extends SocketOptions.BuilderBase<BUILDER, PROTOTYPE>, PROTOTYPE extends SocketOptions> implements Prototype.ConfiguredBuilder<BUILDER, PROTOTYPE> {

        private final Map<SocketOption<?>, Object> socketOptions = new LinkedHashMap<>();
        private boolean socketKeepAlive = true;
        private boolean socketReuseAddress = true;
        private boolean tcpNoDelay = false;
        private Config config;
        private Duration connectTimeout = Duration.parse("PT10S");
        private Duration readTimeout = Duration.parse("PT30S");
        private int socketReceiveBufferSize = 32768;
        private int socketSendBufferSize = 32768;

        /**
         * Protected to support extensibility.
         */
        protected BuilderBase() {
        }

        /**
         * Update this builder from an existing prototype instance.
         *
         * @param prototype existing prototype to update this builder from
         * @return updated builder instance
         */
        public BUILDER from(SocketOptions prototype) {
            addSocketOptions(prototype.socketOptions());
            connectTimeout(prototype.connectTimeout());
            readTimeout(prototype.readTimeout());
            socketReceiveBufferSize(prototype.socketReceiveBufferSize());
            socketSendBufferSize(prototype.socketSendBufferSize());
            socketReuseAddress(prototype.socketReuseAddress());
            socketKeepAlive(prototype.socketKeepAlive());
            tcpNoDelay(prototype.tcpNoDelay());
            return self();
        }

        /**
         * Update this builder from an existing prototype builder instance.
         *
         * @param builder existing builder prototype to update this builder from
         * @return updated builder instance
         */
        public BUILDER from(SocketOptions.BuilderBase<?, ?> builder) {
            addSocketOptions(builder.socketOptions());
            connectTimeout(builder.connectTimeout());
            readTimeout(builder.readTimeout());
            socketReceiveBufferSize(builder.socketReceiveBufferSize());
            socketSendBufferSize(builder.socketSendBufferSize());
            socketReuseAddress(builder.socketReuseAddress());
            socketKeepAlive(builder.socketKeepAlive());
            tcpNoDelay(builder.tcpNoDelay());
            return self();
        }

        /**
         * Update builder from configuration (node of this type).
         * If a value is present in configuration, it would override currently configured values.
         *
         * @param config configuration instance used to obtain values to update this builder
         * @return updated builder instance
         */
        @Override
        public BUILDER config(Config config) {
            Objects.requireNonNull(config);
            this.config = config;
            config.get("connect-timeout").as(Duration.class).ifPresent(this::connectTimeout);
            config.get("read-timeout").as(Duration.class).ifPresent(this::readTimeout);
            config.get("socket-receive-buffer-size").as(Integer.class).ifPresent(this::socketReceiveBufferSize);
            config.get("socket-send-buffer-size").as(Integer.class).ifPresent(this::socketSendBufferSize);
            config.get("socket-reuse-address").as(Boolean.class).ifPresent(this::socketReuseAddress);
            config.get("socket-keep-alive").as(Boolean.class).ifPresent(this::socketKeepAlive);
            config.get("tcp-no-delay").as(Boolean.class).ifPresent(this::tcpNoDelay);
            return self();
        }

        /**
         * This method replaces all values with the new ones.
         *
         * @param socketOptions custom socket options
         * @return updated builder instance
         * @see #socketOptions()
         */
        public BUILDER socketOptions(Map<SocketOption<?>, ?> socketOptions) {
            Objects.requireNonNull(socketOptions);
            this.socketOptions.clear();
            this.socketOptions.putAll(socketOptions);
            return self();
        }

        /**
         * This method keeps existing values, then puts all new values into the map.
         *
         * @param socketOptions custom socket options
         * @return updated builder instance
         * @see #socketOptions()
         */
        public BUILDER addSocketOptions(Map<SocketOption<?>, ?> socketOptions) {
            Objects.requireNonNull(socketOptions);
            this.socketOptions.putAll(socketOptions);
            return self();
        }

        /**
         * This method adds a new value to the map, or replaces it if the key already exists.
         *
         * @param key key to add or replace
         * @param socketOption new value for the key
         * @param <TYPE> Type to correctly map key and value
         * @return updated builder instance
         * @see #socketOptions()
         */
        public <TYPE extends Object> BUILDER putSocketOption(SocketOption<TYPE> key, TYPE socketOption) {
            Objects.requireNonNull(key);
            Objects.requireNonNull(socketOption);
            this.socketOptions.put(key, socketOption);
            return self();
        }

        /**
         * Socket connect timeout. Default is 10 seconds.
         *
         * @param connectTimeout connect timeout duration
         * @return updated builder instance
         * @see #connectTimeout()
         */
        public BUILDER connectTimeout(Duration connectTimeout) {
            Objects.requireNonNull(connectTimeout);
            this.connectTimeout = connectTimeout;
            return self();
        }

        /**
         * Socket read timeout. Default is 30 seconds.
         *
         * @param readTimeout read timeout duration
         * @return updated builder instance
         * @see #readTimeout()
         */
        public BUILDER readTimeout(Duration readTimeout) {
            Objects.requireNonNull(readTimeout);
            this.readTimeout = readTimeout;
            return self();
        }

        /**
         * Socket receive buffer size.
         *
         * @param socketReceiveBufferSize buffer size, in bytes
         * @return updated builder instance
         * @see #socketReceiveBufferSize()
         */
        public BUILDER socketReceiveBufferSize(int socketReceiveBufferSize) {
            this.socketReceiveBufferSize = socketReceiveBufferSize;
            return self();
        }

        /**
         * Socket send buffer size.
         *
         * @param socketSendBufferSize buffer size, in bytes
         * @return updated builder instance
         * @see #socketSendBufferSize()
         */
        public BUILDER socketSendBufferSize(int socketSendBufferSize) {
            this.socketSendBufferSize = socketSendBufferSize;
            return self();
        }

        /**
         * Socket reuse address.
         * Default is {@code true}.
         *
         * @param socketReuseAddress whether to reuse address
         * @return updated builder instance
         * @see #socketReuseAddress()
         */
        public BUILDER socketReuseAddress(boolean socketReuseAddress) {
            this.socketReuseAddress = socketReuseAddress;
            return self();
        }

        /**
         * Configure socket keep alive.
         * Default is {@code true}.
         *
         * @param socketKeepAlive keep alive
         * @return updated builder instance
         * @see #socketKeepAlive()
         */
        public BUILDER socketKeepAlive(boolean socketKeepAlive) {
            this.socketKeepAlive = socketKeepAlive;
            return self();
        }

        /**
         * This option may improve performance on some systems.
         * Default is {@code false}.
         *
         * @param tcpNoDelay whether to use TCP_NODELAY, defaults to {@code false}
         * @return updated builder instance
         * @see #tcpNoDelay()
         */
        public BUILDER tcpNoDelay(boolean tcpNoDelay) {
            this.tcpNoDelay = tcpNoDelay;
            return self();
        }

        /**
         * Arbitrary socket options. Socket options that have dedicated methods
         * in this type will be ignored if configured within the map.
         *
         * @return the socket options
         */
        public Map<SocketOption<?>, Object> socketOptions() {
            return socketOptions;
        }

        /**
         * Socket connect timeout. Default is 10 seconds.
         *
         * @return the connect timeout
         */
        public Duration connectTimeout() {
            return connectTimeout;
        }

        /**
         * Socket read timeout. Default is 30 seconds.
         *
         * @return the read timeout
         */
        public Duration readTimeout() {
            return readTimeout;
        }

        /**
         * Socket receive buffer size.
         *
         * @return the socket receive buffer size
         */
        public int socketReceiveBufferSize() {
            return socketReceiveBufferSize;
        }

        /**
         * Socket send buffer size.
         *
         * @return the socket send buffer size
         */
        public int socketSendBufferSize() {
            return socketSendBufferSize;
        }

        /**
         * Socket reuse address.
         * Default is {@code true}.
         *
         * @return the socket reuse address
         */
        public boolean socketReuseAddress() {
            return socketReuseAddress;
        }

        /**
         * Configure socket keep alive.
         * Default is {@code true}.
         *
         * @return the socket keep alive
         */
        public boolean socketKeepAlive() {
            return socketKeepAlive;
        }

        /**
         * This option may improve performance on some systems.
         * Default is {@code false}.
         *
         * @return the tcp no delay
         */
        public boolean tcpNoDelay() {
            return tcpNoDelay;
        }

        /**
         * If this instance was configured, this would be the config instance used.
         *
         * @return config node used to configure this builder, or empty if not configured
         */
        public Optional<Config> config() {
            return Optional.ofNullable(config);
        }

        @Override
        public String toString() {
            return "SocketOptionsBuilder{"
                    + "socketOptions=" + socketOptions + ","
                    + "connectTimeout=" + connectTimeout + ","
                    + "readTimeout=" + readTimeout + ","
                    + "socketReceiveBufferSize=" + socketReceiveBufferSize + ","
                    + "socketSendBufferSize=" + socketSendBufferSize + ","
                    + "socketReuseAddress=" + socketReuseAddress + ","
                    + "socketKeepAlive=" + socketKeepAlive + ","
                    + "tcpNoDelay=" + tcpNoDelay
                    + "}";
        }

        /**
         * Handles providers and decorators.
         */
        protected void preBuildPrototype() {
            new SocketOptionsBlueprint.BuilderDecorator().decorate(this);
        }

        /**
         * Validates required properties.
         */
        protected void validatePrototype() {
        }

        /**
         * Generated implementation of the prototype, can be extended by descendant prototype implementations.
         */
        protected static class SocketOptionsImpl implements SocketOptions {

            private final boolean socketKeepAlive;
            private final boolean socketReuseAddress;
            private final boolean tcpNoDelay;
            private final Duration connectTimeout;
            private final Duration readTimeout;
            private final int socketReceiveBufferSize;
            private final int socketSendBufferSize;
            private final Map<SocketOption<?>, Object> socketOptions;

            /**
             * Create an instance providing a builder.
             *
             * @param builder extending builder base of this prototype
             */
            protected SocketOptionsImpl(SocketOptions.BuilderBase<?, ?> builder) {
                this.socketOptions = Collections.unmodifiableMap(new LinkedHashMap<>(builder.socketOptions()));
                this.connectTimeout = builder.connectTimeout();
                this.readTimeout = builder.readTimeout();
                this.socketReceiveBufferSize = builder.socketReceiveBufferSize();
                this.socketSendBufferSize = builder.socketSendBufferSize();
                this.socketReuseAddress = builder.socketReuseAddress();
                this.socketKeepAlive = builder.socketKeepAlive();
                this.tcpNoDelay = builder.tcpNoDelay();
            }

            @Override
            public Map<SocketOption<?>, Object> socketOptions() {
                return socketOptions;
            }

            @Override
            public Duration connectTimeout() {
                return connectTimeout;
            }

            @Override
            public Duration readTimeout() {
                return readTimeout;
            }

            @Override
            public int socketReceiveBufferSize() {
                return socketReceiveBufferSize;
            }

            @Override
            public int socketSendBufferSize() {
                return socketSendBufferSize;
            }

            @Override
            public boolean socketReuseAddress() {
                return socketReuseAddress;
            }

            @Override
            public boolean socketKeepAlive() {
                return socketKeepAlive;
            }

            @Override
            public boolean tcpNoDelay() {
                return tcpNoDelay;
            }

            @Override
            public String toString() {
                return "SocketOptions{"
                        + "socketOptions=" + socketOptions + ","
                        + "connectTimeout=" + connectTimeout + ","
                        + "readTimeout=" + readTimeout + ","
                        + "socketReceiveBufferSize=" + socketReceiveBufferSize + ","
                        + "socketSendBufferSize=" + socketSendBufferSize + ","
                        + "socketReuseAddress=" + socketReuseAddress + ","
                        + "socketKeepAlive=" + socketKeepAlive + ","
                        + "tcpNoDelay=" + tcpNoDelay
                        + "}";
            }

            @Override
            public boolean equals(Object o) {
                if (o == this) {
                    return true;
                }
                if (!(o instanceof SocketOptions other)) {
                    return false;
                }
                return Objects.equals(socketOptions, other.socketOptions())
                        && Objects.equals(connectTimeout, other.connectTimeout())
                        && Objects.equals(readTimeout, other.readTimeout())
                        && socketReceiveBufferSize == other.socketReceiveBufferSize()
                        && socketSendBufferSize == other.socketSendBufferSize()
                        && socketReuseAddress == other.socketReuseAddress()
                        && socketKeepAlive == other.socketKeepAlive()
                        && tcpNoDelay == other.tcpNoDelay();
            }

            @Override
            public int hashCode() {
                return Objects.hash(socketOptions, connectTimeout, readTimeout, socketReceiveBufferSize, socketSendBufferSize, socketReuseAddress, socketKeepAlive, tcpNoDelay);
            }

        }

    }

    /**
     * Fluent API builder for {@link SocketOptions}.
     */
    class Builder extends SocketOptions.BuilderBase<SocketOptions.Builder, SocketOptions> implements io.helidon.common.Builder<SocketOptions.Builder, SocketOptions> {

        private Builder() {
        }

        @Override
        public SocketOptions buildPrototype() {
            preBuildPrototype();
            validatePrototype();
            return new SocketOptionsImpl(this);
        }

        @Override
        public SocketOptions build() {
            return buildPrototype();
        }

    }

}
