/*
 * Copyright (c) 2024 Oracle and/or its affiliates.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package io.helidon.common.pki;

import java.util.Objects;
import java.util.Optional;
import java.util.function.Consumer;

import io.helidon.builder.api.Prototype;
import io.helidon.common.Generated;
import io.helidon.common.config.Config;
import io.helidon.common.configurable.Resource;
import io.helidon.common.configurable.ResourceConfig;

/**
 * PEM files based keys - accepts private key and certificate chain.
 * <p>
 * If you have "standard" linux/unix private key, you must run "
 * {@code openssl pkcs8 -topk8 -in ./id_rsa -out ./id_rsa.p8}" on it to work with this builder for password protected
 * file; or "{@code openssl pkcs8 -topk8 -in ./id_rsa -out ./id_rsa_nocrypt.p8 -nocrypt}" for unprotected file.
 * <p>
 * The only supported format is PKCS#8. If you have a different format, you must transform it to PKCS8 PEM format (to
 * use this builder), or to PKCS#12 keystore format (and use {@link io.helidon.common.pki.KeystoreKeys.Builder}).
 *
 * @see #builder()
 * @see #create()
 */
@Generated(value = "io.helidon.builder.processor.BlueprintProcessor", trigger = "io.helidon.common.pki.PemKeysBlueprint")
public interface PemKeys extends PemKeysBlueprint, Prototype.Api {

    /**
     * Create a new fluent API builder to customize configuration.
     *
     * @return a new builder
     */
    static PemKeys.Builder builder() {
        return new PemKeys.Builder();
    }

    /**
     * Create a new fluent API builder from an existing instance.
     *
     * @param instance an existing instance used as a base for the builder
     * @return a builder based on an instance
     */
    static PemKeys.Builder builder(PemKeys instance) {
        return PemKeys.builder().from(instance);
    }

    /**
     * Create a new instance from configuration.
     *
     * @param config used to configure the new instance
     * @return a new instance configured from configuration
     */
    static PemKeys create(Config config) {
        return PemKeys.builder().config(config).buildPrototype();
    }

    /**
     * Create a new instance with default values.
     *
     * @return a new instance
     */
    static PemKeys create() {
        return PemKeys.builder().buildPrototype();
    }

    /**
     * Fluent API builder base for {@link PemKeys}.
     *
     * @param <BUILDER> type of the builder extending this abstract builder
     * @param <PROTOTYPE> type of the prototype interface that would be built by {@link #buildPrototype()}
     */
    abstract class BuilderBase<BUILDER extends PemKeys.BuilderBase<BUILDER, PROTOTYPE>, PROTOTYPE extends PemKeys> implements Prototype.ConfiguredBuilder<BUILDER, PROTOTYPE> {

        private char[] keyPassphrase;
        private Config config;
        private Resource certChain;
        private Resource certificates;
        private Resource key;
        private Resource publicKey;

        /**
         * Protected to support extensibility.
         */
        protected BuilderBase() {
        }

        /**
         * Update this builder from an existing prototype instance.
         *
         * @param prototype existing prototype to update this builder from
         * @return updated builder instance
         */
        public BUILDER from(PemKeys prototype) {
            key(prototype.key());
            keyPassphrase(prototype.keyPassphrase());
            publicKey(prototype.publicKey());
            certChain(prototype.certChain());
            certificates(prototype.certificates());
            return self();
        }

        /**
         * Update this builder from an existing prototype builder instance.
         *
         * @param builder existing builder prototype to update this builder from
         * @return updated builder instance
         */
        public BUILDER from(PemKeys.BuilderBase<?, ?> builder) {
            builder.key().ifPresent(this::key);
            builder.keyPassphrase().ifPresent(this::keyPassphrase);
            builder.publicKey().ifPresent(this::publicKey);
            builder.certChain().ifPresent(this::certChain);
            builder.certificates().ifPresent(this::certificates);
            return self();
        }

        /**
         * Update builder from configuration (node of this type).
         * If a value is present in configuration, it would override currently configured values.
         *
         * @param config configuration instance used to obtain values to update this builder
         * @return updated builder instance
         */
        @Override
        public BUILDER config(Config config) {
            Objects.requireNonNull(config);
            this.config = config;
            config.get("key.resource").map(ResourceConfig::create).ifPresent(this::key);
            config.get("key.passphrase").asString().as(String::toCharArray).ifPresent(this::keyPassphrase);
            config.get("public-key.resource").map(ResourceConfig::create).ifPresent(this::publicKey);
            config.get("cert-chain.resource").map(ResourceConfig::create).ifPresent(this::certChain);
            config.get("certificates.resource").map(ResourceConfig::create).ifPresent(this::certificates);
            return self();
        }

        /**
         * Clear existing value of this property.
         *
         * @return updated builder instance
         * @see #key()
         */
        public BUILDER clearKey() {
            this.key = null;
            return self();
        }

        /**
         * Read a private key from PEM format from a resource definition.
         *
         * @param key key resource (file, classpath, URL etc.)
         * @return updated builder instance
         * @see #key()
         */
        public BUILDER key(Resource key) {
            Objects.requireNonNull(key);
            this.key = key;
            return self();
        }

        /**
         * Read a private key from PEM format from a resource definition.
         *
         * @param keyConfig key resource (file, classpath, URL etc.)
         * @return updated builder instance
         * @see #key()
         */
        public BUILDER key(ResourceConfig keyConfig) {
            Objects.requireNonNull(keyConfig);
            this.key = Resource.create(keyConfig);
            return self();
        }

        /**
         * Read a private key from PEM format from a resource definition.
         *
         * @param consumer key resource (file, classpath, URL etc.)
         * @return updated builder instance
         * @see #key()
         */
        public BUILDER key(Consumer<ResourceConfig.Builder> consumer) {
            Objects.requireNonNull(consumer);
            var builder = ResourceConfig.builder();
            consumer.accept(builder);
            this.key(builder.build());
            return self();
        }

        /**
         * Clear existing value of this property.
         *
         * @return updated builder instance
         * @see #keyPassphrase()
         */
        public BUILDER clearKeyPassphrase() {
            this.keyPassphrase = null;
            return self();
        }

        /**
         * Passphrase for private key. If the key is encrypted (and in PEM PKCS#8 format), this passphrase will be used to
         * decrypt it.
         *
         * @param keyPassphrase passphrase used to encrypt the private key
         * @return updated builder instance
         * @see #keyPassphrase()
         */
        public BUILDER keyPassphrase(char[] keyPassphrase) {
            Objects.requireNonNull(keyPassphrase);
            this.keyPassphrase = keyPassphrase;
            return self();
        }

        /**
         * Passphrase for private key. If the key is encrypted (and in PEM PKCS#8 format), this passphrase will be used to
         * decrypt it.
         *
         * @param keyPassphrase passphrase used to encrypt the private key
         * @return updated builder instance
         * @see #keyPassphrase()
         */
        public BUILDER keyPassphrase(String keyPassphrase) {
            Objects.requireNonNull(keyPassphrase);
            this.keyPassphrase = keyPassphrase.toCharArray();
            return self();
        }

        /**
         * Clear existing value of this property.
         *
         * @return updated builder instance
         * @see #publicKey()
         */
        public BUILDER clearPublicKey() {
            this.publicKey = null;
            return self();
        }

        /**
         * Read a public key from PEM format from a resource definition.
         *
         * @param publicKey public key resource (file, classpath, URL etc.)
         * @return updated builder instance
         * @see #publicKey()
         */
        public BUILDER publicKey(Resource publicKey) {
            Objects.requireNonNull(publicKey);
            this.publicKey = publicKey;
            return self();
        }

        /**
         * Read a public key from PEM format from a resource definition.
         *
         * @param publicKeyConfig public key resource (file, classpath, URL etc.)
         * @return updated builder instance
         * @see #publicKey()
         */
        public BUILDER publicKey(ResourceConfig publicKeyConfig) {
            Objects.requireNonNull(publicKeyConfig);
            this.publicKey = Resource.create(publicKeyConfig);
            return self();
        }

        /**
         * Read a public key from PEM format from a resource definition.
         *
         * @param consumer public key resource (file, classpath, URL etc.)
         * @return updated builder instance
         * @see #publicKey()
         */
        public BUILDER publicKey(Consumer<ResourceConfig.Builder> consumer) {
            Objects.requireNonNull(consumer);
            var builder = ResourceConfig.builder();
            consumer.accept(builder);
            this.publicKey(builder.build());
            return self();
        }

        /**
         * Clear existing value of this property.
         *
         * @return updated builder instance
         * @see #certChain()
         */
        public BUILDER clearCertChain() {
            this.certChain = null;
            return self();
        }

        /**
         * Load certificate chain from PEM resource.
         *
         * @param certChain resource (e.g. classpath, file path, URL etc.)
         * @return updated builder instance
         * @see #certChain()
         */
        public BUILDER certChain(Resource certChain) {
            Objects.requireNonNull(certChain);
            this.certChain = certChain;
            return self();
        }

        /**
         * Load certificate chain from PEM resource.
         *
         * @param certChainConfig resource (e.g. classpath, file path, URL etc.)
         * @return updated builder instance
         * @see #certChain()
         */
        public BUILDER certChain(ResourceConfig certChainConfig) {
            Objects.requireNonNull(certChainConfig);
            this.certChain = Resource.create(certChainConfig);
            return self();
        }

        /**
         * Load certificate chain from PEM resource.
         *
         * @param consumer resource (e.g. classpath, file path, URL etc.)
         * @return updated builder instance
         * @see #certChain()
         */
        public BUILDER certChain(Consumer<ResourceConfig.Builder> consumer) {
            Objects.requireNonNull(consumer);
            var builder = ResourceConfig.builder();
            consumer.accept(builder);
            this.certChain(builder.build());
            return self();
        }

        /**
         * Clear existing value of this property.
         *
         * @return updated builder instance
         * @see #certificates()
         */
        public BUILDER clearCertificates() {
            this.certificates = null;
            return self();
        }

        /**
         * Read one or more certificates in PEM format from a resource definition. Used eg: in a trust store.
         *
         * @param certificates key resource (file, classpath, URL etc.)
         * @return updated builder instance
         * @see #certificates()
         */
        public BUILDER certificates(Resource certificates) {
            Objects.requireNonNull(certificates);
            this.certificates = certificates;
            return self();
        }

        /**
         * Read one or more certificates in PEM format from a resource definition. Used eg: in a trust store.
         *
         * @param certificatesConfig key resource (file, classpath, URL etc.)
         * @return updated builder instance
         * @see #certificates()
         */
        public BUILDER certificates(ResourceConfig certificatesConfig) {
            Objects.requireNonNull(certificatesConfig);
            this.certificates = Resource.create(certificatesConfig);
            return self();
        }

        /**
         * Read one or more certificates in PEM format from a resource definition. Used eg: in a trust store.
         *
         * @param consumer key resource (file, classpath, URL etc.)
         * @return updated builder instance
         * @see #certificates()
         */
        public BUILDER certificates(Consumer<ResourceConfig.Builder> consumer) {
            Objects.requireNonNull(consumer);
            var builder = ResourceConfig.builder();
            consumer.accept(builder);
            this.certificates(builder.build());
            return self();
        }

        /**
         * Read a private key from PEM format from a resource definition.
         *
         * @return the key
         */
        public Optional<Resource> key() {
            return Optional.ofNullable(key);
        }

        /**
         * Passphrase for private key. If the key is encrypted (and in PEM PKCS#8 format), this passphrase will be used to
         * decrypt it.
         *
         * @return the key passphrase
         */
        public Optional<char[]> keyPassphrase() {
            return Optional.ofNullable(keyPassphrase);
        }

        /**
         * Read a public key from PEM format from a resource definition.
         *
         * @return the public key
         */
        public Optional<Resource> publicKey() {
            return Optional.ofNullable(publicKey);
        }

        /**
         * Load certificate chain from PEM resource.
         *
         * @return the cert chain
         */
        public Optional<Resource> certChain() {
            return Optional.ofNullable(certChain);
        }

        /**
         * Read one or more certificates in PEM format from a resource definition. Used eg: in a trust store.
         *
         * @return the certificates
         */
        public Optional<Resource> certificates() {
            return Optional.ofNullable(certificates);
        }

        /**
         * If this instance was configured, this would be the config instance used.
         *
         * @return config node used to configure this builder, or empty if not configured
         */
        public Optional<Config> config() {
            return Optional.ofNullable(config);
        }

        @Override
        public String toString() {
            return "PemKeysBuilder{"
                    + "key=" + key + ","
                    + "keyPassphrase=" + (keyPassphrase == null ? "null" : "****") + ","
                    + "publicKey=" + publicKey + ","
                    + "certChain=" + certChain + ","
                    + "certificates=" + certificates
                    + "}";
        }

        /**
         * Handles providers and decorators.
         */
        protected void preBuildPrototype() {
        }

        /**
         * Validates required properties.
         */
        protected void validatePrototype() {
        }

        /**
         * Read a private key from PEM format from a resource definition.
         *
         * @param key key resource (file, classpath, URL etc.)
         * @return updated builder instance
         * @see #key()
         */
        BUILDER key(Optional<? extends Resource> key) {
            Objects.requireNonNull(key);
            this.key = key.map(io.helidon.common.configurable.Resource.class::cast).orElse(this.key);
            return self();
        }

        /**
         * Passphrase for private key. If the key is encrypted (and in PEM PKCS#8 format), this passphrase will be used to
         * decrypt it.
         *
         * @param keyPassphrase passphrase used to encrypt the private key
         * @return updated builder instance
         * @see #keyPassphrase()
         */
        BUILDER keyPassphrase(Optional<char[]> keyPassphrase) {
            Objects.requireNonNull(keyPassphrase);
            this.keyPassphrase = keyPassphrase.map(char[].class::cast).orElse(this.keyPassphrase);
            return self();
        }

        /**
         * Read a public key from PEM format from a resource definition.
         *
         * @param publicKey public key resource (file, classpath, URL etc.)
         * @return updated builder instance
         * @see #publicKey()
         */
        BUILDER publicKey(Optional<? extends Resource> publicKey) {
            Objects.requireNonNull(publicKey);
            this.publicKey = publicKey.map(io.helidon.common.configurable.Resource.class::cast).orElse(this.publicKey);
            return self();
        }

        /**
         * Load certificate chain from PEM resource.
         *
         * @param certChain resource (e.g. classpath, file path, URL etc.)
         * @return updated builder instance
         * @see #certChain()
         */
        BUILDER certChain(Optional<? extends Resource> certChain) {
            Objects.requireNonNull(certChain);
            this.certChain = certChain.map(io.helidon.common.configurable.Resource.class::cast).orElse(this.certChain);
            return self();
        }

        /**
         * Read one or more certificates in PEM format from a resource definition. Used eg: in a trust store.
         *
         * @param certificates key resource (file, classpath, URL etc.)
         * @return updated builder instance
         * @see #certificates()
         */
        BUILDER certificates(Optional<? extends Resource> certificates) {
            Objects.requireNonNull(certificates);
            this.certificates = certificates.map(io.helidon.common.configurable.Resource.class::cast).orElse(this.certificates);
            return self();
        }

        /**
         * Generated implementation of the prototype, can be extended by descendant prototype implementations.
         */
        protected static class PemKeysImpl implements PemKeys {

            private final Optional<char[]> keyPassphrase;
            private final Optional<Resource> certChain;
            private final Optional<Resource> certificates;
            private final Optional<Resource> key;
            private final Optional<Resource> publicKey;

            /**
             * Create an instance providing a builder.
             *
             * @param builder extending builder base of this prototype
             */
            protected PemKeysImpl(PemKeys.BuilderBase<?, ?> builder) {
                this.key = builder.key();
                this.keyPassphrase = builder.keyPassphrase();
                this.publicKey = builder.publicKey();
                this.certChain = builder.certChain();
                this.certificates = builder.certificates();
            }

            @Override
            public Optional<Resource> key() {
                return key;
            }

            @Override
            public Optional<char[]> keyPassphrase() {
                return keyPassphrase;
            }

            @Override
            public Optional<Resource> publicKey() {
                return publicKey;
            }

            @Override
            public Optional<Resource> certChain() {
                return certChain;
            }

            @Override
            public Optional<Resource> certificates() {
                return certificates;
            }

            @Override
            public String toString() {
                return "PemKeys{"
                        + "key=" + key + ","
                        + "keyPassphrase=" + (keyPassphrase.isPresent() ? "****" : "null") + ","
                        + "publicKey=" + publicKey + ","
                        + "certChain=" + certChain + ","
                        + "certificates=" + certificates
                        + "}";
            }

            @Override
            public boolean equals(Object o) {
                if (o == this) {
                    return true;
                }
                if (!(o instanceof PemKeys other)) {
                    return false;
                }
                return Objects.equals(key, other.key())
                        && Objects.equals(keyPassphrase, other.keyPassphrase())
                        && Objects.equals(publicKey, other.publicKey())
                        && Objects.equals(certChain, other.certChain())
                        && Objects.equals(certificates, other.certificates());
            }

            @Override
            public int hashCode() {
                return Objects.hash(key, keyPassphrase, publicKey, certChain, certificates);
            }

        }

    }

    /**
     * Fluent API builder for {@link PemKeys}.
     */
    class Builder extends PemKeys.BuilderBase<PemKeys.Builder, PemKeys> implements io.helidon.common.Builder<PemKeys.Builder, PemKeys> {

        private Builder() {
        }

        @Override
        public PemKeys buildPrototype() {
            preBuildPrototype();
            validatePrototype();
            return new PemKeysImpl(this);
        }

        @Override
        public PemKeys build() {
            return buildPrototype();
        }

    }

}
