/*
 * Copyright (c) 2023 Oracle and/or its affiliates.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package io.helidon.common.pki;

import java.util.ArrayList;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.Consumer;
import java.util.function.Supplier;

import io.helidon.builder.api.Prototype;
import io.helidon.common.Errors;
import io.helidon.common.Generated;
import io.helidon.common.config.Config;
import io.helidon.common.configurable.Resource;
import io.helidon.common.configurable.ResourceConfig;

/**
 * Resources from a java keystore (PKCS12, JKS etc.).
 *
 * @see #builder()
 */
@Generated(value = "io.helidon.builder.processor.BlueprintProcessor", trigger = "io.helidon.common.pki.KeystoreKeysBlueprint")
public interface KeystoreKeys extends KeystoreKeysBlueprint, Prototype.Api {

    /**
     * Create a new fluent API builder to customize configuration.
     *
     * @return a new builder
     */
    static KeystoreKeys.Builder builder() {
        return new KeystoreKeys.Builder();
    }

    /**
     * Create a new fluent API builder from an existing instance.
     *
     * @param instance an existing instance used as a base for the builder
     * @return a builder based on an instance
     */
    static KeystoreKeys.Builder builder(KeystoreKeys instance) {
        return KeystoreKeys.builder().from(instance);
    }

    /**
     * Create a new instance from configuration.
     *
     * @param config used to configure the new instance
     * @return a new instance configured from configuration
     */
    static KeystoreKeys create(Config config) {
        return KeystoreKeys.builder().config(config).buildPrototype();
    }

    /**
     * Fluent API builder base for {@link KeystoreKeys}.
     *
     * @param <BUILDER> type of the builder extending this abstract builder
     * @param <PROTOTYPE> type of the prototype interface that would be built by {@link #buildPrototype()}
     */
    abstract class BuilderBase<BUILDER extends KeystoreKeys.BuilderBase<BUILDER, PROTOTYPE>, PROTOTYPE extends KeystoreKeys> implements Prototype.ConfiguredBuilder<BUILDER, PROTOTYPE> {

        private final List<String> certAliases = new ArrayList<>();
        private boolean trustStore = false;
        private char[] keyPassphrase;
        private char[] passphrase;
        private Config config;
        private Resource keystore;
        private String certAlias;
        private String certChainAlias;
        private String keyAlias;
        private String type = "PKCS12";

        /**
         * Protected to support extensibility.
         */
        protected BuilderBase() {
        }

        /**
         * Update this builder from an existing prototype instance.
         *
         * @param prototype existing prototype to update this builder from
         * @return updated builder instance
         */
        public BUILDER from(KeystoreKeys prototype) {
            keystore(prototype.keystore());
            type(prototype.type());
            passphrase(prototype.passphrase());
            keyAlias(prototype.keyAlias());
            keyPassphrase(prototype.keyPassphrase());
            certAlias(prototype.certAlias());
            certChainAlias(prototype.certChainAlias());
            addCertAliases(prototype.certAliases());
            trustStore(prototype.trustStore());
            return self();
        }

        /**
         * Update this builder from an existing prototype builder instance.
         *
         * @param builder existing builder prototype to update this builder from
         * @return updated builder instance
         */
        public BUILDER from(KeystoreKeys.BuilderBase<?, ?> builder) {
            builder.keystore().ifPresent(this::keystore);
            type(builder.type());
            builder.passphrase().ifPresent(this::passphrase);
            builder.keyAlias().ifPresent(this::keyAlias);
            builder.keyPassphrase().ifPresent(this::keyPassphrase);
            builder.certAlias().ifPresent(this::certAlias);
            builder.certChainAlias().ifPresent(this::certChainAlias);
            addCertAliases(builder.certAliases());
            trustStore(builder.trustStore());
            return self();
        }

        /**
         * Keystore passphrase.
         *
         * @param passphrase new keystore passphrase
         * @return updated builder instance
         * @deprecated use {@link #passphrase(String)} instead
         */
        public BUILDER keystorePassphrase(String passphrase) {
            KeystoreKeysBlueprint.CustomMethods.keystorePassphrase(this, passphrase);
            return self();
        }

        /**
         * Update builder from configuration (node of this type).
         * If a value is present in configuration, it would override currently configured values.
         *
         * @param config configuration instance used to obtain values to update this builder
         * @return updated builder instance
         */
        @Override
        public BUILDER config(Config config) {
            Objects.requireNonNull(config);
            this.config = config;
            config.get("resource").map(ResourceConfig::create).ifPresent(this::keystore);
            config.get("type").as(String.class).ifPresent(this::type);
            config.get("passphrase").asString().as(String::toCharArray).ifPresent(this::passphrase);
            config.get("key.alias").as(String.class).ifPresent(this::keyAlias);
            config.get("key.passphrase").asString().as(String::toCharArray).ifPresent(this::keyPassphrase);
            config.get("cert.alias").as(String.class).ifPresent(this::certAlias);
            config.get("cert-chain.alias").as(String.class).ifPresent(this::certChainAlias);
            config.get("trust-store").as(Boolean.class).ifPresent(this::trustStore);
            return self();
        }

        /**
         * Keystore resource definition.
         *
         * @param keystore keystore resource, from file path, classpath, URL etc.
         * @return updated builder instance
         * @see #keystore()
         */
        public BUILDER keystore(Resource keystore) {
            Objects.requireNonNull(keystore);
            this.keystore = keystore;
            return self();
        }

        /**
         * Keystore resource definition.
         *
         * @param keystoreConfig keystore resource, from file path, classpath, URL etc.
         * @return updated builder instance
         * @see #keystore()
         */
        public BUILDER keystore(ResourceConfig keystoreConfig) {
            Objects.requireNonNull(keystoreConfig);
            this.keystore = Resource.create(keystoreConfig);
            return self();
        }

        /**
         * Keystore resource definition.
         *
         * @param consumer consumer of builder for
         *                 keystore resource, from file path, classpath, URL etc.
         * @return updated builder instance
         * @see #keystore()
         */
        public BUILDER keystore(Consumer<ResourceConfig.Builder> consumer) {
            Objects.requireNonNull(consumer);
            var builder = ResourceConfig.builder();
            consumer.accept(builder);
            this.keystore(builder.build());
            return self();
        }

        /**
         * Keystore resource definition.
         *
         * @param supplier supplier of
         *                 keystore resource, from file path, classpath, URL etc.
         * @return updated builder instance
         * @see #keystore()
         */
        public BUILDER keystore(Supplier<? extends Resource> supplier) {
            Objects.requireNonNull(supplier);
            this.keystore(supplier.get());
            return self();
        }

        /**
         * Set type of keystore.
         * Defaults to {@value #DEFAULT_KEYSTORE_TYPE},
         * expected are other keystore types supported by java then can store keys under aliases.
         *
         * @param type keystore type to load the key
         * @return updated builder instance
         * @see #type()
         */
        public BUILDER type(String type) {
            Objects.requireNonNull(type);
            this.type = type;
            return self();
        }

        /**
         * Clear existing value of this property.
         *
         * @return updated builder instance
         * @see #passphrase()
         */
        public BUILDER clearPassphrase() {
            this.passphrase = null;
            return self();
        }

        /**
         * Pass-phrase of the keystore (supported with JKS and PKCS12 keystores).
         *
         * @param passphrase keystore password to use
         * @return updated builder instance
         * @see #passphrase()
         */
        public BUILDER passphrase(char[] passphrase) {
            Objects.requireNonNull(passphrase);
            this.passphrase = passphrase;
            return self();
        }

        /**
         * Pass-phrase of the keystore (supported with JKS and PKCS12 keystores).
         *
         * @param passphrase keystore password to use
         * @return updated builder instance
         * @see #passphrase()
         */
        public BUILDER passphrase(String passphrase) {
            Objects.requireNonNull(passphrase);
            this.passphrase = passphrase.toCharArray();
            return self();
        }

        /**
         * Clear existing value of this property.
         *
         * @return updated builder instance
         * @see #keyAlias()
         */
        public BUILDER clearKeyAlias() {
            this.keyAlias = null;
            return self();
        }

        /**
         * Alias of the private key in the keystore.
         *
         * @param keyAlias alias of the key in the keystore
         * @return updated builder instance
         * @see #keyAlias()
         */
        public BUILDER keyAlias(String keyAlias) {
            Objects.requireNonNull(keyAlias);
            this.keyAlias = keyAlias;
            return self();
        }

        /**
         * Clear existing value of this property.
         *
         * @return updated builder instance
         * @see #keyPassphrase()
         */
        public BUILDER clearKeyPassphrase() {
            this.keyPassphrase = null;
            return self();
        }

        /**
         * Pass-phrase of the key in the keystore (used for private keys).
         * This is (by default) the same as keystore passphrase - only configure
         * if it differs from keystore passphrase.
         *
         * @param keyPassphrase pass-phrase of the key
         * @return updated builder instance
         * @see #keyPassphrase()
         */
        public BUILDER keyPassphrase(char[] keyPassphrase) {
            Objects.requireNonNull(keyPassphrase);
            this.keyPassphrase = keyPassphrase;
            return self();
        }

        /**
         * Pass-phrase of the key in the keystore (used for private keys).
         * This is (by default) the same as keystore passphrase - only configure
         * if it differs from keystore passphrase.
         *
         * @param keyPassphrase pass-phrase of the key
         * @return updated builder instance
         * @see #keyPassphrase()
         */
        public BUILDER keyPassphrase(String keyPassphrase) {
            Objects.requireNonNull(keyPassphrase);
            this.keyPassphrase = keyPassphrase.toCharArray();
            return self();
        }

        /**
         * Clear existing value of this property.
         *
         * @return updated builder instance
         * @see #certAlias()
         */
        public BUILDER clearCertAlias() {
            this.certAlias = null;
            return self();
        }

        /**
         * Alias of X.509 certificate of public key.
         * Used to load both the certificate and public key.
         *
         * @param certAlias alias under which the certificate is stored in the keystore
         * @return updated builder instance
         * @see #certAlias()
         */
        public BUILDER certAlias(String certAlias) {
            Objects.requireNonNull(certAlias);
            this.certAlias = certAlias;
            return self();
        }

        /**
         * Clear existing value of this property.
         *
         * @return updated builder instance
         * @see #certChainAlias()
         */
        public BUILDER clearCertChainAlias() {
            this.certChainAlias = null;
            return self();
        }

        /**
         * Alias of an X.509 chain.
         *
         * @param certChainAlias alias of certificate chain in the keystore
         * @return updated builder instance
         * @see #certChainAlias()
         */
        public BUILDER certChainAlias(String certChainAlias) {
            Objects.requireNonNull(certChainAlias);
            this.certChainAlias = certChainAlias;
            return self();
        }

        /**
         * List of aliases used to generate a trusted set of certificates.
         *
         * @param certAliases aliases of certificates
         * @return updated builder instance
         * @see #certAliases()
         */
        public BUILDER certAliases(List<? extends String> certAliases) {
            Objects.requireNonNull(certAliases);
            this.certAliases.clear();
            this.certAliases.addAll(certAliases);
            return self();
        }

        /**
         * List of aliases used to generate a trusted set of certificates.
         *
         * @param certAliases aliases of certificates
         * @return updated builder instance
         * @see #certAliases()
         */
        public BUILDER addCertAliases(List<? extends String> certAliases) {
            Objects.requireNonNull(certAliases);
            this.certAliases.addAll(certAliases);
            return self();
        }

        /**
         * List of aliases used to generate a trusted set of certificates.
         *
         * @param certAlias aliases of certificates
         * @return updated builder instance
         * @see #certAliases()
         */
        public BUILDER addCertAlias(String certAlias) {
            Objects.requireNonNull(certAlias);
            this.certAliases.add(certAlias);
            return self();
        }

        /**
         * If you want to build a trust store, call this method to add all
         * certificates present in the keystore to certificate list.
         *
         * @param trustStore whether this is a trust store
         * @return updated builder instance
         * @see #trustStore()
         */
        public BUILDER trustStore(boolean trustStore) {
            this.trustStore = trustStore;
            return self();
        }

        /**
         * Keystore resource definition.
         *
         * @return the keystore
         */
        public Optional<Resource> keystore() {
            return Optional.ofNullable(keystore);
        }

        /**
         * Set type of keystore.
         * Defaults to {@value #DEFAULT_KEYSTORE_TYPE},
         * expected are other keystore types supported by java then can store keys under aliases.
         *
         * @return the type
         */
        public String type() {
            return type;
        }

        /**
         * Pass-phrase of the keystore (supported with JKS and PKCS12 keystores).
         *
         * @return the passphrase
         */
        public Optional<char[]> passphrase() {
            return Optional.ofNullable(passphrase);
        }

        /**
         * Alias of the private key in the keystore.
         *
         * @return the key alias
         */
        public Optional<String> keyAlias() {
            return Optional.ofNullable(keyAlias);
        }

        /**
         * Pass-phrase of the key in the keystore (used for private keys).
         * This is (by default) the same as keystore passphrase - only configure
         * if it differs from keystore passphrase.
         *
         * @return the key passphrase
         */
        public Optional<char[]> keyPassphrase() {
            return Optional.ofNullable(keyPassphrase);
        }

        /**
         * Alias of X.509 certificate of public key.
         * Used to load both the certificate and public key.
         *
         * @return the cert alias
         */
        public Optional<String> certAlias() {
            return Optional.ofNullable(certAlias);
        }

        /**
         * Alias of an X.509 chain.
         *
         * @return the cert chain alias
         */
        public Optional<String> certChainAlias() {
            return Optional.ofNullable(certChainAlias);
        }

        /**
         * List of aliases used to generate a trusted set of certificates.
         *
         * @return the cert aliases
         */
        public List<String> certAliases() {
            return certAliases;
        }

        /**
         * If you want to build a trust store, call this method to add all
         * certificates present in the keystore to certificate list.
         *
         * @return the trust store
         */
        public boolean trustStore() {
            return trustStore;
        }

        /**
         * If this instance was configured, this would be the config instance used.
         *
         * @return config node used to configure this builder, or empty if not configured
         */
        public Optional<Config> config() {
            return Optional.ofNullable(config);
        }

        @Override
        public String toString() {
            return "KeystoreKeysBuilder{"
                    + "keystore=" + keystore + ","
                    + "type=" + type + ","
                    + "passphrase=" + (passphrase == null ? "null" : "****") + ","
                    + "keyAlias=" + keyAlias + ","
                    + "keyPassphrase=" + (keyPassphrase == null ? "null" : "****") + ","
                    + "certAlias=" + certAlias + ","
                    + "certChainAlias=" + certChainAlias + ","
                    + "certAliases=" + certAliases + ","
                    + "trustStore=" + trustStore
                    + "}";
        }

        /**
         * Handles providers and decorators.
         */
        protected void preBuildPrototype() {
        }

        /**
         * Validates required properties.
         */
        protected void validatePrototype() {
            Errors.Collector collector = Errors.collector();
            if (keystore == null) {
                collector.fatal(getClass(), "Property \"resource\" is required, but not set");
            }
            collector.collect().checkValid();
        }

        /**
         * Pass-phrase of the keystore (supported with JKS and PKCS12 keystores).
         *
         * @param passphrase keystore password to use
         * @return updated builder instance
         * @see #passphrase()
         */
        BUILDER passphrase(Optional<char[]> passphrase) {
            Objects.requireNonNull(passphrase);
            this.passphrase = passphrase.map(char[].class::cast).orElse(this.passphrase);
            return self();
        }

        /**
         * Alias of the private key in the keystore.
         *
         * @param keyAlias alias of the key in the keystore
         * @return updated builder instance
         * @see #keyAlias()
         */
        BUILDER keyAlias(Optional<String> keyAlias) {
            Objects.requireNonNull(keyAlias);
            this.keyAlias = keyAlias.map(java.lang.String.class::cast).orElse(this.keyAlias);
            return self();
        }

        /**
         * Pass-phrase of the key in the keystore (used for private keys).
         * This is (by default) the same as keystore passphrase - only configure
         * if it differs from keystore passphrase.
         *
         * @param keyPassphrase pass-phrase of the key
         * @return updated builder instance
         * @see #keyPassphrase()
         */
        BUILDER keyPassphrase(Optional<char[]> keyPassphrase) {
            Objects.requireNonNull(keyPassphrase);
            this.keyPassphrase = keyPassphrase.map(char[].class::cast).orElse(this.keyPassphrase);
            return self();
        }

        /**
         * Alias of X.509 certificate of public key.
         * Used to load both the certificate and public key.
         *
         * @param certAlias alias under which the certificate is stored in the keystore
         * @return updated builder instance
         * @see #certAlias()
         */
        BUILDER certAlias(Optional<String> certAlias) {
            Objects.requireNonNull(certAlias);
            this.certAlias = certAlias.map(java.lang.String.class::cast).orElse(this.certAlias);
            return self();
        }

        /**
         * Alias of an X.509 chain.
         *
         * @param certChainAlias alias of certificate chain in the keystore
         * @return updated builder instance
         * @see #certChainAlias()
         */
        BUILDER certChainAlias(Optional<String> certChainAlias) {
            Objects.requireNonNull(certChainAlias);
            this.certChainAlias = certChainAlias.map(java.lang.String.class::cast).orElse(this.certChainAlias);
            return self();
        }

        /**
         * Generated implementation of the prototype, can be extended by descendant prototype implementations.
         */
        protected static class KeystoreKeysImpl implements KeystoreKeys {

            private final boolean trustStore;
            private final List<String> certAliases;
            private final Optional<char[]> keyPassphrase;
            private final Optional<char[]> passphrase;
            private final Optional<String> certAlias;
            private final Optional<String> certChainAlias;
            private final Optional<String> keyAlias;
            private final Resource keystore;
            private final String type;

            /**
             * Create an instance providing a builder.
             *
             * @param builder extending builder base of this prototype
             */
            protected KeystoreKeysImpl(KeystoreKeys.BuilderBase<?, ?> builder) {
                this.keystore = builder.keystore().get();
                this.type = builder.type();
                this.passphrase = builder.passphrase();
                this.keyAlias = builder.keyAlias();
                this.keyPassphrase = builder.keyPassphrase();
                this.certAlias = builder.certAlias();
                this.certChainAlias = builder.certChainAlias();
                this.certAliases = List.copyOf(builder.certAliases());
                this.trustStore = builder.trustStore();
            }

            @Override
            public Resource keystore() {
                return keystore;
            }

            @Override
            public String type() {
                return type;
            }

            @Override
            public Optional<char[]> passphrase() {
                return passphrase;
            }

            @Override
            public Optional<String> keyAlias() {
                return keyAlias;
            }

            @Override
            public Optional<char[]> keyPassphrase() {
                return keyPassphrase;
            }

            @Override
            public Optional<String> certAlias() {
                return certAlias;
            }

            @Override
            public Optional<String> certChainAlias() {
                return certChainAlias;
            }

            @Override
            public List<String> certAliases() {
                return certAliases;
            }

            @Override
            public boolean trustStore() {
                return trustStore;
            }

            @Override
            public String toString() {
                return "KeystoreKeys{"
                        + "keystore=" + keystore + ","
                        + "type=" + type + ","
                        + "passphrase=" + (passphrase.isPresent() ? "****" : "null") + ","
                        + "keyAlias=" + keyAlias + ","
                        + "keyPassphrase=" + (keyPassphrase.isPresent() ? "****" : "null") + ","
                        + "certAlias=" + certAlias + ","
                        + "certChainAlias=" + certChainAlias + ","
                        + "certAliases=" + certAliases + ","
                        + "trustStore=" + trustStore
                        + "}";
            }

            @Override
            public boolean equals(Object o) {
                if (o == this) {
                    return true;
                }
                if (!(o instanceof KeystoreKeys other)) {
                    return false;
                }
                return Objects.equals(keystore, other.keystore())
                        && Objects.equals(type, other.type())
                        && Objects.equals(passphrase, other.passphrase())
                        && Objects.equals(keyAlias, other.keyAlias())
                        && Objects.equals(keyPassphrase, other.keyPassphrase())
                        && Objects.equals(certAlias, other.certAlias())
                        && Objects.equals(certChainAlias, other.certChainAlias())
                        && Objects.equals(certAliases, other.certAliases())
                        && trustStore == other.trustStore();
            }

            @Override
            public int hashCode() {
                return Objects.hash(keystore, type, passphrase, keyAlias, keyPassphrase, certAlias, certChainAlias, certAliases, trustStore);
            }

        }

    }

    /**
     * Fluent API builder for {@link KeystoreKeys}.
     */
    class Builder extends KeystoreKeys.BuilderBase<KeystoreKeys.Builder, KeystoreKeys> implements io.helidon.common.Builder<KeystoreKeys.Builder, KeystoreKeys> {

        private Builder() {
        }

        @Override
        public KeystoreKeys buildPrototype() {
            preBuildPrototype();
            validatePrototype();
            return new KeystoreKeysImpl(this);
        }

        @Override
        public KeystoreKeys build() {
            return buildPrototype();
        }

    }

}
