package io.hanko.sdk.webauthn.protocol;

import com.fasterxml.jackson.annotation.JsonCreator;
import com.fasterxml.jackson.annotation.JsonValue;

/**
 * WebAuthn Relying Parties may use AttestationConveyancePreference to specify their preference
 * regarding attestation conveyance during credential generation.
 *
 * @see <a href="https://www.w3.org/TR/webauthn-1/#enumdef-attestationconveyancepreference">
 * Web Authentication Level 1 - 5.4.6. Attestation Conveyance Preference Enumeration</a>
 */
public enum AttestationConveyancePreference {
    /**
     * This value indicates that the Relying Party is not interested in authenticator attestation.
     * For example, in order to potentially avoid having to obtain user consent to relay identifying
     * information to the Relying Party, or to save a roundtrip to an Attestation CA.
     */
    NONE("none"),

    /**
     * This value indicates that the Relying Party prefers an attestation conveyance yielding
     * verifiable attestation statements, but allows the client to decide how to obtain such
     * attestation statements. The client MAY replace the authenticator-generated attestation
     * statements with attestation statements generated by an Anonymization CA, in order to protect
     * the user's privacy, or to assist Relying Parties with attestation verification in a
     * heterogeneous ecosystem.
     */
    INDIRECT("indirect"),

    /**
     * This value indicates that the Relying Party wants to receive the attestation statement as
     * generated by the authenticator.
     */
    DIRECT("direct");

    String conveyancePreference;

    AttestationConveyancePreference(String value) {
        this.conveyancePreference = value;
    }

    /**
     * Construct a AttestationConveyancePreference from a String value.
     *
     * Used for JSON deserialization.
     * @param value the String value
     * @return the AttestationConveyancePreference
     */
    @JsonCreator
    public static AttestationConveyancePreference fromValue(String value) {
        switch (value) {
            case "direct":
                return DIRECT;
            case "indirect":
                return INDIRECT;
            case "none":
                return NONE;
            default:
                return null;
        }
    }

    /**
     * Return the {@link #conveyancePreference} of the constant.
     * @return the {@link #conveyancePreference} value
     */
    @JsonValue
    public String getConveyancePreference() {
        return conveyancePreference;
    }
}
