/*
 * Copyright (c) 2014-2023 Stream.io Inc. All rights reserved.
 *
 * Licensed under the Stream License;
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *    https://github.com/GetStream/stream-video-android/blob/main/LICENSE
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package io.getstream.video.android.compose.ui.components.audio

import android.content.res.Configuration
import androidx.compose.foundation.background
import androidx.compose.foundation.layout.Box
import androidx.compose.foundation.layout.BoxScope
import androidx.compose.foundation.layout.fillMaxSize
import androidx.compose.foundation.layout.fillMaxWidth
import androidx.compose.foundation.layout.padding
import androidx.compose.material.Scaffold
import androidx.compose.runtime.Composable
import androidx.compose.runtime.LaunchedEffect
import androidx.compose.runtime.getValue
import androidx.compose.ui.Modifier
import androidx.compose.ui.platform.LocalContext
import androidx.compose.ui.platform.LocalInspectionMode
import androidx.compose.ui.platform.testTag
import androidx.compose.ui.res.stringResource
import androidx.compose.ui.tooling.preview.Devices
import androidx.compose.ui.tooling.preview.Preview
import androidx.compose.ui.unit.dp
import androidx.lifecycle.compose.collectAsStateWithLifecycle
import io.getstream.video.android.compose.permission.VideoPermissionsState
import io.getstream.video.android.compose.permission.rememberMicrophonePermissionState
import io.getstream.video.android.compose.theme.VideoTheme
import io.getstream.video.android.core.Call
import io.getstream.video.android.core.ParticipantState
import io.getstream.video.android.mock.StreamMockUtils
import io.getstream.video.android.mock.mockCall

/**
 * Represents audio room content based on the call state provided from the [call].

 * @param call The call that contains all the participants state and tracks.
 * @param modifier Modifier for styling.
 * @param permissions Android permissions that should be required to render a audio call properly.
 * @param title A title that will be shown on the app bar.
 * @param appBarContent Content shown that a call information or an additional actions.
 * @param controlsContent Content is shown that allows users to trigger different actions to control a joined call.
 * @param audioRenderer A single audio renderer renders each individual participant.
 * @param onLeaveRoom A lambda that will be invoked when the leave quietly button was clicked.
 * @param audioContent Content is shown by rendering audio when we're connected to a call successfully.
 */
@Composable
public fun AudioRoom(
    modifier: Modifier = Modifier,
    call: Call,
    isShowingAppBar: Boolean = true,
    permissions: VideoPermissionsState = rememberMicrophonePermissionState(call = call),
    title: String = stringResource(id = io.getstream.video.android.ui.common.R.string.stream_video_audio_room_title),
    appBarContent: @Composable (call: Call) -> Unit = {
        AudioAppBar(
            modifier = Modifier.fillMaxWidth(),
            title = title,
        )
    },
    style: AudioRendererStyle = RegularAudioRendererStyle(),
    audioRenderer: @Composable (
        participant: ParticipantState,
        style: AudioRendererStyle
    ) -> Unit = { audioParticipant, audioStyle ->
        ParticipantAudio(
            participant = audioParticipant,
            style = audioStyle
        )
    },
    audioContent: @Composable BoxScope.(call: Call) -> Unit = {
        val participants by call.state.participants.collectAsStateWithLifecycle()
        AudioParticipantsGrid(
            modifier = Modifier
                .testTag("audio_content")
                .fillMaxSize(),
            participants = participants,
            style = style,
            audioRenderer = audioRenderer
        )
    },
    onLeaveRoom: (() -> Unit)? = null,
    controlsContent: @Composable (call: Call) -> Unit = {
        AudioControlActions(
            modifier = Modifier
                .testTag("audio_controls_content")
                .fillMaxWidth(),
            call = call,
            onLeaveRoom = onLeaveRoom
        )
    },
) {
    DefaultPermissionHandler(videoPermission = permissions)

    Scaffold(
        modifier = modifier
            .background(VideoTheme.colors.appBackground)
            .padding(32.dp),
        contentColor = VideoTheme.colors.appBackground,
        backgroundColor = VideoTheme.colors.appBackground,
        topBar = {
            if (isShowingAppBar) {
                appBarContent.invoke(call)
            }
        },
        bottomBar = { controlsContent.invoke(call) },
        content = { paddings ->
            Box(
                modifier = Modifier
                    .background(color = VideoTheme.colors.appBackground)
                    .padding(paddings)
            ) {
                audioContent.invoke(this, call)
            }
        }
    )
}

@Composable
private fun DefaultPermissionHandler(
    videoPermission: VideoPermissionsState,
) {
    if (LocalInspectionMode.current) return

    LaunchedEffect(key1 = videoPermission) {
        videoPermission.launchPermissionRequest()
    }
}

@Preview
@Preview(uiMode = Configuration.UI_MODE_NIGHT_YES)
@Preview(device = Devices.AUTOMOTIVE_1024p, widthDp = 1440, heightDp = 720)
@Composable
private fun AudioRoomPreview() {
    StreamMockUtils.initializeStreamVideo(LocalContext.current)
    VideoTheme {
        AudioRoom(
            modifier = Modifier.fillMaxSize(),
            call = mockCall
        )
    }
}
