package io.gamedock.sdk.utils.session;

import android.content.Context;
import android.content.SharedPreferences;
import android.os.Handler;
import android.os.Looper;

import io.gamedock.sdk.GamedockSDK;
import io.gamedock.sdk.utils.logging.LoggingUtil;
import io.gamedock.sdk.utils.storage.StorageUtil;

/**
 * Utility class that handles all the logic for the following aspects of the SDK:
 * - Session ID (has a 15 minute change window)
 * - Total time opened
 * - Session start/stop time saving
 * - Session Duration
 */
public class SessionUtil {

    private static String sessionId = "";

    private static Handler handler = getHandler();

    private static Runnable timerRunnable;
    private static Runnable savingRunnable;

    private static long tto;
    private static long sessionDuration;
    private static long increment = 1000;

    private static boolean sessionStarted;

    private static long saveTime = 15000;

    /**
     * Required to call this method to ensure proper start measure point.
     *
     * @param context The activity context.
     */
    public synchronized static void startSession(Context context) {
        tto = GamedockSDK.getInstance(context).getStorageUtil().getLong(StorageUtil.Keys.TotalTimeOpen, 0);

        long compare = 7500000000L;

        if (tto > compare) {
            tto = 0;
        }

        sessionDuration = GamedockSDK.getInstance(context).getStorageUtil().getLong(StorageUtil.Keys.SessionDuration, 0);

        checkSessionDetails(context);

        startTimer(context);
    }

    /**
     * This method is required to ensure proper end measure point.
     *
     * @param context The activity context.
     */
    public static void stopSession(Context context) {
        stopTimer();
        saveSessionDetails(context);
    }

    /**
     * Method that checks the current session details.
     * The session timeout is of 15 minutes.
     *
     * @param context The activity context.
     */
    private synchronized static void checkSessionDetails(Context context) {

        StorageUtil storageUtil = GamedockSDK.getInstance(context).getStorageUtil();

        long now = System.currentTimeMillis();

        long elapsed = now - storageUtil.getLong(StorageUtil.Keys.LastTimeOpen, 0);

        if (elapsed > 900000) {
            LoggingUtil.d("Resetting session id and duration!");
            sessionDuration = 0;
            sessionId = "";

            storageUtil.putLong(StorageUtil.Keys.SessionDuration, sessionDuration);
            storageUtil.putString(StorageUtil.Keys.SessionId, sessionId);
        }

        storageUtil.putLong(StorageUtil.Keys.LastTimeOpen, now);
    }

    /**
     * Method that saves the current session details to the {@link SharedPreferences}.
     *
     * @param context The activity context.
     */
    private synchronized static void saveSessionDetails(final Context context) {
        getHandler().post(new Runnable() {
            @Override
            public void run() {
                GamedockSDK.getInstance(context).getStorageUtil().putLong(StorageUtil.Keys.TotalTimeOpen, tto);
                GamedockSDK.getInstance(context).getStorageUtil().putString(StorageUtil.Keys.SessionId, sessionId);
                GamedockSDK.getInstance(context).getStorageUtil().putLong(StorageUtil.Keys.SessionDuration, sessionDuration);
            }
        });
    }

    /**
     * Method that retrieves the total time the app has been opened.
     *
     * @param context The activity context.
     * @return Returns the total time in a {@link String} format.
     */
    public static String getTotalTimeOpen(Context context) {
        if (sessionStarted) {
            return String.valueOf(tto);
        } else {
            startSession(context);
            return String.valueOf(tto);
        }
    }

    /**
     * Method that returns the current session id.
     *
     * @param context The activity context.
     * @return Returns the current session id.
     */
    public static String getSessionId(Context context) {
        checkSessionDetails(context);

        if (sessionId != null && !sessionId.equals("")) {
            return sessionId;
        } else {
            sessionId = GamedockSDK.getInstance(context).getStorageUtil().getString(StorageUtil.Keys.SessionId, generateSessionId());
            saveSessionDetails(context);
            return sessionId;
        }

    }

    /**
     * Method that returns the current calculated session duration.
     *
     * @return Returns the session duration for the current session.
     */
    public static String getSessionDuration() {
        return String.valueOf(sessionDuration);
    }

    /**
     * Method that generates a new session id.
     *
     * @return Returns the newly generated session id.
     */
    private static String generateSessionId() {
        return java.util.UUID.randomUUID().toString();
    }

    /**
     * Method that starts a time for tracking the current session.
     * The timer ticks every second incrementing both the "sessionDuration" and "totalTimeOpen".
     *
     * @param context The activity context.
     */
    private static void startTimer(final Context context) {
        if (!sessionStarted) {
            try {
                timerRunnable = new Runnable() {
                    @Override
                    public void run() {

                        sessionStarted = true;

                        if (tto <= 0) {
                            tto = 1000;
                        }

                        tto = tto + increment;

                        if (sessionDuration <= 0) {
                            sessionDuration = 1000;
                        }

                        sessionDuration = sessionDuration + increment;

                        if (tto < sessionDuration) {
                            tto = sessionDuration;
                        }

                        handler.postDelayed(this, increment);
                    }
                };

                savingRunnable = new Runnable() {
                    @Override
                    public void run() {
                        saveSessionDetails(context);
                        handler.postDelayed(this, saveTime);
                    }
                };

                handler.postDelayed(timerRunnable, increment);
                handler.postDelayed(savingRunnable, saveTime);

                sessionStarted = true;
            } catch (Exception e) {
                e.printStackTrace();
                sessionStarted = false;
            }
        }
    }

    /**
     * Method that stops the current timer of the session.
     */
    private static void stopTimer() {
        handler.removeCallbacks(timerRunnable);
        handler.removeCallbacks(savingRunnable);
        sessionStarted = false;
    }

    public static boolean isSessionStarted() {
        return sessionStarted;
    }

    private static Handler getHandler() {
        if (handler != null) {
            return handler;

        } else {
            handler = new Handler(Looper.getMainLooper());
            return handler;
        }
    }
}
