package io.gamedock.sdk.utils.performance;

import android.os.Build;

import java.io.BufferedReader;
import java.io.FileNotFoundException;
import java.io.FileReader;
import java.io.IOException;
import java.util.Locale;

import io.gamedock.sdk.utils.logging.LoggingUtil;

/**
 * Class used by the SDK to perform various CPU metrics operations.
 */
public class CpuInfo {

    /**
     * Method used to retrieve the CPU usage.
     *
     * @return The CPU usage in percentage.
     */
    private static BufferedReader totalCpuReader;

    private static BufferedReader appPidCpuReader;

    /* Jiffy is a unit of CPU time. */
    private static long totalJiffies;

    private static long totalJiffiesBefore;

    private static long jiffiesMyPid;

    private static long jiffiesMyPidBefore;

    private static boolean warningShowed = false;

    public static float getCpuUsage() {
        float cpuUsageValue = 0;

        if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.O) {
            if (!warningShowed) {
                LoggingUtil.w("Cpu Usage Reading is not supported on Android O and above");
                warningShowed = true;
            }

            return -1;
        }

        //Create reader for overall CPU usage
        try {
            totalCpuReader = new BufferedReader(new FileReader("/proc/stat"));
        } catch (FileNotFoundException e) {
            LoggingUtil.e("Could not open '/proc/stat' - " + e.getMessage());
        }

        //Create reader for app specific CPU usage
        try {
            appPidCpuReader = new BufferedReader(new FileReader("/proc/" + android.os.Process.myPid() + "/stat"));
        } catch (FileNotFoundException e) {
            LoggingUtil.e("Could not open '/proc/" + android.os.Process.myPid() + "/stat' - " + e.getMessage());
        }

        if (totalCpuReader == null || appPidCpuReader == null) {
            LoggingUtil.e("Could not read CPU information!");
            return -1;
        }

        try {
            String[] cpuData = totalCpuReader.readLine().split("[ ]+", 9);
            long jiffies = Long.parseLong(cpuData[1]) + Long.parseLong(cpuData[2]) + Long.parseLong(cpuData[3]);
            totalJiffies = jiffies + Long.parseLong(cpuData[4]) + Long.parseLong(cpuData[6]) + Long.parseLong(cpuData[7]);
        } catch (NullPointerException | IOException ie) {
            LoggingUtil.e("Failed reading total cpu data - " + ie.getMessage());
            return -1;
        }

        if (appPidCpuReader != null) {
            try {
                String[] cpuData = appPidCpuReader.readLine().split("[ ]+", 18);
                jiffiesMyPid = Long.parseLong(cpuData[13]) + Long.parseLong(cpuData[14]) + Long.parseLong(cpuData[15]) + Long.parseLong(cpuData[16]);
            } catch (NullPointerException | IOException ie) {
                LoggingUtil.e("Failed reading my pid cpu data - " + ie.getMessage());
                return -1;
            }
        }

        if (totalJiffiesBefore > 0) {
            long totalDiff = totalJiffies - totalJiffiesBefore;
            long jiffiesMyPidDiff = jiffiesMyPid - jiffiesMyPidBefore;

            String cpuValueString = String.format(Locale.ENGLISH, "%.02f", (getPercentInRange(100f * jiffiesMyPidDiff / totalDiff)));
            cpuValueString = cpuValueString.replace(",", ".");
            cpuUsageValue = Float.parseFloat(cpuValueString);
        }

        totalJiffiesBefore = totalJiffies;
        jiffiesMyPidBefore = jiffiesMyPid;

        //Close readers
        try {
            if (totalCpuReader != null) {
                totalCpuReader.close();
                totalCpuReader = null;
            }
            if (appPidCpuReader != null) {
                appPidCpuReader.close();
                appPidCpuReader = null;
            }
        } catch (IOException e) {
            e.printStackTrace();
        }

        return cpuUsageValue;

    }

    private static double getPercentInRange(double percent) {
        if (percent > 100f) {
            return 100f;
        } else if (percent < 0f) {
            return 0f;
        }
        return percent;
    }

}