package io.gamedock.sdk.utils.logging;

import android.util.Log;

import java.io.BufferedReader;
import java.io.IOException;
import java.io.InputStreamReader;

/**
 * Utility Class used for all the logging in the SDK.
 */
public class LoggingUtil {

    public static boolean fullLogging;
    public static String TAG = "GamedockSDK";

    /**
     * Method that programatically clear the logcat log.
     */
    public static void clearLog() {
        try {
            Runtime.getRuntime().exec("logcat -c");
        } catch (IOException e) {
            e.printStackTrace();
        }
    }

    /**
     * Method that retrieves the current logcat log specific for the application.
     *
     * @return Returns the log as a {@link String}.
     */
    public static String getLog() {
        try {
            Process process = Runtime.getRuntime().exec("logcat -d");
            BufferedReader bufferedReader = new BufferedReader(new InputStreamReader(process.getInputStream()));

            StringBuilder log = new StringBuilder();
            String line = "";
            while ((line = bufferedReader.readLine()) != null) {
                log.append(line);
            }
            return log.toString();
        } catch (IOException e) {
            e.printStackTrace();
        }
        return null;
    }

    private enum enumLogLevel {
        DEBUG,
        ASSERT,
        ERROR,
        WARN,
        INFO,
        VERBOSE
    }

    public static void d(String logMessage) {
        Log(logMessage, enumLogLevel.DEBUG);
    }

    public static void a(String logMessage) {
        Log(logMessage, enumLogLevel.ASSERT);
    }

    public static void e(String logMessage) {
        Log(logMessage, enumLogLevel.ERROR);
    }

    public static void w(String logMessage) {
        Log(logMessage, enumLogLevel.WARN);
    }

    public static void i(String logMessage) {
        Log(logMessage, enumLogLevel.INFO);
    }

    public static void vi(String logMessage) {
        Log(logMessage, enumLogLevel.VERBOSE);
    }

    /**
     * Method used to log all the messages that the SDK is sending
     *
     * @param logMessage The message that has to be outputted to logcat.
     * @param logLevel   The log level at which logcat should output.
     */
    private static void Log(String logMessage, enumLogLevel logLevel) {
        if (fullLogging || (logLevel != enumLogLevel.INFO && logLevel != enumLogLevel.VERBOSE)) {
            switch (logLevel) {
                case DEBUG:
                    Log.d(TAG, logMessage);
                    break;
                case ASSERT:
                case WARN:
                    Log.w(TAG, logMessage);
                    break;
                case ERROR:
                    Log.e(TAG, logMessage);
                    break;
                case INFO:
                    Log.i(TAG, logMessage);
                    break;
                case VERBOSE:
                    Log.v(TAG, logMessage);
                    break;
            }
        }
    }

    /**
     * Method used to print the current stack trace adding an additional message to the log.
     *
     * @param message The message that has to be appended to the printed stacktrace.
     */
    public static void printStackTrace(String message) {
        StringBuilder messageBuilder = new StringBuilder();
        messageBuilder.append("Printing stack trace, message: ").append(message).append(".\r\nStack trace: ");

        for (StackTraceElement stackTraceElement : Thread.currentThread().getStackTrace()) {
            messageBuilder.append("\r\n").append(stackTraceElement.toString());
        }

        LoggingUtil.d(messageBuilder.toString());
    }
}
