package io.gamedock.sdk.utils.images;


import android.content.Context;
import android.graphics.Bitmap;
import android.os.AsyncTask;
import android.view.View;

import com.nostra13.universalimageloader.core.ImageLoader;
import com.nostra13.universalimageloader.core.assist.FailReason;
import com.nostra13.universalimageloader.core.listener.ImageLoadingListener;

import java.io.File;
import java.util.ArrayList;

import io.gamedock.sdk.GamedockSDK;
import io.gamedock.sdk.gamedata.GamedockGameDataManager;
import io.gamedock.sdk.models.gamedata.GamedockGameData;
import io.gamedock.sdk.models.gamedata.bundles.Bundle;
import io.gamedock.sdk.models.gamedata.currencies.Currency;
import io.gamedock.sdk.models.gamedata.items.Item;
import io.gamedock.sdk.models.gamedata.shop.Entry;
import io.gamedock.sdk.models.gamedata.shop.ImageEntry;
import io.gamedock.sdk.models.gamedata.shop.Tab;
import io.gamedock.sdk.models.image.ImageContext;
import io.gamedock.sdk.utils.error.ErrorCodes;
import io.gamedock.sdk.utils.logging.LoggingUtil;

/**
 * Utility Class that handles all the logic regarding the downloading and loading of images.
 */
public class ImageLoaderUtil {
    private static final Object lock = new Object();

    private static ImageLoaderUtil mInstance = null;
    private Context context;

    private ImageLoader imageLoader;

    private ImageLoaderUtil(Context context) {
        this.context = context;
        this.imageLoader = ImageLoader.getInstance();
    }

    public static ImageLoaderUtil getInstance(Context context) {
        if (mInstance == null) {
            synchronized (lock) {
                if (mInstance == null) {
                    mInstance = new ImageLoaderUtil(context);
                }
            }
        }
        return mInstance;
    }

    /**
     * Method that downloads an image from a specified url and stores it in the local storage.
     *
     * @param url                The URL from which the image should be downloaded.
     * @param id                 The id of the item/bundle associated with the image.
     * @param imageType          The the type of object the image is associated with.
     * @param imageLoadCallbacks The callback for image operations.
     */
    public void requestImage(final String url, final int id, final String imageType, final ImageLoadCallbacks imageLoadCallbacks) {
        if (imageLoader != null) {
            imageLoader.loadImage(url, new ImageLoadingListener() {
                @Override
                public void onLoadingStarted(String s, View view) {

                }

                @Override
                public void onLoadingFailed(String s, View view, FailReason failReason) {

                    LoggingUtil.d("Image failed to download with reason: " + failReason.getType().name());

                    ImageContext imageContext = new ImageContext(id, imageType, url);

                    if (imageLoadCallbacks != null) {
                        imageLoadCallbacks.imageLoadFailed(imageContext, ErrorCodes.ImageLoadFailedError);
                    } else {
                        GamedockSDK.getInstance(context).getImageLoadCallbacks().imageLoadFailed(imageContext, ErrorCodes.ImageLoadFailedError);
                    }

                }

                @Override
                public void onLoadingComplete(String s, View view, Bitmap bitmap) {
                    String localPath = getImageLocalPath(url);
                    ImageContext imageContext = new ImageContext(id, imageType, url);

                    if (imageLoadCallbacks != null) {
                        imageLoadCallbacks.imageLoadSuccess(localPath, imageContext);
                    } else {
                        GamedockSDK.getInstance(context).getImageLoadCallbacks().imageLoadSuccess(localPath, imageContext);
                    }

                }

                @Override
                public void onLoadingCancelled(String s, View view) {

                }
            });
        } else {
            LoggingUtil.e("Error retrieving image!");
        }
    }

    /**
     * Method that returns the local path for an image.
     *
     * @param url The URL associated with the image.
     * @return Returns the local path for the image or {@code null} if no image is present in the local cache.
     */
    public String getImageLocalPath(String url) {
        try {
            if (imageLoader != null) {
                File file = imageLoader.getDiskCache().get(url);
                if (file != null && file.exists()) {
                    String filePath = "file://" + file.getAbsolutePath();
                    LoggingUtil.d("Request url " + url);
                    LoggingUtil.d("Local file path: " + filePath);
                    return filePath;
                } else {
                    return null;
                }
            } else {
                LoggingUtil.e("Error retrieving image!");
                return null;
            }

        } catch (Exception e) {
            return null;
        }
    }

    /**
     * Method used natively which loads a {@link Bitmap} object from a local image path.
     *
     * @param url The URL associated with the image.
     * @return Returns a {@link Bitmap} object containg the image information.
     */
    public Bitmap getBitmapImage(String url) {
        if (imageLoader != null) {
            return imageLoader.loadImageSync(url);
        } else {
            LoggingUtil.e("Error loading bitmap!");
            return null;
        }
    }

    /**
     * Method used for pre-loading all the images.
     */
    public void preloadImages() {
        GamedockGameData gamedockGameData = GamedockGameDataManager.getInstance(context).getGameData();

        ArrayList<String> urlList = new ArrayList<>();
        for (Bundle bundle : gamedockGameData.getBundlesMap().values()) {
            if (bundle.getImageUrl() != null) {
                urlList.add(bundle.getImageUrl());
            }
        }

        for (Item item : gamedockGameData.getItemsMap().values()) {
            if (item.getImageUrl() != null) {
                urlList.add(item.getImageUrl());
            }
        }

        for (Currency currency : gamedockGameData.getCurrenciesMap().values()) {
            if (currency.getImageUrl() != null) {
                urlList.add(currency.getImageUrl());
            }
        }

        for (Tab tab : gamedockGameData.getShop()) {
            for (ImageEntry imageEntry : tab.getImageEntries()) {
                if (imageEntry.getImageUrl() != null) {
                    urlList.add(imageEntry.getImageUrl());
                }
            }

            for (Entry entry : tab.getEntries()) {
                for (ImageEntry imageEntry : entry.getImageEntries()) {
                    if (imageEntry.getImageUrl() != null) {
                        urlList.add(imageEntry.getImageUrl());
                    }
                }
            }
        }

        new SynchronousImageDownloading(urlList).execute();
    }

    /**
     * Method used to clear the local disk cache.
     */
    public void clearDiskCache() {
        if (imageLoader != null) {
            imageLoader.clearDiskCache();
        } else {
            LoggingUtil.e("Error clearing disk cache!");
        }
    }

    /**
     * AsyncTask class that handles pre-loading of images.
     */
    private class SynchronousImageDownloading extends AsyncTask<Void, Void, Void> {

        private ArrayList<String> urlList;

        SynchronousImageDownloading(ArrayList<String> urlList) {
            this.urlList = urlList;
        }

        @Override
        protected Void doInBackground(Void... voids) {
            for (int i = 0; i < urlList.size(); i++) {
                LoggingUtil.d("Preload url: " + urlList.get(i));
                imageLoader.loadImageSync(urlList.get(i));
            }
            return null;
        }

        @Override
        protected void onPostExecute(Void aVoid) {
            GamedockSDK.getInstance(context).getImageLoadCallbacks().imagePreLoadingCompleted();
        }
    }
}
