package io.gamedock.sdk.utils.dialog;

import android.annotation.TargetApi;
import android.content.Context;
import android.content.DialogInterface;
import android.graphics.Color;
import android.graphics.PorterDuff;
import android.graphics.drawable.Drawable;
import android.text.method.ScrollingMovementMethod;
import android.util.Log;
import android.view.KeyEvent;
import android.view.LayoutInflater;
import android.view.View;
import android.widget.FrameLayout;
import android.widget.ImageView;
import android.widget.RelativeLayout;
import android.widget.TextView;

import androidx.annotation.DrawableRes;
import androidx.annotation.NonNull;
import androidx.annotation.StringRes;
import androidx.annotation.UiThread;
import androidx.core.content.ContextCompat;
import androidx.core.content.res.ResourcesCompat;

import io.gamedock.sdk.R;
import io.gamedock.sdk.utils.dialog.internal.Duration;
import io.gamedock.sdk.utils.dialog.internal.Style;
import io.gamedock.sdk.utils.dialog.internal.UtilsAnimation;
import io.gamedock.sdk.utils.dialog.internal.UtilsLibrary;

public class MaterialStyledDialog extends DialogBase {
    protected final Builder mBuilder;

    public final Builder getBuilder() {
        return mBuilder;
    }

    protected MaterialStyledDialog(Builder builder) {
        super(builder.context, R.style.MD_Dark);
        mBuilder = builder;
        mBuilder.dialog = initMaterialStyledDialog(builder);
    }

    @UiThread
    public void show() {
        if (mBuilder != null && mBuilder.dialog != null)
            mBuilder.dialog.show();
    }

    @UiThread
    public void dismiss() {
        if (mBuilder != null && mBuilder.dialog != null)
            mBuilder.dialog.dismiss();
    }

    @UiThread
    private MaterialDialog initMaterialStyledDialog(final Builder builder) {
        final MaterialDialog.Builder dialogBuilder = new MaterialDialog.Builder(builder.context);

        // Set cancelable
        dialogBuilder.cancelable(false);
        dialogBuilder.canceledOnTouchOutside(false);
        dialogBuilder.keyListener(new OnKeyListener() {
            @Override
            public boolean onKey(DialogInterface dialog, int keyCode, KeyEvent event) {
                return keyCode == KeyEvent.KEYCODE_BACK && event.getAction() == KeyEvent.ACTION_UP;
            }
        });

        // Set style
        dialogBuilder.customView(initStyle(builder), false);

        // Set positive button
        if (builder.positive != null && builder.positive.length() != 0) {
            dialogBuilder.positiveText(builder.positive);

            if(builder.positiveTextColor != 0) {
                dialogBuilder.positiveColor(ContextCompat.getColor(getContext(), builder.positiveTextColor));
            } else {
                dialogBuilder.positiveColor(ContextCompat.getColor(getContext(), R.color.gamedockColor));
            }
        }
        if (builder.positiveCallback != null)
            dialogBuilder.onPositive(builder.positiveCallback);

        if (builder.positiveDrawable != null) {
            dialogBuilder.positiveDrawable(builder.positiveDrawable);
        }

        // set negative button
        if (builder.negative != null && builder.negative.length() != 0) {
            dialogBuilder.negativeText(builder.negative);

            if(builder.negativeTextColor != 0)  {
                dialogBuilder.negativeColor(ContextCompat.getColor(getContext(), builder.negativeTextColor));
            } else {
                dialogBuilder.negativeColor(ContextCompat.getColor(getContext(), R.color.gamedockColor));
            }
        }
        if (builder.negativeCallback != null)
            dialogBuilder.onNegative(builder.negativeCallback);

        if (builder.negativeDrawable != null) {
            dialogBuilder.negativeDrawable(builder.negativeDrawable);
        }

        // Set neutral button
        if (builder.neutral != null && builder.neutral.length() != 0) {
            dialogBuilder.neutralText(builder.neutral);

            if(builder.neutralTextColor != 0) {
                dialogBuilder.neutralColor(ContextCompat.getColor(getContext(), builder.neutralTextColor));
            } else {
                dialogBuilder.neutralColor(ContextCompat.getColor(getContext(), R.color.backgroundColorDark));
            }
        }
        if (builder.neutralCallback != null)
            dialogBuilder.onNeutral(builder.neutralCallback);

        if (builder.neutralDrawable != null) {
            dialogBuilder.neutralDrawable(builder.neutralDrawable);
        }

        // Set auto dismiss when touching the buttons
        dialogBuilder.autoDismiss(builder.isAutoDismiss);

        // Build the dialog with the previous configuration
        MaterialDialog materialDialog = dialogBuilder.build();

        // Set dialog animation and animation duration
        if (builder.isDialogAnimation) {
            if (builder.duration == Duration.NORMAL) {
                materialDialog.getWindow().getAttributes().windowAnimations = R.style.MaterialStyledDialogs_DialogAnimationNormal;
            } else if (builder.duration == Duration.FAST) {
                materialDialog.getWindow().getAttributes().windowAnimations = R.style.MaterialStyledDialogs_DialogAnimationFast;
            } else if (builder.duration == Duration.SLOW) {
                materialDialog.getWindow().getAttributes().windowAnimations = R.style.MaterialStyledDialogs_DialogAnimationSlow;
            }
        }

        return materialDialog;
    }

    @UiThread
    @TargetApi(16)
    private View initStyle(final Builder builder) {
        LayoutInflater inflater = LayoutInflater.from(builder.context);
        View contentView;

        switch (builder.style) {
            case HEADER_WITH_TITLE:
                contentView = inflater.inflate(R.layout.style_dialog_header_title, null);
                break;
            default:
                contentView = inflater.inflate(R.layout.style_dialog_header_icon, null);
                break;
        }

        // Init Views
        RelativeLayout dialogHeaderColor = contentView.findViewById(R.id.md_styled_header_color);
        ImageView dialogHeader = contentView.findViewById(R.id.md_styled_header);
        ImageView dialogPic = contentView.findViewById(R.id.md_styled_header_pic);
        TextView dialogTitle = contentView.findViewById(R.id.md_styled_dialog_title);
        TextView dialogDescription = contentView.findViewById(R.id.md_styled_dialog_description);
        FrameLayout dialogCustomViewGroup = contentView.findViewById(R.id.md_styled_dialog_custom_view);
        View dialogDivider = contentView.findViewById(R.id.md_styled_dialog_divider);

        // Set header color or drawable
        if (builder.headerDrawable != null) {
            dialogHeader.setImageDrawable(builder.headerDrawable);
            // Apply gray/darker overlay to the header if enabled
            if (builder.isDarkerOverlay) {
                dialogHeader.setColorFilter(Color.rgb(123, 123, 123), PorterDuff.Mode.MULTIPLY);
            }
        }
        dialogHeaderColor.setBackgroundColor(builder.primaryColor);
        dialogHeader.setScaleType(builder.headerScaleType);

        //Set the custom view
        if (builder.customView != null) {
            dialogCustomViewGroup.addView(builder.customView);
            dialogCustomViewGroup.setPadding(builder.customViewPaddingLeft, builder.customViewPaddingTop, builder.customViewPaddingRight, builder.customViewPaddingBottom);
        }

        // Set header icon
        if (builder.iconDrawable != null) {
            if (builder.style == Style.HEADER_WITH_TITLE) {
                Log.e("MaterialStyledDialog", "You can't set an icon to the HEADER_WITH_TITLE style.");
            } else {
                dialogPic.setImageDrawable(builder.iconDrawable);
            }
        }

        // Set dialog title
        if (builder.title != null && builder.title.length() != 0) {
            dialogTitle.setText(builder.title);
            dialogTitle.setTextColor(ContextCompat.getColor(getContext(), R.color.colorPrimary));
        } else {
            dialogTitle.setVisibility(View.GONE);
        }

        // Set dialog description
        if (builder.description != null && builder.description.length() != 0) {
            dialogDescription.setText(builder.description);

            // Set scrollable
            dialogDescription.setVerticalScrollBarEnabled(builder.isScrollable);
            if (builder.isScrollable) {
                dialogDescription.setMaxLines(builder.maxLines);
                dialogDescription.setMovementMethod(new ScrollingMovementMethod());
            } else {
                dialogDescription.setMaxLines(Integer.MAX_VALUE);
            }
        } else {
            dialogDescription.setVisibility(View.GONE);
        }

        // Set icon animation
        if (builder.isIconAnimation) {
            if (builder.style != Style.HEADER_WITH_TITLE) {
                UtilsAnimation.zoomInAndOutAnimation(builder.context, dialogPic);
            }
        }

        // Show dialog divider if enabled
        if (builder.isDialogDivider) {
            dialogDivider.setVisibility(View.VISIBLE);
        }

        return contentView;
    }

    public static class Builder {
        protected Context context;

        // build() and show()
        protected MaterialDialog dialog;

        protected Style style; // setStyle()
        protected Duration duration; // withDialogAnimation()
        protected boolean isIconAnimation, isDialogAnimation, isDialogDivider, isCancelable, isScrollable, isDarkerOverlay, isAutoDismiss; // withIconAnimation(), withDialogAnimation(), withDivider(), setCancelable(), setScrollable(), withDarkerOverlay(), autoDismiss()
        protected Drawable headerDrawable, iconDrawable; // setHeaderDrawable(), setIconDrawable()
        protected Drawable positiveDrawable, negativeDrawable, neutralDrawable;
        protected int positiveTextColor, negativeTextColor, neutralTextColor = 0;
        protected Integer primaryColor, maxLines; // setHeaderColor(), setScrollable()
        protected CharSequence title, description; // setTitle(), setDescription()
        protected View customView; // setCustomView()
        protected int customViewPaddingLeft, customViewPaddingTop, customViewPaddingRight, customViewPaddingBottom;
        protected ImageView.ScaleType headerScaleType;

        // .setPositive(), setNegative() and setNeutral()
        protected CharSequence positive, negative, neutral;
        protected MaterialDialog.SingleButtonCallback positiveCallback, negativeCallback, neutralCallback;

        public Builder(Context context) {
            this.context = context;
            this.style = Style.HEADER_WITH_ICON;
            this.isIconAnimation = true;
            this.isDialogAnimation = false;
            this.isDialogDivider = false;
            this.isDarkerOverlay = false;
            this.duration = Duration.NORMAL;
            this.isCancelable = false;
            this.primaryColor = UtilsLibrary.getPrimaryColor(context);
            this.isScrollable = false;
            this.maxLines = 5;
            this.isAutoDismiss = false;
            this.headerScaleType = ImageView.ScaleType.CENTER_CROP;
        }

        public Builder setCustomView(View customView) {
            this.customView = customView;
            this.customViewPaddingLeft = 0;
            this.customViewPaddingRight = 0;
            this.customViewPaddingTop = 0;
            this.customViewPaddingBottom = 0;
            return this;
        }

        public Builder setCustomView(View customView, int left, int top, int right, int bottom) {
            this.customView = customView;
            this.customViewPaddingLeft = UtilsLibrary.dpToPixels(context, left);
            this.customViewPaddingRight = UtilsLibrary.dpToPixels(context, right);
            this.customViewPaddingTop = UtilsLibrary.dpToPixels(context, top);
            this.customViewPaddingBottom = UtilsLibrary.dpToPixels(context, bottom);
            return this;
        }

        public Builder setStyle(Style style) {
            this.style = style;
            return this;
        }

        public Builder withDialogAnimation(Boolean withAnimation) {
            this.isDialogAnimation = withAnimation;
            return this;
        }

        public Builder setTitle(@NonNull CharSequence title) {
            this.title = title;
            return this;
        }

        public Builder setDescription(@NonNull CharSequence description) {
            this.description = description;
            return this;
        }

        public Builder setHeaderDrawable(@DrawableRes Integer drawableRes) {
            this.headerDrawable = ResourcesCompat.getDrawable(context.getResources(), drawableRes, null);
            return this;
        }

        public Builder setPositiveText(@StringRes int buttonTextRes) {
            setPositiveText(this.context.getString(buttonTextRes));
            return this;
        }

        public Builder setPositiveText(@NonNull CharSequence buttonText) {
            this.positive = buttonText;
            return this;
        }

        public Builder setPositiveTextColor(int id) {
            this.positiveTextColor = id;
            return this;
        }

        public Builder setPositiveDrawable(@NonNull Drawable drawable) {
            this.positiveDrawable = drawable;
            return this;
        }

        public Builder onPositive(@NonNull MaterialDialog.SingleButtonCallback callback) {
            this.positiveCallback = callback;
            return this;
        }

        public Builder setNegativeText(@NonNull CharSequence buttonText) {
            this.negative = buttonText;
            return this;
        }

        public Builder setNegativeTextColor(int id) {
            this.negativeTextColor = id;
            return this;
        }

        public Builder setNegativeDrawable(@NonNull Drawable drawable) {
            this.negativeDrawable = drawable;
            return this;
        }

        public Builder onNegative(@NonNull MaterialDialog.SingleButtonCallback callback) {
            this.negativeCallback = callback;
            return this;
        }

        public Builder setNeutralText(@NonNull CharSequence buttonText) {
            this.neutral = buttonText;
            return this;
        }

        public Builder setNeutralTextColor(int id) {
            this.neutralTextColor = id;
            return this;
        }

        public Builder setNeutralDrawable(@NonNull Drawable drawable) {
            this.neutralDrawable = drawable;
            return this;
        }

        public Builder onNeutral(@NonNull MaterialDialog.SingleButtonCallback callback) {
            this.neutralCallback = callback;
            return this;
        }

        public Builder autoDismiss(Boolean dismiss) {
            this.isAutoDismiss = dismiss;
            return this;
        }

        @UiThread
        public MaterialStyledDialog build() {
            return new MaterialStyledDialog(this);
        }

        @UiThread
        public MaterialStyledDialog show() {
            MaterialStyledDialog materialStyledDialog = build();
            materialStyledDialog.show();
            return materialStyledDialog;
        }

    }

}