package io.gamedock.sdk.userdata.missiondata;

import android.content.Context;

import com.google.gson.JsonObject;

import org.json.JSONArray;
import org.json.JSONException;

import java.util.ArrayList;

import io.gamedock.sdk.GamedockSDK;
import io.gamedock.sdk.events.internal.MissionEvent;
import io.gamedock.sdk.mission.MissionConfigurationManager;
import io.gamedock.sdk.models.mission.Container;
import io.gamedock.sdk.models.mission.ContainerRewards;
import io.gamedock.sdk.models.mission.Mission;
import io.gamedock.sdk.models.mission.MissionRewards;
import io.gamedock.sdk.models.userdata.UpdatedUserData;
import io.gamedock.sdk.models.userdata.UserData;
import io.gamedock.sdk.models.userdata.inventory.PlayerItem;
import io.gamedock.sdk.models.userdata.inventory.UniquePlayerItem;
import io.gamedock.sdk.models.userdata.mission.ContainerProgress;
import io.gamedock.sdk.models.userdata.mission.MissionData;
import io.gamedock.sdk.models.userdata.mission.MissionProgress;
import io.gamedock.sdk.models.userdata.mission.UpdatedMissionData;
import io.gamedock.sdk.userdata.UserDataManager;
import io.gamedock.sdk.userdata.playerdata.PlayerDataManager;
import io.gamedock.sdk.userdata.playerdata.PlayerDataUpdateReasons;
import io.gamedock.sdk.userdata.playerdata.functions.PlayerDataOperations;
import io.gamedock.sdk.userdata.playerdata.functions.PlayerDataSending;
import io.gamedock.sdk.utils.error.ErrorCodes;
import io.gamedock.sdk.utils.features.FeaturesUtil;
import io.gamedock.sdk.utils.logging.LoggingUtil;

/**
 * Class that manages all the mission progress of the user.
 */
public class MissionDataManager {

    public static final String MissionData = "missionData";
    public static final int frequencyLimit = 1000;
    private static final String intervalTimeString = "Interval time: " + frequencyLimit + "ms";

    private Context context;

    public enum Status {
        NULL,
        INACTIVE,
        PENDING_ACTIVATION,
        IN_PROGRESS,
        PENDING_COLLECTION,
        COMPLETED
    }

    public MissionDataManager(Context context) {
        this.context = context;
    }

    /**
     * Method used to process the mission data for the user coming from a user data request.
     *
     * @param missionData The mission data that needs to be updated.
     */
    public void processMissionData(MissionData missionData) {
        if (missionData == null) {
            return;
        }

        UpdatedMissionData updatedMissionData = new UpdatedMissionData();

        UserData userData = UserDataManager.getInstance(context).getUserData();

        if (userData == null) {
            return;
        }

        for (ContainerProgress containerProgress : missionData.getContainerProgress()) {
            containerProgress.setStatus(containerProgress.getStatus().replace(" ", "_"));
        }

        for (MissionProgress missionProgress : missionData.getMissionProgress()) {
            missionProgress.setStatus(missionProgress.getStatus().replace(" ", "_"));
        }

        userData.setMissionData(missionData);

        UserDataManager.getInstance(context).updateUserData(userData);

        if (userData.updated) {
            updatedMissionData.setContainerProgressList(userData.getMissionData().getContainerProgress());
            updatedMissionData.setMissionProgressList(userData.getMissionData().getMissionProgress());

            GamedockSDK.getInstance(context).getUserDataCallbacks().missionDataUpdated(PlayerDataUpdateReasons.ServerUpdate, updatedMissionData, GamedockSDK.getInstance(context).getGson().toJson(getUserAllContainerProgress(MissionDataManager.Status.NULL)), GamedockSDK.getInstance(context).getGson().toJson(getUserAllMissionProgress(MissionDataManager.Status.NULL)));
        }
    }

    /**
     * Method used to get all container progress of the user or progress based on certain @{@link Status}
     *
     * @param containerStatus The @{@link Status} of the container if needed.
     * @return The list of container progress.
     */
    public ArrayList<ContainerProgress> getUserAllContainerProgress(Status containerStatus) {
        if (!FeaturesUtil.isFeatureEnabled(UserDataManager.FEATURE_NAME)) {
            return new ArrayList<>();
        }

        UserData userData = UserDataManager.getInstance(context).getUserData();
        if (userData == null) {
            return new ArrayList<>();
        }

        if (containerStatus == null || containerStatus == Status.NULL) {
            return userData.getMissionData().getContainerProgress();
        }

        ArrayList<ContainerProgress> selectedContainerProgress = new ArrayList<>();
        for (ContainerProgress entry : userData.getMissionData().getContainerProgress()) {
            if (entry.getStatus().equals(containerStatus.toString())) {
                selectedContainerProgress.add(entry);
            }
        }

        return selectedContainerProgress;
    }

    /**
     * Method used to retrieve the container progress based on an id.
     *
     * @param containerId The id of the container.
     * @return The Container Progress or null if no progress was found.
     */
    public ContainerProgress getContainerProgress(String containerId) {
        if (!FeaturesUtil.isFeatureEnabled(UserDataManager.FEATURE_NAME)) {
            return null;
        }

        UserData userData = UserDataManager.getInstance(context).getUserData();
        if (userData == null) {
            return null;
        }

        for (ContainerProgress entry : userData.getMissionData().getContainerProgress()) {
            if (entry.getContainerId().equals(containerId)) {
                return entry.clone();
            }
        }

        return null;
    }

    /**
     * Method used internally to update the container progress.
     *
     * @param userData          The {@link UserData} object that contains all the information regarding the player.
     * @param containerProgress The progress that needs to be updated.
     */
    private void updateContainerProgress(UserData userData, ContainerProgress containerProgress) {
        for (int i = 0; i < userData.getMissionData().getContainerProgress().size(); i++) {
            if (userData.getMissionData().getContainerProgress().get(i).getContainerId().equals(containerProgress.getContainerId())) {
                userData.getMissionData().getContainerProgress().get(i).setStatus(containerProgress.getStatus());
                userData.getMissionData().getContainerProgress().get(i).setLastCompleted(containerProgress.getLastCompleted());
                break;
            }
        }
    }

    /**
     * Method used to get all mission progress of the user or progress based on certain @{@link Status}
     *
     * @param missionStatus The @{@link Status} of the mission if needed.
     * @return The list of mission progress.
     */
    public ArrayList<MissionProgress> getUserAllMissionProgress(Status missionStatus) {
        if (!FeaturesUtil.isFeatureEnabled(UserDataManager.FEATURE_NAME)) {
            return new ArrayList<>();
        }

        UserData userData = UserDataManager.getInstance(context).getUserData();
        if (userData == null) {
            return new ArrayList<>();
        }

        if (missionStatus == null || missionStatus == Status.NULL) {
            return userData.getMissionData().getMissionProgress();
        }

        ArrayList<MissionProgress> selectedMissionProgress = new ArrayList<>();
        for (MissionProgress entry : userData.getMissionData().getMissionProgress()) {
            if (entry.getStatus().equals(missionStatus.toString())) {
                selectedMissionProgress.add(entry);
            }
        }

        return selectedMissionProgress;
    }

    /**
     * Method used to retrieve the mission progress based on an id.
     *
     * @param missionId The id of the mission.
     * @return The Mission Progress or null if no progress was found.
     */
    public MissionProgress getMissionProgress(String missionId) {
        if (!FeaturesUtil.isFeatureEnabled(UserDataManager.FEATURE_NAME)) {
            return null;
        }

        UserData userData = UserDataManager.getInstance(context).getUserData();
        if (userData == null) {
            return null;
        }

        for (MissionProgress entry : userData.getMissionData().getMissionProgress()) {
            if (entry.getMissionId().equals(missionId)) {
                return entry.clone();
            }
        }

        return null;
    }

    /**
     * Method used internally to update the mission progress.
     *
     * @param userData        The {@link UserData} object that contains all the information regarding the player.
     * @param missionProgress The progress that needs to be updated.
     */
    private void updateMissionProgress(UserData userData, MissionProgress missionProgress) {
        for (int i = 0; i < userData.getMissionData().getMissionProgress().size(); i++) {
            if (userData.getMissionData().getMissionProgress().get(i).getMissionId().equals(missionProgress.getMissionId())) {
                userData.getMissionData().getMissionProgress().get(i).setStatus(missionProgress.getStatus());
                userData.getMissionData().getMissionProgress().get(i).setProgress(missionProgress.getProgress());
                userData.getMissionData().getMissionProgress().get(i).setLastCompleted(missionProgress.getLastCompleted());
                break;
            }
        }

        UserDataManager.getInstance(context).updateUserData(userData);
    }

    /**
     * Method used to update both the mission and container progress in one operation based on two lists of progress supplied.
     *
     * @param containerProgressData The container progress list containing updates.
     * @param missionProgressData   The mission progress list containing updates.
     * @param reason                The reason for which the update was triggered. A standard list of reasons can be found at {@link PlayerDataUpdateReasons}.
     * @param reasonDetails         The details of the reason. This parameter is most of the time defined by the developer in combination with the BI team.
     * @param location              The location from which the update event has been triggered.
     * @param transactionId         The transaction id for the IAP if the update was triggered by an IAP. Helps linking the event on the backend.
     */
    public void updateContainerAndMissionProgress(ArrayList<ContainerProgress> containerProgressData, ArrayList<MissionProgress> missionProgressData, String reason, String reasonDetails, String location, String transactionId) {
        if (!FeaturesUtil.isFeatureEnabled(UserDataManager.FEATURE_NAME)) {
            return;
        }

        UserData userData = UserDataManager.getInstance(context).getUserData();
        if (userData == null) {
            GamedockSDK.getInstance(context).getUserDataCallbacks().userDataError(ErrorCodes.LoadFailed);
            return;
        }

        if (containerProgressData == null) {
            GamedockSDK.getInstance(context).getUserDataCallbacks().userDataError(ErrorCodes.ContainerOperation);
            return;
        }

        if (missionProgressData == null) {
            GamedockSDK.getInstance(context).getUserDataCallbacks().userDataError(ErrorCodes.MissionOperation);
            return;
        }

        if (!isAllowedOperationByFrequency(userData)) {
            LoggingUtil.e("Operation triggered too frequently. Please reduce the amount of calls done for this operation: " + "updateContainerAndMissionProgress. " + intervalTimeString);
        }

        ArrayList<ContainerProgress> containerProgressToBeSent = new ArrayList<>();
        ArrayList<MissionProgress> missionProgressToBeSent = new ArrayList<>();

        if (calculateContainerProgress(userData, containerProgressData, containerProgressToBeSent)) {
            return;
        }

        calculateMissionProgress(userData, missionProgressData, missionProgressToBeSent);

        if (containerProgressToBeSent.size() == 0 && missionProgressToBeSent.size() == 0) {
            return;
        }

        userData = UserDataManager.getInstance(context).updateUserDataVersion(userData);
        userData = UserDataManager.getInstance(context).updateUserDataMeta(userData);

        userData.getMissionData().setLastOperationTime(System.currentTimeMillis());

        UserDataManager.getInstance(context).updateUserData(userData);

        sendUpdateUserMissionDataEvent(userData, containerProgressToBeSent, missionProgressToBeSent, reason, reasonDetails, location, transactionId);

        UpdatedMissionData updatedMissionData = new UpdatedMissionData();
        updatedMissionData.setContainerProgressList(containerProgressToBeSent);
        updatedMissionData.setMissionProgressList(missionProgressToBeSent);

        GamedockSDK.getInstance(context).getUserDataCallbacks().missionDataUpdated(reason, updatedMissionData, GamedockSDK.getInstance(context).getGson().toJson(getUserAllContainerProgress(MissionDataManager.Status.NULL)), GamedockSDK.getInstance(context).getGson().toJson(getUserAllMissionProgress(MissionDataManager.Status.NULL)));
    }

    /**
     * Method used to update container progress in one operation based on a lists of progress supplied.
     *
     * @param containerProgressData The container progress list containing updates.
     * @param reason                The reason for which the update was triggered. A standard list of reasons can be found at {@link PlayerDataUpdateReasons}.
     * @param reasonDetails         The details of the reason. This parameter is most of the time defined by the developer in combination with the BI team.
     * @param location              The location from which the update event has been triggered.
     * @param transactionId         The transaction id for the IAP if the update was triggered by an IAP. Helps linking the event on the backend.
     */
    public void updateContainerProgress(ArrayList<ContainerProgress> containerProgressData, String reason, String reasonDetails, String location, String transactionId) {
        if (!FeaturesUtil.isFeatureEnabled(UserDataManager.FEATURE_NAME)) {
            return;
        }

        UserData userData = UserDataManager.getInstance(context).getUserData();
        if (userData == null) {
            GamedockSDK.getInstance(context).getUserDataCallbacks().userDataError(ErrorCodes.LoadFailed);
            return;
        }

        if (containerProgressData == null) {
            GamedockSDK.getInstance(context).getUserDataCallbacks().userDataError(ErrorCodes.ContainerOperation);
            return;
        }

        if (!isAllowedOperationByFrequency(userData)) {
            LoggingUtil.e("Operation triggered too frequently. Please reduce the amount of calls done for this operation: " + "updateContainerProgress. " + intervalTimeString);
        }

        ArrayList<ContainerProgress> progressToBeSent = new ArrayList<>();
        if (calculateContainerProgress(userData, containerProgressData, progressToBeSent)) {
            return;
        }


        if (progressToBeSent.size() == 0) {
            return;
        }

        userData = UserDataManager.getInstance(context).updateUserDataVersion(userData);
        userData = UserDataManager.getInstance(context).updateUserDataMeta(userData);

        userData.getMissionData().setLastOperationTime(System.currentTimeMillis());

        UserDataManager.getInstance(context).updateUserData(userData);

        sendUpdateUserMissionDataEvent(userData, progressToBeSent, new ArrayList<MissionProgress>(), reason, reasonDetails, location, transactionId);

        UpdatedMissionData updatedMissionData = new UpdatedMissionData();
        updatedMissionData.setContainerProgressList(progressToBeSent);

        GamedockSDK.getInstance(context).getUserDataCallbacks().missionDataUpdated(reason, updatedMissionData, GamedockSDK.getInstance(context).getGson().toJson(getUserAllContainerProgress(MissionDataManager.Status.NULL)), GamedockSDK.getInstance(context).getGson().toJson(getUserAllMissionProgress(MissionDataManager.Status.NULL)));
    }

    /**
     * Method used to update mission progress in one operation based on a lists of progress supplied.
     *
     * @param missionProgressData The mission progress list containing updates.
     * @param reason              The reason for which the update was triggered. A standard list of reasons can be found at {@link PlayerDataUpdateReasons}.
     * @param reasonDetails       The details of the reason. This parameter is most of the time defined by the developer in combination with the BI team.
     * @param location            The location from which the update event has been triggered.
     * @param transactionId       The transaction id for the IAP if the update was triggered by an IAP. Helps linking the event on the backend.
     */
    public void updateMissionProgress(ArrayList<MissionProgress> missionProgressData, String reason, String reasonDetails, String location, String transactionId) {
        if (!FeaturesUtil.isFeatureEnabled(UserDataManager.FEATURE_NAME)) {
            return;
        }

        UserData userData = UserDataManager.getInstance(context).getUserData();
        if (userData == null) {
            GamedockSDK.getInstance(context).getUserDataCallbacks().userDataError(ErrorCodes.LoadFailed);
            return;
        }

        if (missionProgressData == null) {
            GamedockSDK.getInstance(context).getUserDataCallbacks().userDataError(ErrorCodes.MissionOperation);
            return;
        }

        if (!isAllowedOperationByFrequency(userData)) {
            LoggingUtil.e("Operation triggered too frequently. Please reduce the amount of calls done for this operation: " + "updateContainerProgress. " + intervalTimeString);
        }

        ArrayList<MissionProgress> progressToBeSent = new ArrayList<>();
        calculateMissionProgress(userData, missionProgressData, progressToBeSent);

        if (progressToBeSent.size() == 0) {
            return;
        }

        userData = UserDataManager.getInstance(context).updateUserDataVersion(userData);
        userData = UserDataManager.getInstance(context).updateUserDataMeta(userData);

        userData.getMissionData().setLastOperationTime(System.currentTimeMillis());

        UserDataManager.getInstance(context).updateUserData(userData);

        sendUpdateUserMissionDataEvent(userData, new ArrayList<ContainerProgress>(), progressToBeSent, reason, reasonDetails, location, transactionId);

        UpdatedMissionData updatedMissionData = new UpdatedMissionData();
        updatedMissionData.setMissionProgressList(progressToBeSent);

        GamedockSDK.getInstance(context).getUserDataCallbacks().missionDataUpdated(reason, updatedMissionData, GamedockSDK.getInstance(context).getGson().toJson(getUserAllContainerProgress(MissionDataManager.Status.NULL)), GamedockSDK.getInstance(context).getGson().toJson(getUserAllMissionProgress(MissionDataManager.Status.NULL)));
    }

    /**
     * Method used to claim the rewards for a list of container if the @{@link Status} is PENDING_COLLECTION.
     *
     * @param containerProgressList The container progress list containing updates.
     * @param reason                The reason for which the update was triggered. A standard list of reasons can be found at {@link PlayerDataUpdateReasons}.
     * @param reasonDetails         The details of the reason. This parameter is most of the time defined by the developer in combination with the BI team.
     * @param location              The location from which the update event has been triggered.
     * @param transactionId         The transaction id for the IAP if the update was triggered by an IAP. Helps linking the event on the backend.
     */
    public void claimContainersReward(ArrayList<ContainerProgress> containerProgressList, String reason, String reasonDetails, String location, String transactionId) {
        if (!FeaturesUtil.isFeatureEnabled(UserDataManager.FEATURE_NAME)) {
            return;
        }

        UserData userData = UserDataManager.getInstance(context).getUserData();
        if (userData == null) {
            GamedockSDK.getInstance(context).getUserDataCallbacks().userDataError(ErrorCodes.LoadFailed);
            return;
        }

        if (containerProgressList == null) {
            GamedockSDK.getInstance(context).getUserDataCallbacks().userDataError(ErrorCodes.ContainerOperation);
            return;
        }

        UpdatedUserData updatedUserData = new UpdatedUserData();
        ArrayList<UniquePlayerItem> uniquePlayerItems = new ArrayList<>();

        for (ContainerProgress containerProgress : containerProgressList) {
            if (!containerProgress.getStatus().equals("PENDING_COLLECTION")) {
                GamedockSDK.getInstance(context).getUserDataCallbacks().userDataError(ErrorCodes.ContainerOperation);
                return;
            }

            Container container = MissionConfigurationManager.getInstance(context).getContainerConfiguration(containerProgress.getContainerId());
            for (ContainerRewards reward : container.getRewards()) {
                if (reward.getType().equals(PlayerDataManager.Currency)) {
                    PlayerDataOperations.updateCurrencyReward(context, userData, reward.getId(), reward.getAmount(), null, updatedUserData);
                } else if (reward.getType().equals(PlayerDataManager.Item) || reward.getType().equals(PlayerDataManager.GachaCheck)) {
                    PlayerDataOperations.updateItemReward(context, userData, reward.getId(), reward.getAmount(), uniquePlayerItems, null, updatedUserData);
                }
            }

            updatedUserData.claimedContainers.add(containerProgress.getContainerId());
        }

        //Updates UserData Version and Meta
        userData = UserDataManager.getInstance(context).updateUserDataVersion(userData);
        userData = UserDataManager.getInstance(context).updateUserDataMeta(userData);

        //Saves updated user data
        UserDataManager.getInstance(context).updateUserData(userData);

        for (PlayerItem playerItem : userData.getInventory().getItemsMap().values()) {
            if (playerItem.isGacha() && playerItem.getDelta() != 0) {
                boolean bFound = false;
                for (PlayerItem existingItem : updatedUserData.items) {
                    if (existingItem.getId() == playerItem.getId()) {
                        bFound = true;
                        break;
                    }
                }
                if (!bFound) {
                    updatedUserData.items.add(playerItem.clone());
                }
            }
        }

        PlayerDataSending.sendUpdatePlayerDataEvent(context, UserDataManager.getInstance(context).getPlayerDataManager().gson, userData, null, reason, reasonDetails, location, transactionId, null, null);

        for (UniquePlayerItem uniquePlayerItem : uniquePlayerItems) {
            GamedockSDK.getInstance(context).getUserDataCallbacks().playerDataNewUniqueItem(uniquePlayerItem, 0, 0, 0, 0, reason);
        }

        GamedockSDK.getInstance(context).getUserDataCallbacks().playerDataUpdated(reason, updatedUserData, GamedockSDK.getInstance(context).getWallet(), GamedockSDK.getInstance(context).getInventory());
    }

    /**
     * Method used to claim the rewards for a list of missions if the @{@link Status} is PENDING_COLLECTION.
     *
     * @param missionProgressList The mission progress list containing updates.
     * @param reason              The reason for which the update was triggered. A standard list of reasons can be found at {@link PlayerDataUpdateReasons}.
     * @param reasonDetails       The details of the reason. This parameter is most of the time defined by the developer in combination with the BI team.
     * @param location            The location from which the update event has been triggered.
     * @param transactionId       The transaction id for the IAP if the update was triggered by an IAP. Helps linking the event on the backend.
     */
    public void claimMissionsReward(ArrayList<MissionProgress> missionProgressList, String reason, String reasonDetails, String location, String transactionId) {
        if (!FeaturesUtil.isFeatureEnabled(UserDataManager.FEATURE_NAME)) {
            return;
        }

        UserData userData = UserDataManager.getInstance(context).getUserData();
        if (userData == null) {
            GamedockSDK.getInstance(context).getUserDataCallbacks().userDataError(ErrorCodes.LoadFailed);
            return;
        }

        if (missionProgressList == null) {
            GamedockSDK.getInstance(context).getUserDataCallbacks().userDataError(ErrorCodes.ContainerOperation);
            return;
        }

        UpdatedUserData updatedUserData = new UpdatedUserData();
        ArrayList<UniquePlayerItem> uniquePlayerItems = new ArrayList<>();

        for (MissionProgress missionProgress : missionProgressList) {
            if (!missionProgress.getStatus().equals("PENDING_COLLECTION")) {
                GamedockSDK.getInstance(context).getUserDataCallbacks().userDataError(ErrorCodes.ContainerOperation);
                return;
            }

            Mission mission = MissionConfigurationManager.getInstance(context).getMissionConfiguration(missionProgress.getMissionId());
            for (MissionRewards reward : mission.getRewards()) {
                if (reward.getType().equals(PlayerDataManager.Currency)) {
                    PlayerDataOperations.updateCurrencyReward(context, userData, reward.getId(), reward.getAmount(), null, updatedUserData);
                } else if (reward.getType().equals(PlayerDataManager.Item) || reward.getType().equals(PlayerDataManager.GachaCheck)) {
                    PlayerDataOperations.updateItemReward(context, userData, reward.getId(), reward.getAmount(), uniquePlayerItems, null, updatedUserData);
                }
            }

            updatedUserData.claimedMissions.add(missionProgress.getMissionId());
        }

        //Updates UserData Version and Meta
        userData = UserDataManager.getInstance(context).updateUserDataVersion(userData);
        userData = UserDataManager.getInstance(context).updateUserDataMeta(userData);

        //Saves updated user data0
        UserDataManager.getInstance(context).updateUserData(userData);

        for (PlayerItem playerItem : userData.getInventory().getItemsMap().values()) {
            if (playerItem.isGacha() && playerItem.getDelta() != 0) {
                boolean bFound = false;
                for (PlayerItem existingItem : updatedUserData.items) {
                    if (existingItem.getId() == playerItem.getId()) {
                        bFound = true;
                        break;
                    }
                }
                if (!bFound) {
                    updatedUserData.items.add(playerItem.clone());
                }
            }
        }

        PlayerDataSending.sendUpdatePlayerDataEvent(context, UserDataManager.getInstance(context).getPlayerDataManager().gson, userData, null, reason, reasonDetails, location, transactionId, null, null);

        for (UniquePlayerItem uniquePlayerItem : uniquePlayerItems) {
            GamedockSDK.getInstance(context).getUserDataCallbacks().playerDataNewUniqueItem(uniquePlayerItem, 0, 0, 0, 0, reason);
        }

        GamedockSDK.getInstance(context).getUserDataCallbacks().playerDataUpdated(reason, updatedUserData, GamedockSDK.getInstance(context).getWallet(), GamedockSDK.getInstance(context).getInventory());
    }

    /**
     * Method used internally to calculate and update the container progress.
     *
     * @param userData                  The {@link UserData} object that contains all the information regarding the player.
     * @param containerProgressList     The list of container progress that needs to be updated.
     * @param containerProgressToBeSent The list of container progress that will be marked and sent to the backend.
     * @return
     */
    private boolean calculateContainerProgress(UserData userData, ArrayList<ContainerProgress> containerProgressList, ArrayList<ContainerProgress> containerProgressToBeSent) {
        if (!FeaturesUtil.isFeatureEnabled(UserDataManager.FEATURE_NAME)) {
            return false;
        }

        for (ContainerProgress containerProgress : containerProgressList) {
            ContainerProgress currentProgress = getContainerProgress(containerProgress.getContainerId());
            if (currentProgress == null) {
                userData.getMissionData().getContainerProgress().add(containerProgress);
                containerProgressToBeSent.add(containerProgress);
            } else {
                if (currentProgress.getStatus().equals("COMPLETED")) {
                    continue;
                }

                if (currentProgress.getStatus().equals("PENDING_ACTIVATION")) {
                    GamedockSDK.getInstance(context).getUserDataCallbacks().userDataError(ErrorCodes.ContainerOperation);
                    return true;
                }

                currentProgress.setStatus(containerProgress.getStatus());

                if (containerProgress.getStatus().equals("COMPLETED") && currentProgress.getLastCompleted() == 0) {
                    currentProgress.setLastCompleted(System.currentTimeMillis());
                }

                updateContainerProgress(userData, currentProgress);

                containerProgressToBeSent.add(currentProgress);
            }
        }
        return false;
    }

    /**
     * Method used internally to calculate and update the mission progress.
     *
     * @param userData            The {@link UserData} object that contains all the information regarding the player.
     * @param missionProgressList The list of mission progress that needs to be updated.
     * @param progressToBeSent    The list of mission progress that will be marked and sent to the backend.
     * @return
     */
    private void calculateMissionProgress(UserData userData, ArrayList<MissionProgress> missionProgressList, ArrayList<MissionProgress> progressToBeSent) {
        if (!FeaturesUtil.isFeatureEnabled(UserDataManager.FEATURE_NAME)) {
            return;
        }

        for (MissionProgress missionProgress : missionProgressList) {
            MissionProgress currentProgress = getMissionProgress(missionProgress.getMissionId());
            if (currentProgress == null) {
                userData.getMissionData().getMissionProgress().add(missionProgress);
                progressToBeSent.add(missionProgress);
            } else {
                if (currentProgress.getStatus().equals("COMPLETED")) {
                    currentProgress.setProgress(new JsonObject());
                    continue;
                }

                currentProgress.setStatus(missionProgress.getStatus());
                currentProgress.setProgress(missionProgress.getProgress());

                if (missionProgress.getStatus().equals("COMPLETED") && currentProgress.getLastCompleted() == 0) {
                    currentProgress.setLastCompleted(System.currentTimeMillis());
                }

                updateMissionProgress(userData, currentProgress);

                progressToBeSent.add(currentProgress);
            }
        }
    }

    /**
     * Method used to check if updates are done in too short time frame.
     *
     * @param userData The {@link UserData} object that contains all the information regarding the player.
     * @return
     */
    private boolean isAllowedOperationByFrequency(UserData userData) {
        long lastOperationTime = userData.getMissionData().getLastOperationTime();
        long currentOperationTime = System.currentTimeMillis();

        return (currentOperationTime - lastOperationTime) > frequencyLimit;
    }

    /**
     * Method used internally to send an update to the server containing the full mission data state in which containers and mission can be marked as being updated.
     *
     * @param userData                      The {@link UserData} object that contains all the information regarding the player.
     * @param modifiedContainerProgressList The list of container progress that has to be marked as updated/modified.
     * @param modifiedMissionProgressList   The list of mission progress that has to be marked as updated/modified.
     * @param reason                        The reason for which the update was triggered. A standard list of reasons can be found at {@link PlayerDataUpdateReasons}.
     * @param reasonDetails                 The details of the reason. This parameter is most of the time defined by the developer in combination with the BI team.
     * @param location                      The location from which the update event has been triggered.
     * @param transactionId                 The transaction id for the IAP if the update was triggered by an IAP. Helps linking the event on the backend.
     */
    private void sendUpdateUserMissionDataEvent(UserData userData, ArrayList<ContainerProgress> modifiedContainerProgressList, ArrayList<MissionProgress> modifiedMissionProgressList, String reason, String reasonDetails, String location, String transactionId) {
        if (!FeaturesUtil.isFeatureEnabled(UserDataManager.FEATURE_NAME)) {
            return;
        }

        try {
            MissionEvent missionEvent = new MissionEvent(context);
            missionEvent.setUpdateUserMissionData();

            JSONArray containerProgressJSON = new JSONArray(GamedockSDK.getInstance(context).getGson().toJson(userData.getMissionData().getContainerProgress()));
            for (int i = 0; i < containerProgressJSON.length(); i++) {
                containerProgressJSON.getJSONObject(i).put("status", containerProgressJSON.getJSONObject(i).getString("status").replace("_", " "));
                for (ContainerProgress modifiedContainer : modifiedContainerProgressList) {
                    if (containerProgressJSON.getJSONObject(i).getString("containerId").equals(modifiedContainer.getContainerId())) {
                        containerProgressJSON.getJSONObject(i).put("modified", true);
                    }
                }

            }

            JSONArray missionProgressJSON = new JSONArray(GamedockSDK.getInstance(context).getGson().toJson(userData.getMissionData().getMissionProgress()));
            for (int i = 0; i < missionProgressJSON.length(); i++) {
                missionProgressJSON.getJSONObject(i).put("status", missionProgressJSON.getJSONObject(i).getString("status").replace("_", " "));
                for (MissionProgress modifiedMission : modifiedMissionProgressList) {
                    if (missionProgressJSON.getJSONObject(i).getString("missionId").equals(modifiedMission.getMissionId())) {
                        missionProgressJSON.getJSONObject(i).put("modified", true);
                    }
                }
            }

            missionEvent.addCustomData("containerProgress", containerProgressJSON);
            missionEvent.addCustomData("missionProgress", missionProgressJSON);

            missionEvent.addCustomData("reason", reason);

            if (reasonDetails != null) {
                missionEvent.addCustomData("reasonDetails", reasonDetails);
            }

            if (location != null) {
                missionEvent.addCustomData("location", location);
            }

            if (transactionId != null) {
                missionEvent.addCustomData("transactionId", transactionId);
            }

            missionEvent.addCustomData(UserDataManager.DeviceVersions, UserDataManager.getInstance(context).createUserDataVersionsJson(UserDataManager.getInstance(context).getUserData().getUserDataVersions()));

            GamedockSDK.getInstance(context).trackEvent(missionEvent, null);
        } catch (JSONException e) {
            e.printStackTrace();
        }
    }

    /**
     * Method used to reset the user's mission data progress (containers and missions).
     */
    public void resetMissionDataProgress() {
        if (!FeaturesUtil.isFeatureEnabled(UserDataManager.FEATURE_NAME)) {
            return;
        }

        UserData userData = UserDataManager.getInstance(context).getUserData();
        if (userData == null) {
            GamedockSDK.getInstance(context).getUserDataCallbacks().userDataError(ErrorCodes.LoadFailed);
            return;
        }

        userData.getMissionData().getContainerProgress().clear();
        userData.getMissionData().getMissionProgress().clear();

        UserDataManager.getInstance(context).updateUserData(userData);

        sendUpdateUserMissionDataEvent(userData, userData.getMissionData().getContainerProgress(), userData.getMissionData().getMissionProgress(), PlayerDataUpdateReasons.Reset, null, null, null);

        GamedockSDK.getInstance(context).getUserDataCallbacks().missionDataUpdated(PlayerDataUpdateReasons.Reset, new UpdatedMissionData(), GamedockSDK.getInstance(context).getGson().toJson(getUserAllContainerProgress(MissionDataManager.Status.NULL)), GamedockSDK.getInstance(context).getGson().toJson(getUserAllMissionProgress(MissionDataManager.Status.NULL)));
    }
}
