package io.gamedock.sdk.userdata.gamestate;


import android.content.Context;

import org.json.JSONArray;
import org.json.JSONException;
import org.json.JSONObject;

import io.gamedock.sdk.GamedockSDK;
import io.gamedock.sdk.events.EventActionListener;
import io.gamedock.sdk.events.internal.UserDataEvent;
import io.gamedock.sdk.models.userdata.UserData;
import io.gamedock.sdk.models.userdata.gamestate.GameState;
import io.gamedock.sdk.models.userdata.inventory.Inventory;
import io.gamedock.sdk.models.userdata.wallet.Wallet;
import io.gamedock.sdk.userdata.UserDataManager;
import io.gamedock.sdk.utils.error.ErrorCodes;
import io.gamedock.sdk.utils.features.FeaturesUtil;
import io.gamedock.sdk.utils.logging.LoggingUtil;

/**
 * Utility class that handles all the logic regarding game state information of the SDK.
 * It handles the saving and retrieving of both "public" and "private" game state data.
 * It has the logic to request the game state for other users.
 */
public class GameStateManager {

    private Context context;

    public static final String Provider = "provider";
    public static final String Users = "users";

    public static final int frequencyLimit = 1000;
    private static final String intervalTimeString = "Interval time: " + frequencyLimit + "ms";

    public GameStateManager(Context context) {
        this.context = context;
    }

    /**
     * Method that requests the "public" game state of other user based on the supplied user id list.
     * In order to request the "public" game state a provider and a custom user id associated with that provider needs to be set.
     *
     * @param provider    The provider with which the user ids are associated.
     * @param userIdsList A String List of all the ids that the "public" game state need to be retrieved.
     */
    public void requestOtherUsersGameState(String provider, String userIdsList) {
        if (!FeaturesUtil.isFeatureEnabled(UserDataManager.FEATURE_NAME)) {
            return;
        }

        UserDataEvent gameStateEvent = new UserDataEvent(context);
        gameStateEvent.setRequestOtherUsersGameState();

        gameStateEvent.addCustomData(Provider, provider);

        JSONArray jsonArray = null;
        try {
            jsonArray = new JSONArray(userIdsList);
        } catch (JSONException e) {
            e.printStackTrace();
        }

        gameStateEvent.addCustomData(Users, jsonArray);

        GamedockSDK.getInstance(context).trackEvent(gameStateEvent, null);
    }

    /**
     * Method that process the response from Gamedock backend regarding Game State.
     * It handles both "public" and "private" game states.
     *
     * @param gameStateData The received game state data in JSON {@link String}
     */
    public void processMyGameStateResponse(String gameStateData) {
        if (gameStateData == null) {
            return;
        }

        try {
            UserData userData = UserDataManager.getInstance(context).getUserData();
            JSONObject object = new JSONObject(gameStateData);

            if (object.has(UserDataManager.GameStates)) {
                JSONArray gameStateArray = object.getJSONArray(UserDataManager.GameStates);

                for (int i = 0; i < gameStateArray.length(); i++) {
                    String access = null;
                    if (gameStateArray.getJSONObject(i).has(UserDataManager.GameStateAccess)) {
                        access = gameStateArray.getJSONObject(i).getString(UserDataManager.GameStateAccess);
                    }

                    String data = null;
                    if (gameStateArray.getJSONObject(i).has(UserDataManager.Data)) {
                        data = gameStateArray.getJSONObject(i).getString(UserDataManager.Data);
                    }

                    if (access != null && data != null && userData != null) {
                        GameState gameState = new GameState(access);
                        gameState.setData(data);

                        if (gameState.getAccess().equals(UserDataManager.PublicGameStateAccess)) {
                            userData.setPublicGameState(gameState);
                        } else if (gameState.getAccess().equals(UserDataManager.PrivateGameStateAccess)) {
                            userData.setPrivateGameState(gameState);
                        }
                    }
                }
            }

            UserDataManager.getInstance(context).updateUserData(userData);
        } catch (JSONException e) {
            e.printStackTrace();
        }
    }

    /**
     * Method that processes the received that regarding the other users game states.
     *
     * @param gameStateData The received game state data in JSON {@link String}
     */
    public void processOtherUsersGameStateResponse(String gameStateData) {
        if (gameStateData != null) {
            try {
                JSONObject object = new JSONObject(gameStateData);

                String provider = null;
                if (object.has(Provider)) {
                    provider = object.getString(Provider);
                }

                JSONObject gameStates = null;
                if (object.has(UserDataManager.GameStates)) {
                    gameStates = object.getJSONObject(UserDataManager.GameStates);
                }

                if (provider != null && gameStates != null) {
                    GamedockSDK.getInstance(context).getUserDataCallbacks().otherUsersGameStateLoaded(provider, gameStates);
                }
            } catch (JSONException e) {
                e.printStackTrace();
            }
        } else {
            GamedockSDK.getInstance(context).getUserDataCallbacks().userDataError(ErrorCodes.GameStateOtherUsersServerError);
        }
    }

    /**
     * Method that saves the public game state to the disk and sends an event with the game state data that has been updated.
     * Throws an error if no custom user id and provider have been set.
     *
     * @param gameStateData  The game state data that has to be saved.
     * @param actionListener A native action listener that fires back the response from the server for the event operation.
     */
    public void setPublicGameState(String gameStateData, EventActionListener actionListener) {
        if (!FeaturesUtil.isFeatureEnabled(UserDataManager.FEATURE_NAME)) {
            return;
        }

        UserData userData = UserDataManager.getInstance(context).getUserData();

        if (userData != null && userData.getUserID() != null && userData.getUserID().length() > 0 && userData.getProvider() != null && userData.getProvider().length() > 0) {
            if (!isAllowedOperationByFrequency(userData, UserDataManager.PublicGameStateAccess)) {
                LoggingUtil.e("Operation triggered too frequently. Please reduce the amount of calls done for this operation: " + "setPublicGameState. " + intervalTimeString);
            }

            GameState gameState = new GameState(UserDataManager.PublicGameStateAccess);
            gameState.setData(gameStateData);
            gameState.setLastOperationTime(System.currentTimeMillis());

            userData.setPublicGameState(gameState);
            userData = UserDataManager.getInstance(context).updateUserDataVersion(userData);
            userData = UserDataManager.getInstance(context).updateUserDataMeta(userData);
            UserDataManager.getInstance(context).updateUserData(userData);

            try {
                UserDataEvent gameStateEvent = new UserDataEvent(context);
                gameStateEvent.setUpdateGameState();

                JSONArray gameStateJSONArray = new JSONArray();
                JSONObject gameStateJSON = new JSONObject();
                gameStateJSON.put(UserDataManager.Data, gameStateData);
                gameStateJSON.put(UserDataManager.GameStateAccess, UserDataManager.PublicGameStateAccess);
                gameStateJSONArray.put(gameStateJSON);

                gameStateEvent.addCustomData(UserDataManager.GameStates, gameStateJSONArray);

                gameStateEvent.addCustomData(UserDataManager.DeviceVersions, UserDataManager.getInstance(context).createUserDataVersionsJson(userData.getUserDataVersions()));

                GamedockSDK.getInstance(context).trackEvent(gameStateEvent, actionListener);
            } catch (JSONException e) {
                e.printStackTrace();
                GamedockSDK.getInstance(context).getUserDataCallbacks().userDataError(ErrorCodes.PublicGameStateOperation);
            }


        } else {
            GamedockSDK.getInstance(context).getUserDataCallbacks().userDataError(ErrorCodes.PublicGameStateOperation);
        }
    }

    /**
     * Method that retrieves the public game state from the local disk.
     *
     * @return Returns the Game State data {@link String} or {@code null} if the custom user id and provider have not been set.
     */
    public String getPublicGameState() {
        if (!FeaturesUtil.isFeatureEnabled(UserDataManager.FEATURE_NAME)) {
            return null;
        }

        if (UserDataManager.getInstance(context).getUserData() != null && UserDataManager.getInstance(context).getUserData().getPublicGameState() != null) {
            return UserDataManager.getInstance(context).getUserData().getPublicGameState().getData();
        } else {
            return null;
        }
    }

    /**
     * Method that saves the private game state to the disk and sends an event with the game state data that has been updated.
     *
     * @param gameStateData  The game state data that has to be saved.
     * @param actionListener A native action listener that fires back the response from the server for the event operation.
     */
    public void setPrivateGameState(String gameStateData, EventActionListener actionListener) {
        if (!FeaturesUtil.isFeatureEnabled(UserDataManager.FEATURE_NAME)) {
            return;
        }

        UserData userData = UserDataManager.getInstance(context).getUserData();

        if (userData == null) {
            userData = new UserData();
            userData.setWallet(new Wallet());
            userData.setInventory(new Inventory());

            UserDataManager.getInstance(context).updateUserData(userData);
        }

        if (!isAllowedOperationByFrequency(userData, UserDataManager.PrivateGameStateAccess)) {
            LoggingUtil.e("Operation triggered too frequently. Please reduce the amount of calls done for this operation: " + "setPrivateGameState. " + intervalTimeString);
        }

        GameState gameState = new GameState(UserDataManager.PrivateGameStateAccess);
        gameState.setData(gameStateData);
        gameState.setLastOperationTime(System.currentTimeMillis());

        userData.setPrivateGameState(gameState);
        userData = UserDataManager.getInstance(context).updateUserDataVersion(userData);
        userData = UserDataManager.getInstance(context).updateUserDataMeta(userData);
        UserDataManager.getInstance(context).updateUserData(userData);

        try {
            UserDataEvent gameStateEvent = new UserDataEvent(context);
            gameStateEvent.setUpdateGameState();

            JSONArray gameStateJSONArray = new JSONArray();
            JSONObject gameStateJSON = new JSONObject();
            gameStateJSON.put(UserDataManager.Data, gameStateData);
            gameStateJSON.put(UserDataManager.GameStateAccess, UserDataManager.PrivateGameStateAccess);
            gameStateJSONArray.put(gameStateJSON);

            gameStateEvent.addCustomData(UserDataManager.GameStates, gameStateJSONArray);

            gameStateEvent.addCustomData(UserDataManager.DeviceVersions, UserDataManager.getInstance(context).createUserDataVersionsJson(userData.getUserDataVersions()));

            GamedockSDK.getInstance(context).trackEvent(gameStateEvent, actionListener);
        } catch (JSONException e) {
            e.printStackTrace();
        }
    }

    /**
     * Method that retrieves the private game state from the local disk.
     *
     * @return Returns the Game State data {@link String} or {@code null} if no data has been saved.
     */
    public String getPrivateGameState() {
        if (!FeaturesUtil.isFeatureEnabled(UserDataManager.FEATURE_NAME)) {
            return null;
        }

        if (UserDataManager.getInstance(context).getUserData() == null) {
            UserData userData = new UserData();
            userData.setWallet(new Wallet());
            userData.setInventory(new Inventory());

            UserDataManager.getInstance(context).updateUserData(userData);
        }
        if (UserDataManager.getInstance(context).getUserData().getPrivateGameState() != null) {
            return UserDataManager.getInstance(context).getUserData().getPrivateGameState().getData();
        } else {
            return null;
        }
    }

    private boolean isAllowedOperationByFrequency(UserData userData, String access) {
        long lastOperationTime = 0;

        if (access.equals(UserDataManager.PublicGameStateAccess) && userData.getPublicGameState() != null) {
            lastOperationTime = userData.getPublicGameState().getLastOperationTime();
        } else if (access.equals(UserDataManager.PrivateGameStateAccess) && userData.getPrivateGameState() != null) {
            lastOperationTime = userData.getPrivateGameState().getLastOperationTime();
        }

        long currentOperationTime = System.currentTimeMillis();

        return (currentOperationTime - lastOperationTime) > frequencyLimit;
    }
}
