package io.gamedock.sdk.userdata;

import android.content.Context;
import android.os.Build;
import android.os.Handler;
import android.os.Looper;

import androidx.annotation.NonNull;

import com.google.gson.Gson;
import com.google.gson.JsonArray;
import com.google.gson.JsonObject;
import com.google.gson.JsonParser;

import org.json.JSONArray;
import org.json.JSONException;
import org.json.JSONObject;

import java.util.ArrayList;
import java.util.Calendar;
import java.util.TimeZone;

import io.gamedock.sdk.GamedockSDK;
import io.gamedock.sdk.R;
import io.gamedock.sdk.events.internal.UserDataEvent;
import io.gamedock.sdk.gamedata.GamedockGameDataManager;
import io.gamedock.sdk.models.gamedata.GamedockGameData;
import io.gamedock.sdk.models.gamedata.currencies.Currency;
import io.gamedock.sdk.models.gamedata.items.Item;
import io.gamedock.sdk.models.userdata.UserData;
import io.gamedock.sdk.models.userdata.UserDataMeta;
import io.gamedock.sdk.models.userdata.UserDataVersion;
import io.gamedock.sdk.models.userdata.gamestate.GameState;
import io.gamedock.sdk.models.userdata.inventory.Inventory;
import io.gamedock.sdk.models.userdata.inventory.PlayerItem;
import io.gamedock.sdk.models.userdata.inventory.UniquePlayerItem;
import io.gamedock.sdk.models.userdata.mission.ContainerProgress;
import io.gamedock.sdk.models.userdata.mission.MissionData;
import io.gamedock.sdk.models.userdata.mission.MissionProgress;
import io.gamedock.sdk.models.userdata.wallet.PlayerCurrency;
import io.gamedock.sdk.models.userdata.wallet.Wallet;
import io.gamedock.sdk.userdata.gamestate.GameStateManager;
import io.gamedock.sdk.userdata.missiondata.MissionDataManager;
import io.gamedock.sdk.userdata.playerdata.PlayerDataManager;
import io.gamedock.sdk.userdata.playerdata.functions.PlayerDataInitialisation;
import io.gamedock.sdk.utils.dialog.MaterialDialog;
import io.gamedock.sdk.utils.dialog.MaterialStyledDialog;
import io.gamedock.sdk.utils.dialog.internal.DialogAction;
import io.gamedock.sdk.utils.error.ErrorCodes;
import io.gamedock.sdk.utils.features.FeaturesUtil;
import io.gamedock.sdk.utils.logging.LoggingUtil;
import io.gamedock.sdk.utils.storage.StorageUtil;
import io.gamedock.sdk.utils.userid.UserIDGenerator;
import io.gamedock.sdk.web.WebViewActivity;

public class UserDataManager {

    private static final Object lock = new Object();

    public static final String FEATURE_NAME = "userData";

    private static volatile UserDataManager mInstance = null;
    private Context context;

    private PlayerDataManager playerDataManager;
    private GameStateManager gameStateManager;
    private MissionDataManager missionDataManager;

    private UserData userData;

    private ArrayList<UserDataVersion> remoteUserDataVersions;

    public static final String PlayerData = "playerData";
    public static final String Wallet = "wallet";
    public static final String Inventory = "inventory";
    public static final String Offset = "offset";

    public static final String GameState = "gameState";
    public static final String GameStates = "gameStates";
    public static final String GameStateAccess = "access";
    public static final String PrivateGameStateAccess = "private";
    public static final String PublicGameStateAccess = "public";

    public static final String MissionData = "missionData";

    public static final String DeviceVersions = "deviceVersions";
    public static final String Data = "data";

    public static final String MetaData = "metaData";

    public static final String MergeType = "mergeType";
    public static final String Local = "local";
    public static final String Remote = "remote";
    public static final String Merge = "merge";

    private UserDataManager(Context context) {
        this.context = context;
        playerDataManager = new PlayerDataManager(context, GamedockSDK.getInstance(context).getGson(), GamedockSDK.getInstance(context).getStorageUtil(), GamedockSDK.getFileAssetsUtil());
        gameStateManager = new GameStateManager(context);
        missionDataManager = new MissionDataManager(context);

        this.userData = getUserData();
    }

    public static UserDataManager getInstance(Context context) {
        if (mInstance == null) {
            synchronized (lock) {
                if (mInstance == null) {
                    mInstance = new UserDataManager(context);
                }
            }
        }
        return mInstance;
    }

    public PlayerDataManager getPlayerDataManager() {
        return playerDataManager;
    }

    public GameStateManager getGameStateManager() {
        return gameStateManager;
    }

    public MissionDataManager getMissionDataManager() {
        return missionDataManager;
    }

    /**
     * Method used to populate the user data at the start of the application.
     * Contains the information for Player Data and Game State.
     * Sends the current stored offset for the wallet and inventory.
     * Initialises the Player Data if there is no internet connection.
     */
    public void requestUserData() {
        if (!FeaturesUtil.isFeatureEnabled(FEATURE_NAME)) {
            return;
        }

        UserDataEvent userDataEvent = new UserDataEvent(context);
        userDataEvent.setRequestUserData();

        UserData userData = getUserData();

        if (userData == null) {
            LoggingUtil.e("Could not retrieve user data for requestUserData event!");
            return;
        }

        generatePlayerDataRequest(userDataEvent, userData);
        generateGameStateRequest(userDataEvent, userData);
        generateMissionDataRequest(userDataEvent);

        if (!GamedockSDK.getInstance(context).isCoppaEnabled()) {
            userDataEvent.addCustomData(UserDataManager.DeviceVersions, UserDataManager.getInstance(context).createUserDataVersionsJson(userData.getUserDataVersions()));
        } else {
            userDataEvent.addCustomData(UserDataManager.DeviceVersions, new JsonObject());
        }

        GamedockSDK.getInstance(context).trackEvent(userDataEvent, null);

        playerDataManager.initialisePlayerDataIfNoInternet(userData);
    }

    /**
     * Generates the event parameters for the Player Data part of the event.
     *
     * @param userDataEvent The event that needs in which the data has to be generated.
     * @param userData      The {@link UserData} object that contains the information.
     */
    private void generatePlayerDataRequest(UserDataEvent userDataEvent, UserData userData) {
        JsonObject playerDataJSON = new JsonObject();

        JsonObject walletJSON = new JsonObject();
        walletJSON.addProperty(Offset, userData.getWallet().getOffset());
        playerDataJSON.add(Wallet, walletJSON);

        JsonObject inventoryJSON = new JsonObject();
        inventoryJSON.addProperty(Offset, userData.getInventory().getOffset());
        playerDataJSON.add(Inventory, inventoryJSON);

        userDataEvent.addCustomData(PlayerData, playerDataJSON);
    }

    /**
     * Generates the event parameters for the Game State part of the event.
     * The method will by default request the "private" game state of the user.
     * If a "public" game state has been saved it will also request that.
     *
     * @param userDataEvent The event that needs in which the data has to be generated.
     * @param userData      The {@link UserData} object that contains the information.
     */
    private void generateGameStateRequest(UserDataEvent userDataEvent, UserData userData) {
        JsonObject gameStateJSON = new JsonObject();

        JsonArray gameStateArray = new JsonArray();
        gameStateArray.add(PrivateGameStateAccess);

        if (userData != null && userData.getUserID() != null && userData.getProvider() != null) {
            gameStateArray.add(PublicGameStateAccess);
        }

        gameStateJSON.add(GameStateAccess, gameStateArray);

        userDataEvent.addCustomData(GameState, gameStateJSON);
    }

    /**
     * Generates the event parameters for the Mission Data part of the event.
     *
     * @param userDataEvent The event that needs in which the data has to be generated.
     */
    private void generateMissionDataRequest(UserDataEvent userDataEvent) {
        JsonObject gameStateJSON = new JsonObject();

        userDataEvent.addCustomData(MissionData, gameStateJSON);
    }

    /**
     * Method that process the response in case of a "request" user data. This case happens if no merge conflicts have occurred on the backend.
     *
     * @param receivedWallet           The {@link Wallet} information received from Gamedock backend.
     * @param receivedInventory        The {@link Inventory} information received from Gamedock backend.
     * @param externalChange           Flag that notifies the SDK if any other device has updated on the backend. If true, then SDK will overwrite the local values of {@link Wallet} and {@link Inventory} with the values received from the backend.
     * @param gameState                The {@link GameState} information received from Gamedock backend. Contains both private and public data.
     * @param receivedUserDataVersions The list of device versions that have written for this user.
     * @param metaData                 The meta data containing the information of which device wrote last on the backend and at what time.
     */
    public void processRequestUserData(Wallet receivedWallet, Inventory receivedInventory, MissionData missionData, boolean externalChange, String gameState, ArrayList<UserDataVersion> receivedUserDataVersions, UserDataMeta metaData) {
        UserData userData = getUserData();

        UserDataVersion localUserDataVersion = getUserDataVersion(userData);
        UserDataVersion remoteUserDataVersion = null;

        playerDataManager.processPlayerDataInit(receivedWallet, receivedInventory, externalChange);

        gameStateManager.processMyGameStateResponse(gameState);

        missionDataManager.processMissionData(missionData);

        userData.getUserDataVersions().clear();

        if (receivedUserDataVersions != null) {
            for (int i = 0; i < receivedUserDataVersions.size(); i++) {
                if (!receivedUserDataVersions.get(i).getDeviceId().equals(localUserDataVersion.getDeviceId())) {
                    userData.getUserDataVersions().add(receivedUserDataVersions.get(i));
                } else {
                    remoteUserDataVersion = receivedUserDataVersions.get(i);
                }
            }
        }

        if (remoteUserDataVersion != null && (localUserDataVersion.getVersion() < remoteUserDataVersion.getVersion())) {
            userData.getUserDataVersions().add(remoteUserDataVersion);
        } else {
            userData.getUserDataVersions().add(localUserDataVersion);
        }

        userData = updateUserDataMeta(userData);
        if (userData.getUserDataMeta() != null) {
            if (metaData != null) {
                userData.getUserDataMeta().serverTime = metaData.serverTime;
            } else {
                userData.getUserDataMeta().serverTime = System.currentTimeMillis();
            }
        }

        updateUserData(userData);

        if (userData.updated) {
            userData.updated = false;
        }

        GamedockSDK.getInstance(context).getUserDataCallbacks().userDataAvailable(playerDataManager.getWallet(), playerDataManager.getInventory(), GamedockSDK.getInstance(context).getGson().toJson(missionDataManager.getUserAllContainerProgress(MissionDataManager.Status.NULL)), GamedockSDK.getInstance(context).getGson().toJson(missionDataManager.getUserAllMissionProgress(MissionDataManager.Status.NULL)));
    }

    /**
     * Method that process the response in case of merge conflict.
     * Fires a callback towards the developer presenting both the local and the remote state.
     *
     * @param receivedWallet    The {@link Wallet} information stored in Gamedock backend.
     * @param receivedInventory The {@link Inventory} information stored in Gamedock backend.
     * @param receivedGameState The {@link GameState} information stored in Gamedock backend.
     * @param receivedMetaData  The meta data containing the information of which device wrote last on the backend and at what time.
     */
    public void processMergeConflict(Wallet receivedWallet, Inventory receivedInventory, MissionData receivedMissionData, String receivedGameState, ArrayList<UserDataVersion> receivedUserDataVersions, UserDataMeta receivedMetaData) {
        JSONObject localData = new JSONObject();
        JSONObject remoteData = new JSONObject();

        Gson gson = playerDataManager.gson;
        UserData userData = getUserData();
        GamedockGameData gamedockGameData = GamedockGameDataManager.getInstance(context).getGameData();

        if (userData == null || gamedockGameData == null) {
            GamedockSDK.getInstance(context).getUserDataCallbacks().userDataError(ErrorCodes.LoadFailed);
            return;
        }

        //Processing Local User Data
        Wallet localWallet = userData.getWallet().BuildForJson();
        Inventory localInventory = userData.getInventory().BuildForJson();
        GameState localPrivateGameState = userData.getPrivateGameState();
        GameState localPublicGameState = userData.getPublicGameState();
        MissionData localMissionData = userData.getMissionData();
        UserDataMeta localUserDataMeta = userData.getUserDataMeta();

        try {
            //Local PlayerData
            JSONObject localPlayerData = new JSONObject();

            JSONObject walletJSON = localWallet.toJson(gson);
            localPlayerData.put(Wallet, walletJSON);

            JSONObject inventoryJSON = localInventory.toJson(gson);
            localPlayerData.put(Inventory, inventoryJSON);

            localData.put(PlayerData, localPlayerData);

            //Local GameState
            JSONObject privateGameStateJSON = new JSONObject();
            if (localPrivateGameState != null) {
                privateGameStateJSON.put(GameStateAccess, PrivateGameStateAccess);
                privateGameStateJSON.put(Data, localPrivateGameState.getData());
            }

            JSONObject publicGameStateJSON = new JSONObject();
            if (localPublicGameState != null) {
                publicGameStateJSON.put(GameStateAccess, PublicGameStateAccess);
                publicGameStateJSON.put(Data, localPublicGameState.getData());
            }

            JSONArray localGameStateArray = new JSONArray();
            localGameStateArray.put(privateGameStateJSON);
            localGameStateArray.put(publicGameStateJSON);

            JSONObject localGameState = new JSONObject();
            localGameState.put(GameStates, localGameStateArray);

            localData.put(GameState, localGameState);

            JSONObject localMissionDataJSON = new JSONObject(gson.toJson(localMissionData));
            localData.put(MissionData, localMissionDataJSON);

            //Local Meta
            localData.put(MetaData, new JSONObject(gson.toJson(localUserDataMeta)));
        } catch (JSONException e) {
            e.printStackTrace();
        }

        //Processing Remote User Data
        try {
            //Remote PlayerData
            JSONObject remotePlayerData = new JSONObject();

            for (PlayerCurrency receivedPlayerCurrency : receivedWallet.getCurrenciesMap().values()) {
                Currency currency = gamedockGameData.getCurrenciesMap().get(receivedPlayerCurrency.getId());
                receivedPlayerCurrency.setName(currency.getName());
                receivedPlayerCurrency.setType(currency.getType());
                receivedPlayerCurrency.setDisplayName(currency.getDisplayName());
                receivedPlayerCurrency.setDisplayDescription(currency.getDisplayDescription());
                receivedPlayerCurrency.setImageUrl(currency.getImageUrl());
                receivedPlayerCurrency.setInitialValue(currency.getInitialValue());
            }
            remotePlayerData.put(Wallet, receivedWallet.BuildForJson().toJson(gson));

            for (PlayerItem receivedPlayerItem : receivedInventory.getItemsMap().values()) {
                Item item = gamedockGameData.getItemsMap().get(receivedPlayerItem.getId());
                if (item != null) {
                    receivedPlayerItem.setName(item.getName());
                    receivedPlayerItem.setType(item.getType());
                    receivedPlayerItem.setDisplayName(item.getDisplayName());
                    receivedPlayerItem.setDisplayDescription(item.getDisplayDescription());
                    receivedPlayerItem.setImageUrl(item.getImageUrl());
                    receivedPlayerItem.setInitialValue(item.getInitialValue());
                    receivedPlayerItem.setGacha(item.isGacha());
                    receivedPlayerItem.setContent(item.getContent());
                    receivedPlayerItem.setAllowDuplicates(item.doesAllowDuplicates());
                    receivedPlayerItem.setShouldReroll(item.shouldReroll());
                    receivedPlayerItem.setDuplicateReward(item.getDuplicateReward());
                    receivedPlayerItem.setProperties(item.getProperties());
                    receivedPlayerItem.setLimit(item.getLimit());
                    receivedPlayerItem.setUnique(item.isUnique());
                }
            }

            for (UniquePlayerItem receivedUniquePlayerItem : receivedInventory.getUniqueItemsMap().values()) {
                Item item = gamedockGameData.getItemsMap().get(receivedUniquePlayerItem.getId());
                if (item != null) {
                    receivedUniquePlayerItem.setName(item.getName());
                    receivedUniquePlayerItem.setType(item.getType());
                    receivedUniquePlayerItem.setDisplayName(item.getDisplayName());
                    receivedUniquePlayerItem.setDisplayDescription(item.getDisplayDescription());
                    receivedUniquePlayerItem.setImageUrl(item.getImageUrl());
                    receivedUniquePlayerItem.setInitialValue(item.getInitialValue());
                    receivedUniquePlayerItem.setGacha(item.isGacha());
                    receivedUniquePlayerItem.setContent(item.getContent());
                    receivedUniquePlayerItem.setAllowDuplicates(item.doesAllowDuplicates());
                    receivedUniquePlayerItem.setShouldReroll(item.shouldReroll());
                    receivedUniquePlayerItem.setDuplicateReward(item.getDuplicateReward());
                    receivedUniquePlayerItem.setProperties(item.getProperties());
                    receivedUniquePlayerItem.setLimit(item.getLimit());
                    receivedUniquePlayerItem.setUnique(item.isUnique());
                }
            }

            remotePlayerData.put(Inventory, receivedInventory.BuildForJson().toJson(gson));

            remoteData.put(PlayerData, remotePlayerData);

            //Remote GameState
            remoteData.put(GameState, new JSONObject(receivedGameState));

            //Remote MissionData
            for (ContainerProgress containerProgress : receivedMissionData.getContainerProgress()) {
                containerProgress.setStatus(containerProgress.getStatus().replace(" ", "_"));
            }

            for (MissionProgress missionProgress : receivedMissionData.getMissionProgress()) {
                missionProgress.setStatus(missionProgress.getStatus().replace(" ", "_"));
            }
            remoteData.put(MissionData, new JSONObject(playerDataManager.gson.toJson(receivedMissionData)));

            //Remote Meta
            remoteData.put(MetaData, new JSONObject(gson.toJson(receivedMetaData)));

            //Remote Device Versions
            remoteUserDataVersions = receivedUserDataVersions;
        } catch (NullPointerException | JSONException e) {
            e.printStackTrace();
        }

        GamedockSDK.getInstance(context).getUserDataCallbacks().userDataMergeConflict(localData, remoteData);
    }

    /**
     * Method that process the response when a sync error occurred on the backend.
     * Fires a callback towards the developer to notify him.
     */
    public void processSyncError() {
        //Closing any web views that might be open when a sync error occurs
        if (WebViewActivity.getActivity() != null) {
            WebViewActivity.getActivity().finish();
        }

        GamedockSDK.getInstance(context).getUserDataCallbacks().userDataSyncError();
    }

    /**
     * Method that process the response when a lock error occurred on the backend.
     * Fires a callback towards the developer to notify him.
     */
    public void processLockError() {
        GamedockSDK.getInstance(context).getUserDataCallbacks().userDataLockError();
    }

    /**
     * Method that process the response when an event was dropped on the backend.
     */
    public void processDroppedResponse(String message) {
        LoggingUtil.e("Update event dropped. Message: " + message);
    }

    /**
     * Method used to merge the data based on the selection made by the user/developer.
     * Sends an event to the backend to notify which data was merged.
     *
     * @param mergeData The merged data in a JSON String format
     * @param mergeType The merge type. Can be "local", "remote" or "merge".
     */
    public void mergeUserData(String mergeData, String mergeType) {
        if (!FeaturesUtil.isFeatureEnabled(FEATURE_NAME)) {
            return;
        }

        UserData userData = getUserData();
        UserDataEvent userDataEvent = new UserDataEvent(context);

        try {
            if (userData == null) {
                GamedockSDK.getInstance(context).getUserDataCallbacks().userDataMergeFailed(new JSONObject(mergeData), mergeType);
                return;
            }

            userDataEvent.setMergeUserData();

            //Prepare and clean merge data for processing and sending
            JSONObject mergeDataJSON = new JSONObject(mergeData);

            JSONObject mergePlayerDataJSON = mergeDataJSON.getJSONObject(PlayerData);
            JSONObject mergeGameStateJSON = mergeDataJSON.getJSONObject(GameState);
            JSONObject mergeMissionDataJSON = mergeDataJSON.getJSONObject(MissionData);

            JSONObject mergeWallet = mergePlayerDataJSON.getJSONObject(Wallet);
            JSONArray mergeCurrencies = new JSONArray();

            for (int i = 0; i < mergeWallet.getJSONArray(PlayerDataManager.Currencies).length(); i++) {
                JSONObject currency = new JSONObject();
                currency.put(PlayerDataManager.Id, mergeWallet.getJSONArray(PlayerDataManager.Currencies).getJSONObject(i).getInt(PlayerDataManager.Id));
                if (mergeWallet.getJSONArray(PlayerDataManager.Currencies).getJSONObject(i).has(PlayerDataManager.Name)) {
                    currency.put(PlayerDataManager.Name, mergeWallet.getJSONArray(PlayerDataManager.Currencies).getJSONObject(i).getString(PlayerDataManager.Name));
                }
                currency.put(PlayerDataManager.Type, mergeWallet.getJSONArray(PlayerDataManager.Currencies).getJSONObject(i).getInt(PlayerDataManager.Type));
                currency.put(PlayerDataManager.CurrentBalance, mergeWallet.getJSONArray(PlayerDataManager.Currencies).getJSONObject(i).getInt(PlayerDataManager.CurrentBalance));
                currency.put(PlayerDataManager.Delta, mergeWallet.getJSONArray(PlayerDataManager.Currencies).getJSONObject(i).getInt(PlayerDataManager.Delta));
                mergeCurrencies.put(currency);
            }
            mergeWallet.remove(PlayerDataManager.Currencies);
            mergeWallet.put(PlayerDataManager.Currencies, mergeCurrencies);

            JSONObject mergeInventory = mergePlayerDataJSON.getJSONObject(Inventory);
            JSONArray mergeItems = new JSONArray();
            JSONArray mergeUniqueItems = new JSONArray();

            for (int i = 0; i < mergeInventory.getJSONArray(PlayerDataManager.Items).length(); i++) {
                JSONObject item = new JSONObject();
                item.put(PlayerDataManager.Id, mergeInventory.getJSONArray(PlayerDataManager.Items).getJSONObject(i).getInt(PlayerDataManager.Id));
                if (mergeInventory.getJSONArray(PlayerDataManager.Items).getJSONObject(i).has(PlayerDataManager.Name)) {
                    item.put(PlayerDataManager.Name, mergeInventory.getJSONArray(PlayerDataManager.Items).getJSONObject(i).getString(PlayerDataManager.Name));
                }
                item.put(PlayerDataManager.Type, mergeInventory.getJSONArray(PlayerDataManager.Items).getJSONObject(i).getInt(PlayerDataManager.Type));
                item.put(PlayerDataManager.Amount, mergeInventory.getJSONArray(PlayerDataManager.Items).getJSONObject(i).getInt(PlayerDataManager.Amount));
                item.put(PlayerDataManager.Delta, mergeInventory.getJSONArray(PlayerDataManager.Items).getJSONObject(i).getInt(PlayerDataManager.Delta));
                item.put(PlayerDataManager.Value, mergeInventory.getJSONArray(PlayerDataManager.Items).getJSONObject(i).getInt(PlayerDataManager.Value));
                mergeItems.put(item);
            }
            mergeInventory.remove(PlayerDataManager.Items);
            mergeInventory.put(PlayerDataManager.Items, mergeItems);

            for (int i = 0; i < mergeInventory.getJSONArray(PlayerDataManager.UniqueItems).length(); i++) {
                JSONObject uniqueItem = new JSONObject();
                uniqueItem.put(PlayerDataManager.UniqueId, mergeInventory.getJSONArray(PlayerDataManager.UniqueItems).getJSONObject(i).getString(PlayerDataManager.UniqueId));
                uniqueItem.put(PlayerDataManager.Id, mergeInventory.getJSONArray(PlayerDataManager.UniqueItems).getJSONObject(i).getInt(PlayerDataManager.Id));
                if (mergeInventory.getJSONArray(PlayerDataManager.UniqueItems).getJSONObject(i).has(PlayerDataManager.Name)) {
                    uniqueItem.put(PlayerDataManager.Name, mergeInventory.getJSONArray(PlayerDataManager.UniqueItems).getJSONObject(i).getString(PlayerDataManager.Name));
                }
                uniqueItem.put(PlayerDataManager.Type, mergeInventory.getJSONArray(PlayerDataManager.UniqueItems).getJSONObject(i).getInt(PlayerDataManager.Type));
                uniqueItem.put(PlayerDataManager.Amount, mergeInventory.getJSONArray(PlayerDataManager.UniqueItems).getJSONObject(i).getInt(PlayerDataManager.Amount));
                uniqueItem.put(PlayerDataManager.Delta, mergeInventory.getJSONArray(PlayerDataManager.UniqueItems).getJSONObject(i).getInt(PlayerDataManager.Delta));
                uniqueItem.put(PlayerDataManager.Status, mergeInventory.getJSONArray(PlayerDataManager.UniqueItems).getJSONObject(i).getString(PlayerDataManager.Status));
                uniqueItem.put(PlayerDataManager.UniqueProperties, mergeInventory.getJSONArray(PlayerDataManager.UniqueItems).getJSONObject(i).getJSONObject(PlayerDataManager.UniqueProperties));
                mergeUniqueItems.put(uniqueItem);
            }

            mergeInventory.remove(PlayerDataManager.UniqueItems);
            mergeInventory.put(PlayerDataManager.UniqueItems, mergeUniqueItems);

            //Save the received merge if the merge type is remote or merge
            if (!mergeType.equals(Local)) {
                Wallet wallet = playerDataManager.gson.fromJson(mergeWallet.toString(), Wallet.class);
                Inventory inventory = playerDataManager.gson.fromJson(mergeInventory.toString(), Inventory.class);

                userData.setWallet(wallet);

                GamedockGameData gamedockGameData = GamedockGameDataManager.getInstance(context).getGameData();
                for (PlayerItem playerItem : inventory.getItems()) {
                    Item item = gamedockGameData.getItemsMap().get(playerItem.getId());
                    if (item != null) {
                        if (item.isGacha()) {
                            playerItem.setGacha(true);
                        }
                    }
                }
                for (UniquePlayerItem uniquePlayerItem : inventory.getUniqueItems()) {
                    Item item = gamedockGameData.getItemsMap().get(uniquePlayerItem.getId());
                    if (item != null) {
                        if (item.isGacha()) {
                            uniquePlayerItem.setGacha(true);
                            uniquePlayerItem.setUnique(true);
                        }
                    }
                }

                userData.setInventory(inventory);

                userData.Build();

                JSONArray mergeGameStatesArray = mergeGameStateJSON.getJSONArray(GameStates);

                for (int i = 0; i < mergeGameStatesArray.length(); i++) {
                    GameState gameState = playerDataManager.gson.fromJson(mergeGameStatesArray.getJSONObject(i).toString(), GameState.class);

                    if (gameState.getAccess().equals(PrivateGameStateAccess)) {
                        userData.setPrivateGameState(gameState);
                    } else if (gameState.getAccess().equals(PublicGameStateAccess)) {
                        userData.setPublicGameState(gameState);
                    }
                }

                JSONArray mergeContainerProgressArray = mergeMissionDataJSON.getJSONArray("containerProgress");
                userData.getMissionData().getContainerProgress().clear();
                for (int i = 0; i < mergeContainerProgressArray.length(); i++) {
                    ContainerProgress containerProgress = playerDataManager.gson.fromJson(mergeContainerProgressArray.getJSONObject(i).toString(), ContainerProgress.class);

                    userData.getMissionData().getContainerProgress().add(containerProgress);
                }

                JSONArray mergeMissionProgressArray = mergeMissionDataJSON.getJSONArray("missionProgress");
                userData.getMissionData().getMissionProgress().clear();
                JsonParser parser = new JsonParser();
                for (int i = 0; i < mergeMissionProgressArray.length(); i++) {
                    MissionProgress missionProgress = playerDataManager.gson.fromJson(mergeMissionProgressArray.getJSONObject(i).toString(), MissionProgress.class);
                    JsonObject progress = (JsonObject) parser.parse(mergeMissionProgressArray.getJSONObject(i).getJSONObject("progress").toString());
                    missionProgress.setProgress(progress);

                    userData.getMissionData().getMissionProgress().add(missionProgress);
                }

                userData = updateUserDataMeta(userData);
                updateUserData(userData);
            }

            userDataEvent.addCustomData(PlayerData, mergePlayerDataJSON);
            userDataEvent.addCustomData(GameState, mergeGameStateJSON);

            io.gamedock.sdk.models.userdata.mission.MissionData mergeMissionData = playerDataManager.gson.fromJson(mergeMissionDataJSON.toString(), io.gamedock.sdk.models.userdata.mission.MissionData.class);
            for (ContainerProgress containerProgress : mergeMissionData.getContainerProgress()) {
                containerProgress.setStatus(containerProgress.getStatus().replace("_", " "));
            }

            for (MissionProgress missionProgress : mergeMissionData.getMissionProgress()) {
                missionProgress.setStatus(missionProgress.getStatus().replace("_", " "));
            }

            userDataEvent.addCustomData(MissionData, new JSONObject(playerDataManager.gson.toJson(mergeMissionData)));

            //Local and Remote Device Versions
            JsonObject deviceVersionsJSON = new JsonObject();
            deviceVersionsJSON.add(Local, createUserDataVersionsJson(userData.getUserDataVersions()));

            if (remoteUserDataVersions != null) {
                deviceVersionsJSON.add(Remote, createUserDataVersionsJson(remoteUserDataVersions));
            } else {
                deviceVersionsJSON.add(Remote, createUserDataVersionsJson(new ArrayList<UserDataVersion>()));
            }

            userDataEvent.addCustomData(DeviceVersions, deviceVersionsJSON);
        } catch (JSONException e) {
            e.printStackTrace();
        }

        userDataEvent.addCustomData(MergeType, mergeType);

        GamedockSDK.getInstance(context).trackEvent(userDataEvent, null);
    }

    /**
     * Method used to process the response after a merge has been done.
     * Updates the local device versions with the ones received from the backend.
     *
     * @param userDataVersions The received device versions from Gamedock backend.
     */
    public void processMergeUserData(ArrayList<UserDataVersion> userDataVersions) {
        UserData userData = getUserData();

        if (userData == null) {
            GamedockSDK.getInstance(context).getUserDataCallbacks().userDataError(ErrorCodes.LoadFailed);
            return;
        }

        userData.setUserDataVersions(userDataVersions);
        updateUserData(userData);

        remoteUserDataVersions = null;

        GamedockSDK.getInstance(context).getUserDataCallbacks().userDataMergeSuccessful(playerDataManager.getWallet(), playerDataManager.getInventory(), GamedockSDK.getInstance(context).getGson().toJson(missionDataManager.getUserAllContainerProgress(MissionDataManager.Status.NULL)), GamedockSDK.getInstance(context).getGson().toJson(missionDataManager.getUserAllMissionProgress(MissionDataManager.Status.NULL)));
    }

    /**
     * Method that retrieves the user profile, containing Wallet, Inventory and Game State, from either the shared prefs or the local JSON file.
     * If the {@link UserData} has not been created yet, the values for the {@link Wallet} and {@link Inventory} will be initialised with values described in the {@link GamedockGameData}.
     *
     * @return Returns the {@link UserData} object associated with the current player.
     */
    public synchronized UserData getUserData() {
        if (userData != null) {
            return userData;
        }

        String userProfileString = playerDataManager.storageUtil.getString(StorageUtil.Keys.SpilUserData, null);

        UserData userData = null;
        if (userProfileString != null) {
            userData = PlayerDataInitialisation.initUserDataFromPrefs(context, playerDataManager.gson, userProfileString);
        } else {
            userData = PlayerDataInitialisation.initUserDataFromAssets(context, playerDataManager.gson, playerDataManager.fileAssetsUtil);
            playerDataManager.storageUtil.putString(StorageUtil.Keys.SpilUserData, playerDataManager.gson.toJson(userData));
        }
        this.userData = userData;
        return userData;
    }

    /**
     * Method used to display a dialog that tells the user that a sync error has occurred and that the merging process needs to be initialised.
     *
     * @param title       The title of the dialog.
     * @param message     The message of the dialog.
     * @param mergeButton The button text.
     */
    public void showSyncErrorDialog(final String title, final String message, final String mergeButton) {
        if (!FeaturesUtil.isFeatureEnabled(FEATURE_NAME)) {
            return;
        }

        final int dialogHeader;
        int resourceId = context.getResources().getIdentifier("permission_header_custom", "drawable", context.getPackageName());

        if (resourceId != 0) {
            dialogHeader = resourceId;
        } else {
            dialogHeader = R.drawable.permission_header;
        }

        Handler handler = new Handler(Looper.getMainLooper());
        handler.post(new Runnable() {
            @Override
            public void run() {
                MaterialStyledDialog.Builder builder = new MaterialStyledDialog.Builder(context)
                        .setTitle(title)
                        .setDescription(message)
                        .setHeaderDrawable(dialogHeader)
                        .autoDismiss(false)
                        .withDialogAnimation(true)
                        .setPositiveText(mergeButton)
                        .onPositive(new MaterialDialog.SingleButtonCallback() {
                            @Override
                            public void onClick(@NonNull MaterialDialog materialDialog, @NonNull DialogAction dialogAction) {
                                requestUserData();
                                materialDialog.dismiss();
                            }
                        });

                builder.show();
            }
        });
    }

    /**
     * Method used to display a dialog that tells the user that the merge has failed and that the merging process needs to be restarted.
     *
     * @param title       The title of the dialog.
     * @param message     The message of the dialog.
     * @param retryButton The text for the retry button.
     * @param mergeData   The merge data that has to be resend.
     * @param mergeType   The merge type that has to be resend.
     */
    public void showMergeFailedDialog(final String title, final String message, final String retryButton, final String mergeData, final String mergeType) {
        if (!FeaturesUtil.isFeatureEnabled(FEATURE_NAME)) {
            return;
        }

        final int dialogHeader;
        int resourceId = context.getResources().getIdentifier("permission_header_custom", "drawable", context.getPackageName());

        if (resourceId != 0) {
            dialogHeader = resourceId;
        } else {
            dialogHeader = R.drawable.permission_header;
        }

        Handler handler = new Handler(Looper.getMainLooper());
        handler.post(new Runnable() {
            @Override
            public void run() {
                MaterialStyledDialog.Builder builder = new MaterialStyledDialog.Builder(context)
                        .setTitle(title)
                        .setDescription(message)
                        .setHeaderDrawable(dialogHeader)
                        .autoDismiss(false)
                        .withDialogAnimation(true)
                        .setPositiveText(retryButton)
                        .onPositive(new MaterialDialog.SingleButtonCallback() {
                            @Override
                            public void onClick(@NonNull MaterialDialog materialDialog, @NonNull DialogAction dialogAction) {
                                mergeUserData(mergeData, mergeType);
                                materialDialog.dismiss();
                            }
                        });

                builder.show();
            }
        });
    }

    /**
     * Method used to display a dialog in which the user can select which state he wants to be the new state (local, remote or merge).
     *
     * @param title        The title of the dialog.
     * @param message      The message of the dialog.
     * @param localButton  The text of the local button.
     * @param remoteButton The text of the remote button.
     * @param mergeButton  The text of the merge button. Can be null or empty and if that is the case the button will not be shown.
     */
    public void showMergeConflictDialog(final String title, final String message, final String localButton, final String remoteButton, final String mergeButton) {
        if (!FeaturesUtil.isFeatureEnabled(FEATURE_NAME)) {
            return;
        }

        final int dialogHeader;
        int resourceId = context.getResources().getIdentifier("permission_header_custom", "drawable", context.getPackageName());

        if (resourceId != 0) {
            dialogHeader = resourceId;
        } else {
            dialogHeader = R.drawable.permission_header;
        }

        Handler handler = new Handler(Looper.getMainLooper());
        handler.post(new Runnable() {
            @Override
            public void run() {
                MaterialStyledDialog.Builder builder = new MaterialStyledDialog.Builder(context)
                        .setTitle(title)
                        .setDescription(message)
                        .setHeaderDrawable(dialogHeader)
                        .autoDismiss(false)
                        .withDialogAnimation(true)
                        .setPositiveText(localButton)
                        .onPositive(new MaterialDialog.SingleButtonCallback() {
                            @Override
                            public void onClick(@NonNull MaterialDialog materialDialog, @NonNull DialogAction dialogAction) {
                                GamedockSDK.getInstance(context).getUserDataCallbacks().userDataHandleMerge(Local);
                                materialDialog.dismiss();
                            }
                        })
                        .setNegativeText(remoteButton)
                        .onNegative(new MaterialDialog.SingleButtonCallback() {
                            @Override
                            public void onClick(@NonNull MaterialDialog materialDialog, @NonNull DialogAction which) {
                                GamedockSDK.getInstance(context).getUserDataCallbacks().userDataHandleMerge(Remote);
                                materialDialog.dismiss();
                            }
                        });


                if (mergeButton != null && !mergeButton.equals("")) {
                    builder.setNeutralText(mergeButton)
                            .onNeutral(new MaterialDialog.SingleButtonCallback() {
                                @Override
                                public void onClick(@NonNull MaterialDialog materialDialog, @NonNull DialogAction which) {
                                    GamedockSDK.getInstance(context).getUserDataCallbacks().userDataHandleMerge(Merge);
                                    materialDialog.dismiss();
                                }
                            });
                }
                builder.show();
            }
        });
    }


    /**
     * Method that updates the user profile, containing Wallet and Inventory information into the shared preferences.
     *
     * @param userData The {@link UserData} object that has to be updated.
     */
    public synchronized void updateUserData(UserData userData) {
        if (userData != null) {
            final UserData cloneUserData = userData.clone();
            playerDataManager.storageUtil.putString(StorageUtil.Keys.SpilUserData, new Gson().toJson(cloneUserData));
            this.userData = userData;
        }
    }

    /**
     * Method that returns the {@link UserDataVersion} specific to the current device.
     *
     * @param userData The local {@link UserData} information.
     * @return The device {@link UserDataVersion}.
     */
    private UserDataVersion getUserDataVersion(UserData userData) {
        String deviceId = UserIDGenerator.getUniqueDeviceId(context);

        for (int i = 0; i < userData.getUserDataVersions().size(); i++) {
            if (userData.getUserDataVersions().get(i).getDeviceId().equals(deviceId)) {
                return userData.getUserDataVersions().get(i);
            }
        }

        UserDataVersion userDataVersion = new UserDataVersion();
        userDataVersion.setDeviceId(deviceId);
        userDataVersion.setVersion(0);
        return userDataVersion;
    }

    /**
     * Method used to update the data version of the current device.
     *
     * @param userData The local {@link UserData} information.
     * @return The updated local {@link UserData} information.
     */
    public synchronized UserData updateUserDataVersion(UserData userData) {
        String deviceId = UserIDGenerator.getUniqueDeviceId(context);

        boolean check = false;
        for (UserDataVersion userDataVersion : userData.getUserDataVersions()) {
            if (userDataVersion.getDeviceId().equals(deviceId)) {
                check = true;
            }
        }

        if (!check) {
            UserDataVersion userDataVersion = new UserDataVersion();
            userDataVersion.setDeviceId(deviceId);
            userDataVersion.setVersion(0);

            userData.getUserDataVersions().add(userDataVersion);
        }

        for (int i = 0; i < userData.getUserDataVersions().size(); i++) {
            if (userData.getUserDataVersions().get(i).getDeviceId().equals(deviceId)) {
                int version = userData.getUserDataVersions().get(i).getVersion();
                userData.getUserDataVersions().get(i).setVersion(version + 1);
            }
        }

        return userData;
    }

    /**
     * Helper method used to generate a JsonObject based on the stored {@link UserDataVersion} list.
     *
     * @param userDataVersions The {@link UserDataVersion} list.
     * @return The JSON representation of the list.
     */
    public JsonObject createUserDataVersionsJson(ArrayList<UserDataVersion> userDataVersions) {
        JsonObject deviceVersionsJSON = new JsonObject();
        for (UserDataVersion userDataVersion : userDataVersions) {
            deviceVersionsJSON.addProperty(userDataVersion.getDeviceId(), userDataVersion.getVersion());
        }

        return deviceVersionsJSON;
    }

    /**
     * Method used to update the local stored {@link UserDataMeta}.
     *
     * @param userData The local {@link UserData} information.
     * @return The updated local {@link UserData} information.
     */
    public UserData updateUserDataMeta(UserData userData) {
        if (userData.getUserDataMeta() == null) {
            userData.setUserDataMeta(new UserDataMeta());
        }

        if (userData.getUserDataMeta().deviceModel == null) {
            userData.getUserDataMeta().deviceModel = Build.MANUFACTURER + " " + Build.MODEL;
        }

        if (userData.getUserDataMeta().timezoneOffset == 0) {
            Calendar cal = Calendar.getInstance(TimeZone.getDefault());
            userData.getUserDataMeta().timezoneOffset = (cal.get(Calendar.ZONE_OFFSET) + cal.get(Calendar.DST_OFFSET)) / 60000;
        }

        if (userData.getUserDataMeta().appVersion == null) {
            userData.getUserDataMeta().appVersion = GamedockSDK.getInstance(context).appVersion;
        }

        userData.getUserDataMeta().clientTime = System.currentTimeMillis();

        return userData;
    }

    public void resetUserData() {
        userData = null;

        playerDataManager.resetPlayerData(false);

        mInstance = null;
    }
}
