package io.gamedock.sdk.pushnotifications;

import android.app.Activity;
import android.app.Notification;
import android.app.NotificationChannel;
import android.app.PendingIntent;
import android.content.Context;
import android.content.Intent;
import android.content.SharedPreferences;
import android.content.pm.ApplicationInfo;
import android.content.pm.PackageManager;
import android.content.res.Resources;
import android.graphics.Bitmap;
import android.graphics.BitmapFactory;
import android.graphics.Canvas;
import android.graphics.drawable.BitmapDrawable;
import android.graphics.drawable.Drawable;
import android.net.Uri;
import android.os.Build;
import android.os.Handler;
import android.os.Looper;
import android.widget.RemoteViews;

import androidx.annotation.NonNull;
import androidx.core.app.NotificationCompat;
import androidx.core.app.TaskStackBuilder;

import com.evernote.android.job.Job;
import com.evernote.android.job.JobManager;
import com.evernote.android.job.JobRequest;
import com.evernote.android.job.util.support.PersistableBundleCompat;
import com.google.gson.reflect.TypeToken;

import org.json.JSONObject;

import java.lang.reflect.Type;
import java.util.ArrayList;
import java.util.Date;
import java.util.Random;
import java.util.UUID;

import io.gamedock.sdk.GamedockSDK;
import io.gamedock.sdk.R;
import io.gamedock.sdk.events.Event;
import io.gamedock.sdk.models.image.ImageContext;
import io.gamedock.sdk.models.notification.GamedockNotification;
import io.gamedock.sdk.utils.error.ErrorCodes;
import io.gamedock.sdk.utils.images.ImageLoadCallbacks;
import io.gamedock.sdk.utils.images.ImageLoaderUtil;
import io.gamedock.sdk.utils.images.OnImageLoadListener;
import io.gamedock.sdk.utils.logging.LoggingUtil;
import io.gamedock.sdk.utils.storage.StorageUtil;

/**
 * Manager class that handles all incoming push notifications.
 * Push notifications can also be enabled and disabled here.
 * Additional data received with a notification is stored in shared preferences.
 */
public class NotificationManager {

    public static String TAG = "notificationManager";
    public static final String NotificationChannel = "gamedock_notification_channel_01";

    public static void sendNotificationEvent(Context context, String uniqueNotificationID, String name, boolean didLaunchApp, String type) {
        Event event = new Event(context);
        event.setName(name);
        event.addCustomData("uniqueNotificationId", uniqueNotificationID);
        event.addCustomData("type", type);

        if (name.equals("notificationOpened")) {
            event.addCustomData("didLaunchApp", didLaunchApp);
        }

        GamedockSDK.getInstance(context).trackEvent(event, null);
    }

    /**
     * Method that enables the showing of push notifications.
     * Flag is saved in the {@link SharedPreferences}.
     *
     * @param context The activity context.
     */
    public static void enableNotifications(Context context) {
        GamedockSDK.getInstance(context).getStorageUtil().putBoolean(StorageUtil.Keys.NotificationStatus, true);
    }

    /**
     * Method that disables the showing of push notifications.
     * Flag is saved in the {@link SharedPreferences}.
     *
     * @param context The activity context.
     */
    public static void disableNotifications(Context context) {
        GamedockSDK.getInstance(context).getStorageUtil().putBoolean(StorageUtil.Keys.NotificationStatus, false);
    }

    /**
     * Method that checks if a notification should be shown or not.
     *
     * @param context The activity context.
     * @return Returns {@code true} if a notification can be shown or {@code false} otherwise.
     */
    public static boolean checkNotificationStatus(Context context) {
        return GamedockSDK.getInstance(context).getStorageUtil().getBoolean(StorageUtil.Keys.NotificationStatus, false);
    }

    /**
     * Method that handles the processing of a push notification that has been received from Gamedock backend through GCM.
     * Contains logic for displaying a push notification with or without reward.
     *
     * @param context    The activity context.
     * @param jsonObject The data that is contained in the push notification.
     */
    public static void processNotificationForShowing(Context context, JSONObject jsonObject) {
        try {
            if (!GamedockSDK.getInstance(context).isAppRunning()) {
                String title = "";
                if (jsonObject.has("title")) {
                    title = jsonObject.getString("title");
                }

                String message = "";
                if (jsonObject.has("message")) {
                    message = jsonObject.getString("message");
                }

                String uniqueNotificationID = "";
                if (jsonObject.has("uniqueNotificationID")) {
                    uniqueNotificationID = jsonObject.getString("uniqueNotificationID");
                }

                String imageUrl = "";
                if(jsonObject.has("imageUrl")) {
                    imageUrl = jsonObject.getString("imageUrl");
                }

                String type = "";
                if(jsonObject.has("type")) {
                    type = jsonObject.getString("type");
                }

                String color = "";
                if(jsonObject.has("color")) {
                    color = jsonObject.getString("color");
                }

                GamedockNotification gamedockNotification = new GamedockNotification(title, message, uniqueNotificationID, imageUrl, type, color);

                String action;
                if (jsonObject.has("action")) {
                    action = jsonObject.getString("action");

                    switch (action.toLowerCase().trim()) {
                        case "reward":
                            JSONObject notificationData;
                            if (jsonObject.has("notificationData")) {
                                notificationData = jsonObject.getJSONObject("notificationData");
                                gamedockNotification.notificationData = notificationData.toString();
                            }
                            break;
                        case "crosspromotion":
                            String destination;
                            if (jsonObject.has("destination")) {
                                destination = jsonObject.getString("destination");
                                gamedockNotification.destination = destination;
                            }
                            break;
                    }
                }

                showNotification(context, gamedockNotification);

                sendNotificationEvent(context, uniqueNotificationID, "notificationReceived", false, "remote");

                GamedockSDK.resetContext();
            } else {
                LoggingUtil.d("Ignored showing notification as the game is already running");

                String uniqueNotificationID = null;
                if (jsonObject.has("uniqueNotificationID")) {
                    uniqueNotificationID = jsonObject.getString("uniqueNotificationID");
                    sendNotificationEvent(context, uniqueNotificationID, "notificationReceived", false, "remote");
                }

                JSONObject notificationData;
                if (jsonObject.has("notificationData")) {
                    notificationData = jsonObject.getJSONObject("notificationData");
                    GamedockSDK.getInstance(context).getRewardCallbacks().sendNotificationRewards(notificationData.toString());
                }
                sendNotificationEvent(context, uniqueNotificationID, "notificationOpened", false, "remote");
            }
        } catch (Exception e) {
            e.printStackTrace();
        }
    }

    /**
     * Method that handles the processing of a push notification that doesn't need to be displayed.
     *
     * @param context    The activity context.
     * @param jsonObject The data that is contained in the push notification.
     */
    public static void processNotificationForHidden(Context context, JSONObject jsonObject) {
        try {
            String uniqueNotificationID;
            if (jsonObject.has("uniqueNotificationID")) {
                uniqueNotificationID = jsonObject.getString("uniqueNotificationID");
                sendNotificationEvent(context, uniqueNotificationID, "notificationReceived", false, "remote");
            } else {
                sendNotificationEvent(context, "", "notificationReceived", false, "remote");
            }
        } catch (Exception e) {
            e.printStackTrace();
        }
    }

    /**
     * Method that handles the displaying of a simple push notification (no reward, no destination to Google Play Store).
     *
     * @param context          The activity context.
     * @param gamedockNotification The object containing the information about the push notification
     */
    private static void showNotification(final Context context, final GamedockNotification gamedockNotification) {
        if (checkNotificationStatus(context)) {
            Handler handler = new Handler(Looper.getMainLooper());
            handler.post(new Runnable() {
                @Override
                public void run() {
                    try {
                        Random r = new Random();

                        gamedockNotification.id = r.nextInt(9999 - 1000) + 1000;

                        String packageName = context.getPackageName();
                        final PackageManager pm = context.getPackageManager();
                        ApplicationInfo applicationInfo = pm.getApplicationInfo(packageName, PackageManager.GET_META_DATA);
                        final Resources resources = pm.getResourcesForApplication(applicationInfo);

                        final int appIconResId = applicationInfo.icon;

                        final int resourceId = context.getResources().getIdentifier("notify_icon_small", "drawable",
                                context.getPackageName());

                        int smallIconId;

                        if (resourceId != 0) {
                            smallIconId = resourceId;
                        } else {
                            smallIconId = appIconResId;
                        }

                        final Bitmap appIconBitmap = BitmapFactory.decodeResource(resources, appIconResId);

                        final NotificationCompat.Builder mBuilder = new NotificationCompat.Builder(context, NotificationChannel)
                                .setSmallIcon(smallIconId)
                                .setLargeIcon(appIconBitmap)
                                .setDefaults(Notification.DEFAULT_SOUND | Notification.DEFAULT_VIBRATE | Notification.FLAG_SHOW_LIGHTS)
                                .setAutoCancel(true)
                                .setLights(0xff00ff00, 300, 100)
                                .setDeleteIntent(createOnDismissedIntent(context, gamedockNotification.id, gamedockNotification.uniqueNotificationId))
                                .setPriority(NotificationCompat.PRIORITY_HIGH);

                        Intent resultIntent = pm.getLaunchIntentForPackage(packageName);
                        resultIntent.setFlags(Intent.FLAG_ACTIVITY_CLEAR_TOP | Intent.FLAG_ACTIVITY_SINGLE_TOP);

                        saveNotificationDataForProcessing(context, gamedockNotification);

                        TaskStackBuilder stackBuilder = TaskStackBuilder.create(context);
                        stackBuilder.addNextIntent(resultIntent);

                        PendingIntent resultPendingIntent = stackBuilder.getPendingIntent(0, PendingIntent.FLAG_UPDATE_CURRENT);
                        mBuilder.setContentIntent(resultPendingIntent);

                        final android.app.NotificationManager mNotificationManager = (android.app.NotificationManager) context.getSystemService(Context.NOTIFICATION_SERVICE);

                        if(mNotificationManager == null) {
                            return;
                        }

                        if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.O) {
                            NotificationChannel channel = new NotificationChannel(NotificationChannel,
                                    "Gamedock Notifications",
                                    android.app.NotificationManager.IMPORTANCE_HIGH);
                            mNotificationManager.createNotificationChannel(channel);
                        }

                        try {
                            mBuilder.getClass().getMethod("setChannelId", String.class);
                            mBuilder.setChannelId(NotificationChannel);
                        } catch (NoSuchMethodException e) {
                            LoggingUtil.e("Error while showing push notification! Gamedock SDK requires appcompat-v4-26.+ in order to show push notifications to Android 8 phones. Please update your appcompat-v4 library!");
                        }

                        switch (gamedockNotification.type) {
                            case "largeImage":
                                mBuilder.setContentTitle(gamedockNotification.title);
                                mBuilder.setContentText(gamedockNotification.message);
                                if(!gamedockNotification.imageUrl.isEmpty()) {
                                    GamedockSDK.getInstance(context).requestImage(gamedockNotification.imageUrl, 0, "notificationImage", new ImageLoadCallbacks(new OnImageLoadListener() {
                                        @Override
                                        public void ImageLoadSuccess(String localPath, ImageContext imageContext) {
                                            Bitmap image = ImageLoaderUtil.getInstance(context).getBitmapImage(gamedockNotification.imageUrl);
                                            NotificationCompat.BigPictureStyle s = new NotificationCompat.BigPictureStyle().bigPicture(image).bigLargeIcon(null);
                                            mBuilder.setStyle(s);

                                            mNotificationManager.notify(gamedockNotification.id, mBuilder.build());
                                        }

                                        @Override
                                        public void ImageLoadFailed(ImageContext imageContext, ErrorCodes errorCode) {
                                            mNotificationManager.notify(gamedockNotification.id, mBuilder.build());
                                        }

                                        @Override
                                        public void ImagePreLoadingCompleted() {

                                        }
                                    }));
                                }
                                break;
                            case "largeText":
                                mBuilder.setContentTitle(gamedockNotification.title);
                                mBuilder.setContentText(gamedockNotification.message);
                                mBuilder.setStyle(new NotificationCompat.BigTextStyle().bigText(gamedockNotification.message));
                                mNotificationManager.notify(gamedockNotification.id, mBuilder.build());
                                break;
                            case "custom":
                                final RemoteViews notificationLayout = new RemoteViews(context.getPackageName(), R.layout.custom_notification);

                                notificationLayout.setImageViewBitmap(R.id.notificationAppIconImage, drawableToBitmap(context.getPackageManager().getApplicationIcon(context.getPackageName())));

                                notificationLayout.setTextViewText(R.id.notificationTitle, gamedockNotification.title);
                                notificationLayout.setTextViewText(R.id.notificationMessage, gamedockNotification.message);

                                if(!gamedockNotification.color.isEmpty()){
                                    if(gamedockNotification.color.equals("white")) {
                                        notificationLayout.setTextColor(R.id.notificationTitle, context.getResources().getColor(R.color.notification_title_white));
                                        notificationLayout.setTextColor(R.id.notificationMessage, context.getResources().getColor(R.color.notification_message_white));
                                    } else if(gamedockNotification.color.equals("black")) {
                                        notificationLayout.setTextColor(R.id.notificationTitle, context.getResources().getColor(R.color.notification_title_black));
                                        notificationLayout.setTextColor(R.id.notificationMessage, context.getResources().getColor(R.color.notification_message_black));
                                    }
                                }

                                mBuilder.setCustomContentView(notificationLayout);

                                if(!gamedockNotification.imageUrl.isEmpty()) {
                                    GamedockSDK.getInstance(context).requestImage(gamedockNotification.imageUrl, 0, "notificationImage", new ImageLoadCallbacks(new OnImageLoadListener() {
                                        @Override
                                        public void ImageLoadSuccess(String localPath, ImageContext imageContext) {
                                            Bitmap image = ImageLoaderUtil.getInstance(context).getBitmapImage(gamedockNotification.imageUrl);
                                            notificationLayout.setImageViewBitmap(R.id.notificationImage, image);

                                            mNotificationManager.notify(gamedockNotification.id, mBuilder.build());
                                        }

                                        @Override
                                        public void ImageLoadFailed(ImageContext imageContext, ErrorCodes errorCode) {
                                            mNotificationManager.notify(gamedockNotification.id, mBuilder.build());
                                        }

                                        @Override
                                        public void ImagePreLoadingCompleted() {

                                        }
                                    }));
                                }
                                break;
                            case "normal":
                            default:
                                mBuilder.setContentTitle(gamedockNotification.title);
                                mBuilder.setContentText(gamedockNotification.message);
                                mNotificationManager.notify(gamedockNotification.id, mBuilder.build());
                                break;
                        }

                        LoggingUtil.d("Show notification done!");
                    } catch (Exception e) {
                        e.printStackTrace();
                    }
                }
            });
        }
    }

    private static void saveNotificationDataForProcessing(Context context, GamedockNotification gamedockNotification) {
        ArrayList<GamedockNotification> notificationsList;
        String notificationListString = GamedockSDK.getInstance(context).getStorageUtil().getString(StorageUtil.Keys.NotificationDataList, null);

        if (notificationListString != null) {
            Type notificationListType = new TypeToken<ArrayList<GamedockNotification>>() {
            }.getType();
            notificationsList = GamedockSDK.getInstance(context).getGson().fromJson(notificationListString, notificationListType);
        } else {
            notificationsList = new ArrayList<>();
        }

        notificationsList.add(gamedockNotification);
        GamedockSDK.getInstance(context).getStorageUtil().putString(StorageUtil.Keys.NotificationDataList, GamedockSDK.getInstance(context).getGson().toJson(notificationsList));
    }

    public static void processNotificationData(Context context) {
        if (context instanceof Activity) {
            android.app.NotificationManager mNotificationManager = (android.app.NotificationManager) context.getSystemService(Context.NOTIFICATION_SERVICE);
            ArrayList<GamedockNotification> notificationsList;
            String notificationListString = GamedockSDK.getInstance(context).getStorageUtil().getString(StorageUtil.Keys.NotificationDataList, null);

            if (notificationListString != null) {
                Type notificationListType = new TypeToken<ArrayList<GamedockNotification>>() {
                }.getType();
                notificationsList = GamedockSDK.getInstance(context).getGson().fromJson(notificationListString, notificationListType);
            } else {
                notificationsList = new ArrayList<>();
            }

            ArrayList<GamedockNotification> temp = new ArrayList<>();
            temp.addAll(notificationsList);

            for (int i = 0; i < temp.size(); i++) {
                GamedockNotification gamedockNotification = temp.get(i);

                if (gamedockNotification.uniqueNotificationId != null && !gamedockNotification.dismissed) {
                    NotificationManager.sendNotificationEvent(context, gamedockNotification.uniqueNotificationId, "notificationOpened", true, "remote");
                }

                if (gamedockNotification.notificationData != null) {
                    GamedockSDK.getInstance(context).getRewardCallbacks().sendNotificationRewards(gamedockNotification.notificationData);
                }

                if (gamedockNotification.destination != null && !gamedockNotification.dismissed) {
                    Intent intent = new Intent(Intent.ACTION_VIEW);
                    intent.setData(Uri.parse("market://details?id=" + gamedockNotification.destination));
                    context.startActivity(intent);
                }

                try {
                    mNotificationManager.cancel(gamedockNotification.id);
                } catch (Exception e) {
                    LoggingUtil.d("Could not cancel notification with id: " + gamedockNotification.id);
                }

                notificationsList.remove(gamedockNotification);
            }

            GamedockSDK.getInstance(context).getStorageUtil().putString(StorageUtil.Keys.NotificationDataList, GamedockSDK.getInstance(context).getGson().toJson(notificationsList));
        }
    }

    /**
     * Method that creates a dismiss intent which tracks whether or not a notification has been dismissed.
     *
     * @param context              The activity context.
     * @param creationId           The unique id with which the notification has been created.
     * @param uniqueNotificationID The unique identifier generated from the Gamedock backend.
     * @return Returns {@link PendingIntent} object created for the tracking of notification dismissing.
     */
    private static PendingIntent createOnDismissedIntent(Context context, int creationId, String uniqueNotificationID) {
        Intent intent = new Intent(context, NotificationDismissedReceiver.class);
        intent.putExtra("creationId", creationId);
        intent.putExtra("uniqueNotificationID", uniqueNotificationID);

        return PendingIntent.getBroadcast(context.getApplicationContext(), creationId, intent, 0);
    }

    /**
     * Method used to schedule a local push notification at a given {@link Date}.
     *
     * @param scheduleDate The date and time at which the push notification should be shown.
     * @param context      The activity context.
     * @param title        The title that will be displayed for the notification as defined in the Gamedock backend.
     * @param message      The message that will be displayed for the notification as defined in the Gamedock backend.
     */
    public static int scheduleLocalNotification(Date scheduleDate, Context context, String title, String message) {
        LoggingUtil.d("Scheduling local push notification at time: " + scheduleDate.getTime());
        String uniqueNotificationID = UUID.randomUUID().toString();

        PersistableBundleCompat extras = new PersistableBundleCompat();
        extras.putString("notificationTitle", title);
        extras.putString("notificationMessage", message);
        extras.putString("notificationId", uniqueNotificationID);

        long exactTime = scheduleDate.getTime() - System.currentTimeMillis();

        int notificationJobId = NotificationJob.schedule(exactTime, extras);

        sendNotificationEvent(context, uniqueNotificationID, "notificationSent", false, "local");

        return notificationJobId;
    }

    /**
     * Method that cancels a scheduled local push notification.
     *
     * @param notificationJobId The unique request code with which the notification has been created.
     */
    public static void cancelLocalNotification(int notificationJobId) {
        NotificationJob.cancel(notificationJobId);
    }

    /**
     * Method creates a local push notification of the simple type.
     *
     * @param context              The activity context.
     * @param title                The title that will be displayed for the notification as defined in the Gamedock backend.
     * @param message              The message that will be displayed for the notification as defined in the Gamedock backend.
     * @param uniqueNotificationID The unique identifier generated from the Gamedock backend.
     */
    public static void showLocalNotification(final Context context, final String title, final String message, final String uniqueNotificationID) {

        if (checkNotificationStatus(context)) {

            Handler handler = new Handler(Looper.getMainLooper());
            handler.post(new Runnable() {
                @Override
                public void run() {
                    try {

                        Random r = new Random();

                        int requestCode = r.nextInt(9999 - 1000) + 1000;

                        String packageName = context.getPackageName();
                        final PackageManager pm = context.getPackageManager();
                        ApplicationInfo applicationInfo = pm.getApplicationInfo(packageName, PackageManager.GET_META_DATA);
                        final Resources resources = pm.getResourcesForApplication(applicationInfo);

                        final int appIconResId = applicationInfo.icon;

                        final int resourceId = context.getResources().getIdentifier("notify_icon_small", "drawable",
                                context.getPackageName());

                        int smallIconId;

                        if (resourceId != 0) {
                            smallIconId = resourceId;
                        } else {
                            smallIconId = appIconResId;
                        }

                        final Bitmap appIconBitmap = BitmapFactory.decodeResource(resources, appIconResId);

                        NotificationCompat.Builder mBuilder =
                                new NotificationCompat.Builder(context)
                                        .setSmallIcon(smallIconId)
                                        .setLargeIcon(appIconBitmap)
                                        .setContentTitle(title)
                                        .setContentText(message)
                                        .setDefaults(Notification.DEFAULT_SOUND | Notification.DEFAULT_VIBRATE | Notification.FLAG_SHOW_LIGHTS)
                                        .setAutoCancel(true)
                                        .setLights(0xff00ff00, 300, 100)
                                        .setDeleteIntent(createOnDismissedIntent(context, requestCode, uniqueNotificationID));

                        if (!GamedockSDK.getInstance(context).isAppRunning()) {
                            Intent resultIntent = pm.getLaunchIntentForPackage(packageName);

                            resultIntent.putExtra("uniqueNotificationID", uniqueNotificationID);

                            TaskStackBuilder stackBuilder = TaskStackBuilder.create(context);
                            stackBuilder.addNextIntent(resultIntent);

                            PendingIntent resultPendingIntent = stackBuilder.getPendingIntent(0, PendingIntent.FLAG_UPDATE_CURRENT);
                            mBuilder.setContentIntent(resultPendingIntent);
                        } else {
                            PendingIntent resultPendingIntent =
                                    PendingIntent.getActivity(context, 0, new Intent(), PendingIntent.FLAG_UPDATE_CURRENT);
                            mBuilder.setContentIntent(resultPendingIntent);
                        }

                        android.app.NotificationManager mNotificationManager =
                                (android.app.NotificationManager) context.getSystemService(Context.NOTIFICATION_SERVICE);

                        if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.O) {
                            NotificationChannel channel = new NotificationChannel(NotificationChannel,
                                    "Gamedock Notifications",
                                    android.app.NotificationManager.IMPORTANCE_DEFAULT);
                            mNotificationManager.createNotificationChannel(channel);
                        }

                        try {
                            mBuilder.getClass().getMethod("setChannelId", String.class);
                            mBuilder.setChannelId(NotificationChannel);
                        } catch (Exception e) {
                            LoggingUtil.d("Error while showing push notification! Gamedock SDK requires appcompat-v4-26.+ in order to show push notifications to Android 8 phones. Please update your appcompat-v4 library!");
                        }

                        // mId allows you to update the notification later on.
                        mNotificationManager.notify(requestCode, mBuilder.build());
                        LoggingUtil.d("Show local notification done!");
                    } catch (Exception e) {
                        e.printStackTrace();
                    }
                }
            });
        }
    }

    public static Bitmap drawableToBitmap(Drawable drawable) {
        if (drawable instanceof BitmapDrawable) {
            return ((BitmapDrawable)drawable).getBitmap();
        }

        int width = drawable.getIntrinsicWidth();
        width = width > 0 ? width : 1;
        int height = drawable.getIntrinsicHeight();
        height = height > 0 ? height : 1;

        Bitmap bitmap = Bitmap.createBitmap(width, height, Bitmap.Config.ARGB_8888);
        Canvas canvas = new Canvas(bitmap);
        drawable.setBounds(0, 0, canvas.getWidth(), canvas.getHeight());
        drawable.draw(canvas);

        return bitmap;
    }

    public static class NotificationJob extends Job {

        public static final String TAG = "gamedockNotifications";

        @NonNull
        @Override
        protected Result onRunJob(Params params) {

            PersistableBundleCompat extras = params.getExtras();

            String notificationTitle = extras.getString("notificationTitle", null);
            String notificationMessage = extras.getString("notificationMessage", null);
            String notificationId = extras.getString("notificationId", null);

            LoggingUtil.d("Showing Local Push notification with title: " + notificationTitle + " and message: " + notificationMessage);

            if (notificationTitle != null && notificationMessage != null && notificationId != null) {
                NotificationManager.sendNotificationEvent(getContext(), notificationId, "notificationReceived", false, "local");

                NotificationManager.showLocalNotification(getContext(), notificationTitle, notificationMessage, notificationId);
            }

            return Result.SUCCESS;
        }

        public static int schedule(long exactTime, PersistableBundleCompat extras) {
            return new JobRequest.Builder(NotificationJob.TAG)
                    .setExecutionWindow(exactTime, exactTime + 5_000L)
                    .setBackoffCriteria(5_000L, JobRequest.BackoffPolicy.EXPONENTIAL)
                    .setExtras(extras)
                    .setUpdateCurrent(false)
                    .build()
                    .schedule();
        }

        public static void cancel(int notificationJobId) {
            JobManager.instance().cancel(notificationJobId);
        }
    }

}