package io.gamedock.sdk.network;

import java.io.BufferedInputStream;
import java.io.BufferedReader;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.net.ConnectException;
import java.net.HttpURLConnection;
import java.net.URL;
import java.util.Scanner;

import javax.net.ssl.HttpsURLConnection;

import io.gamedock.sdk.utils.logging.LoggingUtil;

/**
 *  * Class that handles the sending and receiving of data depending on the url.
 */
public class NetworkSimpleCall {
    /**
     * Method used to make the network request.
     * @return
     * @throws Exception
     */
    static String makeNetworkCall(String urlValue) throws Exception {
        String response = "";
        HttpsURLConnection connection = null;
        String errorResponse = null;
        int responseCode = 0;

        try {
            if (urlValue == null || urlValue.isEmpty()) {
                LoggingUtil.d("Request url is null or empty. Canceling request");
                throw new NullPointerException();
            }

            LoggingUtil.d("Making network request with Url: " + urlValue);

            URL url = new URL(urlValue);
            connection = (HttpsURLConnection) url.openConnection();
            connection.setRequestMethod("GET");
            connection.setReadTimeout(8000);
            connection.setConnectTimeout(4000);
            connection.setDoInput(true);

            responseCode = connection.getResponseCode();

            if (responseCode == HttpsURLConnection.HTTP_OK) {
                response = readInputStreamToString(connection);
            } else {
                InputStream stream = connection.getErrorStream();
                Scanner scanner = new Scanner(stream);
                scanner.useDelimiter("\\Z");
                errorResponse = scanner.next();

                if (responseCode == HttpsURLConnection.HTTP_UNAUTHORIZED) {
                    LoggingUtil.w("Response error: " + errorResponse);
                    return response;
                } else if (responseCode == HttpURLConnection.HTTP_BAD_GATEWAY
                        || responseCode == HttpURLConnection.HTTP_UNAVAILABLE
                        || responseCode == HttpURLConnection.HTTP_GATEWAY_TIMEOUT) {
                    throw new ConnectException("\n" + errorResponse);
                }
            }
        } catch (Exception e) {
            throw e;
        } finally {
            try {
                assert connection != null;
                connection.disconnect();
            } catch (Exception e) {
                e.printStackTrace(); //If you want further info on failure...
            }
        }
        return response;
    }

    /**
     * Method used to read and process the response from the connection input stream.
     * @param connection
     * @return
     */
    private static String readInputStreamToString(HttpURLConnection connection) {
        String response = null;
        StringBuilder sb = new StringBuilder();
        InputStream is = null;

        try {
            is = new BufferedInputStream(connection.getInputStream());
            BufferedReader br = new BufferedReader(new InputStreamReader(is));
            String inputLine;
            while ((inputLine = br.readLine()) != null) {
                sb.append(inputLine);
            }
            response = sb.toString();
        } catch (Exception e) {
            LoggingUtil.i("Error reading InputStream");
            response = null;
        } finally {
            if (is != null) {
                try {
                    is.close();
                } catch (IOException e) {
                    LoggingUtil.i("Error closing InputStream");
                }
            }
        }
        return response;
    }
}
