package io.gamedock.sdk.network;

import android.content.Context;

import com.google.gson.JsonObject;

import java.net.URLEncoder;
import java.util.Calendar;
import java.util.TimeZone;
import java.util.concurrent.CountDownLatch;

import io.gamedock.sdk.GamedockEnvironment;
import io.gamedock.sdk.GamedockSDK;
import io.gamedock.sdk.config.internal.GamedockConfigManager;
import io.gamedock.sdk.events.Event;
import io.gamedock.sdk.events.EventActionListener;
import io.gamedock.sdk.events.EventManager;
import io.gamedock.sdk.events.response.ResponseEvent;
import io.gamedock.sdk.utils.logging.LoggingUtil;
import io.gamedock.sdk.utils.storage.StorageUtil;
import io.reactivex.Observable;
import io.reactivex.ObservableEmitter;
import io.reactivex.ObservableOnSubscribe;
import io.reactivex.Observer;
import io.reactivex.annotations.NonNull;
import io.reactivex.disposables.Disposable;

/**
 * Api Class used to make network calls. A new object needs to be created each time a network call is done.
 */
public class NetworkApi {

    private EventActionListener actionListener;
    private Event event;

    private NetworkSimpleCallListener simpleCallListener;
    private String simpleCallUrl;

    private CountDownLatch countDownLatch;

    private Context context;

    NetworkApi(Context context, Event event, EventActionListener actionListener) {
        this.context = context;
        this.event = event;
        this.actionListener = actionListener;
    }

    public NetworkApi(Context context, String url, NetworkSimpleCallListener simpleCallListener) {
        this.context = context;
        this.simpleCallUrl = url;
        this.simpleCallListener = simpleCallListener;
    }

    Observer<String> dataSenderObserver = new Observer<String>() {
        @Override
        public void onSubscribe(@NonNull Disposable d) {

        }

        @Override
        public void onNext(@NonNull String response) {
            addToRecentEvents();

            ResponseEvent responseEvent;

            if (event.isQueued()) {
                responseEvent = new ResponseEvent();
                responseEvent.setName(event.getName());
                responseEvent.setType("queued");

                if (GamedockSDK.getInstance(context).globalEventActionListener != null) {
                    GamedockSDK.getInstance(context).globalEventActionListener.onResponse(responseEvent);
                } else {
                    GamedockSDK.getInstance(context).processResponseEvent(responseEvent, null);
                }
            } else {
                responseEvent = ResponseEvent.Build(response);
            }

            if (actionListener != null) {
                actionListener.onResponse(responseEvent);
            }
        }

        @Override
        public void onError(@NonNull Throwable e) {
            e.printStackTrace();
        }

        @Override
        public void onComplete() {
            if (countDownLatch != null) {
                countDownLatch.countDown();
            }

            context = null;
        }
    };

    /**
     * Method used to make the network call to the backend and send back the response to the observer.
     */
    Observable<String> sendEvent() {
        final NetworkApi api = this;
        return Observable.create(new ObservableOnSubscribe<String>() {
            @Override
            public void subscribe(ObservableEmitter<String> observableEmitter) throws Exception {
                try {
                    String response = NetworkCall.makeNetworkCall(context, api);
                    observableEmitter.onNext(response);
                    observableEmitter.onComplete();
                } catch (Exception e) {
                    observableEmitter.onError(e);
                }
            }
        });
    }

    /**
     * Method that adds an event to a recent event tracker which checks for duplicate events.
     */
    void addToRecentEvents() {
        //Add to recent events
        Event recentEvent = new Event(context);
        recentEvent.setName(event.getName());
        recentEvent.setTimestamp(event.getTimestamp());
        recentEvent.setQueued(event.isQueued());
        EventManager.getInstance(context).addToRecentSentEvents(recentEvent);
    }

    /**
     * Method used to retrieve the API url based on the Environment.
     *
     * @return The backend URL.
     */
    String getApiUrl() {
        String initialUrl = GamedockSDK.getInstance(context).getEnvironment().getURL();
        String finalUrl;

        if (GamedockSDK.getInstance(context).getEnvironment() == GamedockEnvironment.PRODUCTION) {
            finalUrl = "https://"
                    + GamedockConfigManager.getInstance(context).spil.subDomain
                    + initialUrl
                    + GamedockSDK.getInstance(context).getApplicationPackageName()
                    + "/" + event.getName();
        } else {
            finalUrl = "https://"
                    + initialUrl
                    + GamedockSDK.getInstance(context).getApplicationPackageName()
                    + "/" + event.getName();
        }

        return finalUrl;
    }

    /**
     * Method used to get the payload of the Event that will be sent to the backend.
     *
     * @return A query parsed string that will be sent to the backend.
     */
    String getEventPayload() {
        try {
            String qsRetryReason = "";
            if (event.hasRetryReason()) {
                qsRetryReason = "&retryreason=" + event.getRetryReason();
            }

            JsonObject sendJSONObject = new JsonObject();
            sendJSONObject.addProperty("id", EventManager.getInstance(context).getSendId());
            sendJSONObject.addProperty("ts", System.currentTimeMillis());
            sendJSONObject.addProperty("timezoneOffset", ((Calendar.getInstance(TimeZone.getDefault()).get(Calendar.ZONE_OFFSET) + Calendar.getInstance(TimeZone.getDefault()).get(Calendar.DST_OFFSET)) / 60000));

            int priv = GamedockSDK.getInstance(context).getStorageUtil().getInt(StorageUtil.Keys.GDPRStatus, -1);

            String log = "GamedockSDK Event: " + ("name=" + event.getName()
                    + "&data=" + event.data.toString()
                    + "&customData=" + event.customData.toString()
                    + "&send=" + sendJSONObject.toString()
                    + "&queued=" + (event.isQueued() ? 1 : 0))
                    + (priv > -1 ? "&priv=" + priv : "&priv=0")
                    + qsRetryReason;

            String payload = "name=" + URLEncode(event.getName())
                    + "&data=" + URLEncode(event.data.toString())
                    + "&customData=" + URLEncode(event.customData.toString())
                    + "&send=" + sendJSONObject.toString()
                    + "&queued=" + (event.isQueued() ? 1 : 0)
                    + (priv > -1 ? "&priv=" + priv : "&priv=0")
                    + qsRetryReason;

            if (GamedockConfigManager.getInstance(context).isDebugModeEnabled()) {
                payload = payload + "&debugMode=true";
                log = log + "&debugMode=true";
            }

            //Check if social
            if (event.getSpilToken() != null) {
                payload = payload + "&spilToken=" + URLEncode(event.getSpilToken());
                log = log + "&spilToken=" + event.getSpilToken();
            }

            LoggingUtil.d(log);
            LoggingUtil.d("Environment set to: " + GamedockSDK.getInstance(context).getEnvironment().getName());

            return payload;
        } catch (Exception e) {
            e.printStackTrace();
            return "";
        }
    }

    /**
     * Method used to make the network call to the backend and send back the response to the observer.
     */
    public Observable<String> sendSimpleCall() {
        final NetworkApi api = this;
        return Observable.create(new ObservableOnSubscribe<String>() {
            @Override
            public void subscribe(ObservableEmitter<String> observableEmitter) throws Exception {
                try {
                    String response = NetworkSimpleCall.makeNetworkCall(simpleCallUrl);
                    if (simpleCallListener != null) {
                        simpleCallListener.onSuccess(response);
                    }
                    observableEmitter.onNext(response);
                    observableEmitter.onComplete();
                } catch (Exception e) {
                    observableEmitter.onError(e);
                }
            }
        });
    }

    /**
     * Method that URL Encodes the event payload.
     *
     * @param input The {@link Event} that has to be sent in a {@link String} format.
     * @return The URL Encoded {@link Event}.
     * @throws java.io.UnsupportedEncodingException
     */
    private static String URLEncode(String input) throws java.io.UnsupportedEncodingException {
        return URLEncoder.encode(input, "UTF-8");
    }

    public CountDownLatch getCountDownLatch() {
        return countDownLatch;
    }

    public void setCountDownLatch(CountDownLatch countDownLatch) {
        this.countDownLatch = countDownLatch;
    }

    public Event getEvent() {
        return event;
    }

    EventActionListener getActionListener() {
        return actionListener;
    }
}
