package io.gamedock.sdk.mission;

import android.content.Context;

import com.google.gson.Gson;

import java.io.InputStream;
import java.util.ArrayList;

import io.gamedock.sdk.GamedockSDK;
import io.gamedock.sdk.events.internal.MissionEvent;
import io.gamedock.sdk.models.mission.Container;
import io.gamedock.sdk.models.mission.Mission;
import io.gamedock.sdk.models.mission.MissionsConfiguration;
import io.gamedock.sdk.utils.error.ErrorCodes;
import io.gamedock.sdk.utils.features.FeaturesUtil;
import io.gamedock.sdk.utils.logging.LoggingUtil;
import io.gamedock.sdk.utils.storage.StorageUtil;

/**
 * Class used to manage mission configuration operations.
 */
public class MissionConfigurationManager {
    private static final Object lock = new Object();

    public static final String FEATURE_NAME = "missions";

    private static volatile MissionConfigurationManager mInstance = null;
    private Context context;

    private MissionsConfiguration missionConfiguration = null;
    private Gson gson;
    private StorageUtil storageUtil;

    public boolean loadFailedFired;

    private MissionConfigurationManager(Context context) {
        this.context = context;
        gson = GamedockSDK.getInstance(context).getGson();
        storageUtil = GamedockSDK.getInstance(context).getStorageUtil();
    }

    public static MissionConfigurationManager getInstance(Context context) {
        if (mInstance == null) {
            synchronized (lock) {
                if (mInstance == null) {
                    mInstance = new MissionConfigurationManager(context);
                }
            }
        }
        return mInstance;
    }

    /**
     * Method that requests the mission configuration data from the Gamedock backend.
     */
    public void requestMissionConfig() {
        if (!FeaturesUtil.isFeatureEnabled(FEATURE_NAME)) {
            return;
        }

        MissionEvent missionEvent = new MissionEvent(context);
        missionEvent.setRequestMissionConfig();

        GamedockSDK.getInstance(context).trackEvent(missionEvent, null);
    }

    /**
     * Method used to process the mission configuration response from the backend containing containers and missions.
     *
     * @param receivedMissionContainers The containers list.
     * @param receivedMissions          The missions list.
     */
    public void processMissionConfig(ArrayList<Container> receivedMissionContainers, ArrayList<Mission> receivedMissions) {
        if (receivedMissionContainers == null || receivedMissions == null) {
            GamedockSDK.getInstance(context).getMissionConfigurationCallbacks().missionConfigurationError(ErrorCodes.MissionConfigurationServerError);
            return;
        }
        MissionsConfiguration missionConfiguration = getMissionConfiguration();

        if (missionConfiguration == null) {
            return;
        }

        missionConfiguration.setContainers(receivedMissionContainers);
        missionConfiguration.setMissions(receivedMissions);

        updateMissionConfiguration(missionConfiguration);

        if (missionConfiguration.getMissions().isEmpty()) {
            GamedockSDK.getInstance(context).getMissionConfigurationCallbacks().missionConfigurationNotAvailable();
        } else {
            GamedockSDK.getInstance(context).getMissionConfigurationCallbacks().missionConfigurationAvailable(GamedockSDK.getInstance(context).getGson().toJson(GamedockSDK.getInstance(context).getContainersConfiguration()), GamedockSDK.getInstance(context).getGson().toJson(GamedockSDK.getInstance(context).getMissionsConfiguration(null)));
        }
    }

    /**
     * Retrieves the mission configuration data from either the shared prefs or the local JSON file.
     *
     * @return Returns the {@link MissionsConfiguration} object containing all the mission configuration information.
     */
    public MissionsConfiguration getMissionConfiguration() {
        if (missionConfiguration != null) {
            return missionConfiguration;
        } else {
            String missionConfigurationJSON = storageUtil.getString(StorageUtil.Keys.SpilMissionConfiguration, null);
            if (missionConfigurationJSON != null) {
                missionConfiguration = gson.fromJson(missionConfigurationJSON, MissionsConfiguration.class);
                return missionConfiguration;
            } else {
                missionConfigurationJSON = loadMissionConfigFromAssets();

                if (missionConfigurationJSON != null && missionConfigurationJSON.length() > 0) {
                    missionConfiguration = gson.fromJson(missionConfigurationJSON, MissionsConfiguration.class);
                    storageUtil.putString(StorageUtil.Keys.SpilMissionConfiguration, gson.toJson(missionConfiguration));
                    return missionConfiguration;
                } else {
                    if (!loadFailedFired) {
                        GamedockSDK.getInstance(context).getMissionConfigurationCallbacks().missionConfigurationError(ErrorCodes.LoadFailed);
                        loadFailedFired = true;
                    }
                    return null;
                }
            }
        }
    }

    /**
     * Method used to retrieve the containers configuration.
     *
     * @return The containers list.
     */
    public ArrayList<Container> getContainersConfiguration() {
        if (!FeaturesUtil.isFeatureEnabled(FEATURE_NAME)) {
            return new ArrayList<>();
        }

        return getMissionConfiguration().getContainers();
    }

    /**
     * Method used to retrieve a specific container based on the id.
     *
     * @param containerId The container id.
     * @return The Container object.
     */
    public Container getContainerConfiguration(String containerId) {
        if (!FeaturesUtil.isFeatureEnabled(FEATURE_NAME)) {
            return null;
        }

        for (Container container : getMissionConfiguration().getContainers()) {
            if (container.getContainerId().equals(containerId)) {
                return container;
            }
        }
        return null;
    }

    /**
     * Method used to retrieve the missions configuration full list or the list which match the container id.
     *
     * @param containerId The container id.
     * @return The missions list.
     */
    public ArrayList<Mission> getMissionsConfiguration(String containerId) {
        if (!FeaturesUtil.isFeatureEnabled(FEATURE_NAME)) {
            return new ArrayList<>();
        }

        if (containerId == null) {
            return getMissionConfiguration().getMissions();
        }

        ArrayList<Mission> selectedMissions = new ArrayList<>();
        for (Mission entry : getMissionConfiguration().getMissions()) {
            if (entry.getContainerId().equals(containerId)) {
                selectedMissions.add(entry);
            }
        }

        return selectedMissions;
    }

    /**
     * Method used to retrieve the missions matching a specific list of mission prerequisites.
     *
     * @param missionPrerequisites A list of mission ids.
     * @return The mission list matching the condition.
     */
    public ArrayList<Mission> getMissionsConfigurationWithPrerequisites(ArrayList<String> missionPrerequisites) {
        if (!FeaturesUtil.isFeatureEnabled(FEATURE_NAME)) {
            return new ArrayList<>();
        }

        if (missionPrerequisites == null || missionPrerequisites.size() == 0) {
            return new ArrayList<>();
        }

        ArrayList<Mission> selectedMissions = new ArrayList<>();
        for (Mission entry : getMissionConfiguration().getMissions()) {
            for (String missionPrerequisite : missionPrerequisites) {
                if (entry.getMissionPrerequisites().contains(missionPrerequisite)) {
                    selectedMissions.add(entry);
                }
            }
        }

        return selectedMissions;
    }

    /**
     * Method used to retrieve the missions matching a specific list of mission unlock id.
     *
     * @param missionUnlocks A list of mission unlock ids.
     * @return The mission list matching the condition.
     */
    public ArrayList<Mission> getMissionsConfigurationWithUnlocks(ArrayList<String> missionUnlocks) {
        if (!FeaturesUtil.isFeatureEnabled(FEATURE_NAME)) {
            return new ArrayList<>();
        }

        if (missionUnlocks == null || missionUnlocks.size() == 0) {
            return new ArrayList<>();
        }

        ArrayList<Mission> selectedMissions = new ArrayList<>();
        for (Mission entry : getMissionConfiguration().getMissions()) {
            for (String missionUnlock : missionUnlocks) {
                if (entry.getMissionUnlocks().contains(missionUnlock)) {
                    selectedMissions.add(entry);
                }
            }
        }

        return selectedMissions;
    }

    /**
     * Method used to retrieve a mission based on an id.
     *
     * @param missionId The mission id.
     * @return The Mission object or null if no mission was found.
     */
    public Mission getMissionConfiguration(String missionId) {
        if (!FeaturesUtil.isFeatureEnabled(FEATURE_NAME)) {
            return null;
        }

        for (Mission mission : getMissionConfiguration().getMissions()) {
            if (mission.getMissionId().equals(missionId)) {
                return mission;
            }
        }
        return null;
    }

    /**
     * Retrieves the game data in string format from either the shared prefs or the local JSON file.
     *
     * @return Returns the {@link MissionsConfiguration} as a String JSON.
     */
    public String getMissionConfigurationJSON() {
        if (!FeaturesUtil.isFeatureEnabled(FEATURE_NAME)) {
            return null;
        }

        if (getMissionConfiguration() != null) {
            return gson.toJson(getMissionConfiguration());
        }

        return null;
    }

    /**
     * Method that updates the mission configuration by storing it in the shared preferences.
     *
     * @param updatedMissionConfiguration The {@link MissionsConfiguration} object that needs to be updated.
     */
    public void updateMissionConfiguration(MissionsConfiguration updatedMissionConfiguration) {

        storageUtil.putString(StorageUtil.Keys.SpilMissionConfiguration, gson.toJson(updatedMissionConfiguration));

        this.missionConfiguration = updatedMissionConfiguration;
    }

    /**
     * Retrieves Mission Configuration from the JSON file called "defaultMissionConfig.json".
     *
     * @return The JSON String of the game data containing all the information.
     */
    private String loadMissionConfigFromAssets() {
        String json = null;

        try {
            InputStream is = context.getAssets().open("defaultMissionConfig.json");
            int size = is.available();
            byte[] buffer = new byte[size];
            is.read(buffer);
            is.close();
            json = new String(buffer, "UTF-8");

        } catch (Exception ex) {
            LoggingUtil.e("The 'defaultMissionConfig.json' file is missing from your assets folder. If you want to use the Missions functionality please include this file.");
        }

        return json;
    }

    public void resetMissionConfiguration() {
        missionConfiguration = null;
        storageUtil.remove(StorageUtil.Keys.SpilMissionConfiguration);
        mInstance = null;
    }


}
