package io.gamedock.sdk.localization;

import android.content.Context;

import androidx.annotation.VisibleForTesting;

import com.google.gson.Gson;
import com.google.gson.reflect.TypeToken;

import org.json.JSONObject;

import java.io.InputStream;
import java.lang.reflect.Type;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.Map;

import io.gamedock.sdk.GamedockSDK;
import io.gamedock.sdk.events.internal.LocalizationEvent;
import io.gamedock.sdk.models.localization.LocalizationData;
import io.gamedock.sdk.network.NetworkApi;
import io.gamedock.sdk.network.NetworkSimpleCallListener;
import io.gamedock.sdk.utils.assets.FileAssetsUtil;
import io.gamedock.sdk.utils.error.ErrorCodes;
import io.gamedock.sdk.utils.features.FeaturesUtil;
import io.gamedock.sdk.utils.logging.LoggingUtil;
import io.gamedock.sdk.utils.storage.StorageUtil;
import io.reactivex.Observable;
import io.reactivex.Observer;
import io.reactivex.android.schedulers.AndroidSchedulers;
import io.reactivex.annotations.NonNull;
import io.reactivex.disposables.Disposable;
import io.reactivex.schedulers.Schedulers;

public class LocalizationManager {

    private static final Object lock = new Object();

    public static final String FEATURE_NAME = "localization";

    private static volatile LocalizationManager mInstance = null;
    private Context context;

    private final Gson gson;
    private final StorageUtil storageUtil;
    private final FileAssetsUtil fileAssetsUtil;

    public boolean fallbackToDefaultLocale;
    private String requestedLocale;
    public LocalizationData defaultLocalizationData;
    public LocalizationData localizationData;

    private final ArrayList<String> sentKeysForStatistics = new ArrayList<>();

    private LocalizationManager(Context context, Gson gson, StorageUtil storageUtil, FileAssetsUtil fileAssetsUtil) {
        this.context = context;
        this.gson = gson;
        this.storageUtil = storageUtil;
        this.fileAssetsUtil = fileAssetsUtil;
    }

    public static LocalizationManager getInstance(Context context) {
        if (mInstance == null) {
            synchronized (lock) {
                if (mInstance == null) {
                    mInstance = new LocalizationManager(context, GamedockSDK.getInstance(context).getGson(), GamedockSDK.getInstance(context).getStorageUtil(), GamedockSDK.getFileAssetsUtil());
                }
            }
        }
        return mInstance;
    }

    /**
     * Method used to request a localization based on a supplied locale.
     *
     * @param locale                  The requested locale. For a list of the local codes, check {@link LocalizationManager.Locales}.
     * @param fallbackToDefaultLocale Flag to inform that when a key in the localization is not present, should it fall back to the default locale or not.
     */
    public void requestLocalization(String locale, boolean fallbackToDefaultLocale) {
        if (!FeaturesUtil.isFeatureEnabled(FEATURE_NAME)) {
            return;
        }

        if (storageUtil.contains(StorageUtil.Keys.LocalizationDefaultLocale)) {
            defaultLocalizationData = gson.fromJson(storageUtil.getString(StorageUtil.Keys.LocalizationDefaultLocale, "{}"), LocalizationData.class);
        } else {
            String localDefaultLocalizationJSON = loadLocalizationDataFromAssets(context, fileAssetsUtil);
            createDefaultLocalizationData(localDefaultLocalizationJSON, "en", "localStorage");
        }

        if (storageUtil.contains(StorageUtil.Keys.LocalizationLocale)) {
            localizationData = gson.fromJson(storageUtil.getString(StorageUtil.Keys.LocalizationLocale, "{}"), LocalizationData.class);
        }

        this.fallbackToDefaultLocale = fallbackToDefaultLocale;
        this.requestedLocale = locale;

        LocalizationEvent localizationEvent = new LocalizationEvent(context);
        localizationEvent.setRequestLocalization();
        localizationEvent.addCustomData("locale", locale);

        GamedockSDK.getInstance(context).trackEvent(localizationEvent, null);
    }

    /**
     * Method used to process the response for the localization.
     * Based on this response, it will initiate network request based on whether the requested locale should be downloaded (if it's not already cached) and the default locale.
     *
     * @param responseData The response JSON received from the Gamedock Backend.
     */
    public void processRequestLocalizationResponse(JSONObject responseData) {
        try {
            String locale = null;
            if (responseData.has("locale")) {
                locale = responseData.getString("locale");
            }
            final String defaultLocale = responseData.getString("defaultLocale");

            String url = null;
            if (responseData.has("url")) {
                url = responseData.getString("url");
            }
            final String defaultUrl = responseData.getString("defaultUrl");

            boolean shouldDownloadLocale = false;
            boolean shouldDownloadDefaultLocale = false;

            //Check if requested and the default locale should be downloaded based on the initial request
            if (locale == null || url == null) {
                if (defaultLocalizationData == null) {
                    shouldDownloadDefaultLocale = true;
                } else {
                    if (!defaultLocale.equals(defaultLocalizationData.getLocale()) || !defaultUrl.equals(defaultLocalizationData.getUrl())) {
                        shouldDownloadDefaultLocale = true;
                    }
                }
            } else {
                if (localizationData == null) {
                    shouldDownloadLocale = true;
                } else {
                    if (!defaultLocale.equals(localizationData.getLocale()) || !defaultUrl.equals(localizationData.getUrl())) {
                        shouldDownloadLocale = true;
                    }
                }
            }

            if (!shouldDownloadLocale && !shouldDownloadDefaultLocale) {
                GamedockSDK.getInstance(context).getLocalizationCallbacks().localizationAvailable(requestedLocale, true);
                return;
            }

            //Create a list of Observable objects, either containing 1 or 2 elements
            //Will have either the default locale downloading, the requested locale downloading or both
            Observable<String>[] list = new Observable[2];
            final boolean finalShouldDownloadLocale = shouldDownloadLocale;
            final String finalLocale = locale;
            final String finalUrl = url;

            if (shouldDownloadDefaultLocale) {
                NetworkApi defaultLocaleSenderApi = new NetworkApi(context, defaultUrl, new NetworkSimpleCallListener() {
                    @Override
                    public void onSuccess(String response) {
                        createDefaultLocalizationData(response, defaultLocale, defaultUrl);
                    }

                    @Override
                    public void onFailure(ErrorCodes error) {
                        GamedockSDK.getInstance(context).getLocalizationCallbacks().localizationError(error);
                    }
                });

                list[0] = defaultLocaleSenderApi.sendSimpleCall();
            }

            if (shouldDownloadLocale) {
                NetworkApi localeSenderApi = new NetworkApi(context, url, new NetworkSimpleCallListener() {
                    @Override
                    public void onSuccess(String response) {
                        createLocalizationData(response, finalLocale, finalUrl);
                    }

                    @Override
                    public void onFailure(ErrorCodes error) {
                        GamedockSDK.getInstance(context).getLocalizationCallbacks().localizationError(error);
                    }
                });

                list[1] = localeSenderApi.sendSimpleCall();
            }

            if (list[0] == null) {
                list = new Observable[]{list[1]};
            } else if (list[1] == null) {
                list = new Observable[]{list[0]};
            }

            //The mergeArray function only accepts a predefined list of Observable objects so in cases where we don't need to download both default and localization information the list needs to be recreated
            Observable.mergeArray(list)
                    .subscribeOn(Schedulers.newThread())
                    .observeOn(AndroidSchedulers.mainThread())
                    .subscribe(new Observer<String>() {
                        @Override
                        public void onSubscribe(@NonNull Disposable d) {
                        }

                        @Override
                        public void onNext(@NonNull String s) {
                        }

                        @Override
                        public void onError(@NonNull Throwable e) {
                            e.printStackTrace();
                            GamedockSDK.getInstance(context).getLocalizationCallbacks().localizationError(ErrorCodes.LocalizationError);
                        }

                        @Override
                        public void onComplete() {
                            if (finalShouldDownloadLocale) {
                                GamedockSDK.getInstance(context).getLocalizationCallbacks().localizationAvailable(finalLocale, false);
                            } else {
                                GamedockSDK.getInstance(context).getLocalizationCallbacks().localizationAvailable(defaultLocale, true);
                            }

                        }
                    });
        } catch (Exception e) {
            GamedockSDK.getInstance(context).getLocalizationCallbacks().localizationNotAvailable();
            e.printStackTrace();
        }
    }

    private void createDefaultLocalizationData(String response, String defaultLocale, String defaultUrl) {
        Type listType = new TypeToken<HashMap<String, String>>() {
        }.getType();

        defaultLocalizationData = new LocalizationData();
        defaultLocalizationData.setDefault(true);
        defaultLocalizationData.setLocale(defaultLocale);
        defaultLocalizationData.setUrl(defaultUrl);
        HashMap<String, String> localizationDataMap = new HashMap<>();
        if (response != null) {
            localizationDataMap = gson.fromJson(response, listType);
        }
        defaultLocalizationData.setData(localizationDataMap);

        storageUtil.putString(StorageUtil.Keys.LocalizationDefaultLocale, gson.toJson(defaultLocalizationData));
    }


    private void createLocalizationData(String response, String finalLocale, String finalUrl) {
        Type listType = new TypeToken<HashMap<String, String>>() {
        }.getType();

        localizationData = new LocalizationData();
        localizationData.setDefault(false);
        localizationData.setLocale(finalLocale);
        localizationData.setUrl(finalUrl);
        HashMap<String, String> localizationDataMap = new HashMap<>();
        if (response != null) {
            localizationDataMap = gson.fromJson(response, listType);
        }
        localizationData.setData(localizationDataMap);

        storageUtil.putString(StorageUtil.Keys.LocalizationLocale, gson.toJson(localizationData));
    }

    /**
     * Method used to retrieve the localization value based on a key.
     * Will fallback to default if the fallback to default is true when requesting a localization.
     * If the key is not present and the fallback to default flag is false, it will return the defaultValue.
     *
     * @param key          The key for which the localization value should be supplied.
     * @param defaultValue The default value passed by the developer in cases where the key is not found.
     * @return The localization value.
     */
    public String getLocalization(String key, String defaultValue) {
        LocalizationData currentLocalizationData = null;

        if (localizationData != null && requestedLocale.equals(localizationData.getLocale())) {
            currentLocalizationData = localizationData;
        } else if (defaultLocalizationData != null && (requestedLocale.equals(defaultLocalizationData.getLocale()) || fallbackToDefaultLocale)) {
            currentLocalizationData = defaultLocalizationData;
        }

        if (currentLocalizationData == null) {
            if (requestedLocale != null) {
                sendKeyNotFoundEvent(key, "-", requestedLocale);
            } else {
                sendKeyNotFoundEvent(key, "-", "-");
            }

            return defaultValue;
        }

        if (currentLocalizationData.getData().containsKey(key)) {
            return currentLocalizationData.getData().get(key);
        } else {
            sendKeyNotFoundEvent(key, currentLocalizationData.getUrl(), currentLocalizationData.getLocale());
            if (!fallbackToDefaultLocale) {
                return defaultValue;
            }

            if (defaultLocalizationData == null) {
                sendKeyNotFoundEvent(key, "-", "-");
                return defaultValue;
            }

            if (defaultLocalizationData.getData().containsKey(key)) {
                return defaultLocalizationData.getData().get(key);
            } else {
                sendKeyNotFoundEvent(key, defaultLocalizationData.getUrl(), defaultLocalizationData.getLocale());
                return defaultValue;
            }
        }
    }

    /**
     * Method used to retrieve the localization value based on a key.
     * Will fallback to default if the fallback to default is true when requesting a localization.
     * If the key is not present and the fallback to default flag is false, it will return the defaultValue.
     *
     * @param key          The key for which the localization value should be supplied.
     * @param defaultValue The default value passed by the developer in cases where the key is not found.
     * @param args         List of arguments which will be used to replace dynamic values in the string ({0}, {1}, ...).
     * @return The localization value.
     */
    public String getLocalization(String key, String defaultValue, String... args) {
        String localization = getLocalization(key, defaultValue);

        for (int i = 0; i < args.length; i++) {
            localization = localization.replace("{" + i + "}", args[i]);
        }

        return localization;
    }

    /**
     * Method used to retrieve the localization value based on a key.
     * Will fallback to default if the fallback to default is true when requesting a localization.
     * If the key is not present and the fallback to default flag is false, it will return the defaultValue.
     *
     * @param key          The key for which the localization value should be supplied.
     * @param defaultValue The default value passed by the developer in cases where the key is not found.
     * @param args         Map of key value pairs used to replace instances of the supplied keys in the localized string ({someKey}, {someOtherKey}, ...).
     * @return The localization value.
     */
    public String getLocalization(String key, String defaultValue, HashMap<String, String> args) {
        String localization = getLocalization(key, defaultValue);

        for (Map.Entry<String, String> entry : args.entrySet()) {
            localization = localization.replace("%{" + entry.getKey() + "}", entry.getValue());
        }

        return localization;
    }

    /**
     * Method used to send a "localizationKeyNotFound" event once per key, per session.
     *
     * @param key    The key which was not found.
     * @param url    The url where the supposed key should be present.
     * @param locale The locale where the key should be present.
     */
    private void sendKeyNotFoundEvent(String key, String url, String locale) {
        if (sentKeysForStatistics.contains(key + "-" + locale)) {
            return;
        }

        LocalizationEvent localizationEvent = new LocalizationEvent(context);
        localizationEvent.setLocalizationKeyNotFound();

        localizationEvent.addCustomData("localizationKey", key);
        localizationEvent.addCustomData("url", url);
        localizationEvent.addCustomData("locale", locale);

        GamedockSDK.getInstance(context).trackEvent(localizationEvent, null);

        sentKeysForStatistics.add(key + "-" + locale);
    }

    /**
     * Retrieves Localization Data from the JSON file called "defaultLocalizationData.json".
     *
     * @return The JSON String of the localization data containing all the information.
     */
    @VisibleForTesting
    private String loadLocalizationDataFromAssets(Context context, FileAssetsUtil fileAssetsUtil) {
        String json = "{}";

        try {
            InputStream is = fileAssetsUtil.getLocalizationFileAsset(context);
            int size = is.available();
            byte[] buffer = new byte[size];
            is.read(buffer);
            is.close();
            json = new String(buffer, "UTF-8");

        } catch (Exception e) {
            LoggingUtil.e("The 'defaultLocalizationData.json' file is missing from your assets folder. If you want to use the Localization offline functionality please include this file.");
        }

        return json;
    }

    /**
     * Class containing all the locale accepted by the Gamedock Backend.
     */
    public static class Locales {
        public static final String Afrikaans = "af_AF";
        public static final String Albanian = "sq_SQ";
        public static final String Arabic = "ar_AA";
        public static final String Arabic_Algeria = "ar_DZ";
        public static final String Arabic_Bahrain = "ar_BH";
        public static final String Arabic_Egypt = "ar_EG";
        public static final String Arabic_Iraq = "ar_IQ";
        public static final String Arabic_Jordan = "ar_JO";
        public static final String Arabic_Kuwait = "ar_KW";
        public static final String Arabic_Lebanon = "ar_LB";
        public static final String Arabic_Libya = "ar_LY";
        public static final String Arabic_Morocco = "ar_MA";
        public static final String Arabic_Oman = "ar_OM";
        public static final String Arabic_Qatar = "ar_QA";
        public static final String Arabic_Saudi_Arabia = "ar_SA";
        public static final String Arabic_Syria = "ar_SY";
        public static final String Arabic_Tunisia = "ar_TN";
        public static final String Arabic_UAE = "ar_AE";
        public static final String Arabic_Yemen = "ar_YE";
        public static final String Basque = "eu_EU";
        public static final String Belarusian = "be_BE";
        public static final String Bulgarian = "bg_BG";
        public static final String Catalan = "ca_CA";
        public static final String Chinese_Simplified = "zh_CN";
        public static final String Chinese_Traditional = "zh_TW";
        public static final String Croatian = "hr_HR";
        public static final String Czech = "cs_CS";
        public static final String Danish = "da_DA";
        public static final String Dutch_Belgium = "nl_BE";
        public static final String Dutch = "nl_NL";
        public static final String English = "en_EN";
        public static final String English_Australia = "en_AU";
        public static final String English_Belize = "en_BZ";
        public static final String English_Canada = "en_CA";
        public static final String English_Ireland = "en_IE";
        public static final String English_Jamaica = "en_JM";
        public static final String English_New_Zealand = "en_NZ";
        public static final String English_South_Africa = "en_ZA";
        public static final String English_Trinidad = "en_TT";
        public static final String English_United_Kingdom = "en_GB";
        public static final String English_United_States = "en_US";
        public static final String Estonian = "et_ET";
        public static final String Faeroese = "fo_FO";
        public static final String Farsi = "fa_FA";
        public static final String Finnish = "fi_FI";
        public static final String French_Belgium = "fr_BE";
        public static final String French_Canada = "fr_CA";
        public static final String French_Luxembourg = "fr_LU";
        public static final String French = "fr_FR";
        public static final String French_Switzerland = "fr_CH";
        public static final String Gaelic_Scotland = "gd_GD";
        public static final String German_Austria = "de_AT";
        public static final String German_Liechtenstein = "de_LI";
        public static final String German_Luxembourg = "de_LU";
        public static final String German = "de_DE";
        public static final String German_Switzerland = "de_CH";
        public static final String Greek = "el_EL";
        public static final String Hebrew = "he_HE";
        public static final String Hindi = "hi_HI";
        public static final String Hungarian = "hu_HU";
        public static final String Icelandic = "is_IS";
        public static final String Indonesian = "id_ID";
        public static final String Irish = "ga_GA";
        public static final String Italian = "it_IT";
        public static final String Italian_Switzerland = "it_CH";
        public static final String Japanese = "ja_JP";
        public static final String Korean = "ko_KR";
        public static final String Korean_Johab = "ko_KO";
        public static final String Kurdish = "ku_KU";
        public static final String Latvian = "lv_LV";
        public static final String Lithuanian = "lt_LT";
        public static final String Macedonian_FYROM = "mk_MK";
        public static final String Malayalam = "ml_ML";
        public static final String Malaysian = "ms_MS";
        public static final String Maltese = "mt_MT";
        public static final String Norwegian = "no_NO";
        public static final String Norwegian_Bokmål = "nb_NB";
        public static final String Norwegian_Nynorsk = "nn_NN";
        public static final String Polish = "pl_PL";
        public static final String Portuguese_Brazil = "pt_BR";
        public static final String Portuguese_Portugal = "pt_PT";
        public static final String Punjabi = "pa_PA";
        public static final String Rhaeto_Romanic = "rm_RM";
        public static final String Romanian = "ro_RO";
        public static final String Romanian_Republic_of_Moldova = "ro_MD";
        public static final String Russian = "ru_RU";
        public static final String Russian_Republic_of_Moldova = "ru_MD";
        public static final String Serbian = "sr_SR";
        public static final String Slovak = "sk_SK";
        public static final String Slovenian = "sl_SL";
        public static final String Sorbian = "sb_SB";
        public static final String Spanish_Argentina = "es_AR";
        public static final String Spanish_Bolivia = "es_BO";
        public static final String Spanish_Chile = "es_CL";
        public static final String Spanish_Colombia = "es_CO";
        public static final String Spanish_Costa_Rica = "es_CR";
        public static final String Spanish_Dominican_Republic = "es_DO";
        public static final String Spanish_Ecuador = "es_EC";
        public static final String Spanish_El_Salvador = "es_SV";
        public static final String Spanish_Guatemala = "es_GT";
        public static final String Spanish_Honduras = "es_HN";
        public static final String Spanish_Mexico = "es_MX";
        public static final String Spanish_Nicaragua = "es_NI";
        public static final String Spanish_Panama = "es_PA";
        public static final String Spanish_Paraguay = "es_PY";
        public static final String Spanish_Peru = "es_PE";
        public static final String Spanish_Puerto_Rico = "es_PR";
        public static final String Spanish_Spain = "es_ES";
        public static final String Spanish_Uruguay = "es_UT";
        public static final String Spanish_Venezuela = "es_VE";
        public static final String Spanish_Latin_America = "es_419";
        public static final String Swedish = "sv_SV";
        public static final String Swedish_Finland = "sv_FI";
        public static final String Thai = "th_TH";
        public static final String Tsonga = "ts_TS";
        public static final String Tswana = "tn_TN";
        public static final String Turkish = "tr_TR";
        public static final String Ukrainian = "uk_UK";
        public static final String Urdu = "ur_UR";
        public static final String Venda = "ve_VE";
        public static final String Vietnamese = "vi_VI";
        public static final String Welsh = "cy_CY";
        public static final String Xhosa = "xh_XH";
        public static final String Yiddish = "ji_JI";
        public static final String Zulu = "zu_ZU";
    }
}
