package io.gamedock.sdk.events.response;


import android.content.Context;

import com.google.gson.Gson;
import com.google.gson.JsonObject;
import com.google.gson.JsonParser;

import org.json.JSONArray;
import org.json.JSONObject;

import java.util.concurrent.ConcurrentHashMap;

import io.gamedock.sdk.models.userdata.inventory.Inventory;
import io.gamedock.sdk.models.userdata.inventory.PlayerItem;
import io.gamedock.sdk.models.userdata.inventory.UniquePlayerItem;
import io.gamedock.sdk.models.userdata.wallet.PlayerCurrency;
import io.gamedock.sdk.models.userdata.wallet.Wallet;
import io.gamedock.sdk.userdata.UserDataManager;
import io.gamedock.sdk.utils.logging.LoggingUtil;

/**
 * Class that processes information regarding player data responses.
 */
public class PlayerDataResponseEvent extends ResponseEvent {

    private Wallet walletObject = null;
    private Inventory inventoryObject = null;

    PlayerDataResponseEvent(ResponseEvent responseEvent) {
        super();

        try {
            Gson gson = new Gson();
            if (responseEvent.getAction().equals("update")) {
                if (responseEvent.responseData.has("wallet")) {
                    JSONObject walletJSON = responseEvent.responseData.getJSONObject("wallet");

                    walletObject = new Wallet();

                    if (walletJSON.has("currencies")) {
                        JSONArray currenciesJSON = walletJSON.getJSONArray("currencies");
                        ConcurrentHashMap<Integer, PlayerCurrency> currencyHashMap = new ConcurrentHashMap<>();

                        for (int i = 0; i < currenciesJSON.length(); i++) {
                            JSONObject jsonObject = currenciesJSON.getJSONObject(i);

                            PlayerCurrency currency = new PlayerCurrency();
                            currency.setId(jsonObject.getInt("id"));
                            currency.setCurrentBalance(jsonObject.getInt("currentBalance"));

                            currency.setDelta(jsonObject.getInt("delta"));

                            currencyHashMap.put(currency.getId(), currency);
                        }

                        walletObject.setCurrenciesMap(currencyHashMap);
                    }


                    if (walletJSON.has("offset")) {
                        walletObject.setOffset(walletJSON.getLong("offset"));
                    }

                    if (walletJSON.has("logic")) {
                        walletObject.setLogic(walletJSON.getString("logic"));
                    }
                }

                if (responseEvent.responseData.has("inventory")) {
                    JSONObject inventoryJSON = responseEvent.responseData.getJSONObject("inventory");

                    inventoryObject = new Inventory();

                    if (inventoryJSON.has("items")) {
                        JSONArray itemsJSON = inventoryJSON.getJSONArray("items");

                        ConcurrentHashMap<Integer, PlayerItem> itemsHashMap = new ConcurrentHashMap<>();

                        for (int i = 0; i < itemsJSON.length(); i++) {
                            JSONObject jsonObject = itemsJSON.getJSONObject(i);

                            PlayerItem item = new PlayerItem();
                            item.setId(jsonObject.getInt("id"));
                            item.setAmount(jsonObject.getInt("amount"));

                            item.setDelta(jsonObject.getInt("delta"));

                            itemsHashMap.put(item.getId(), item);
                        }
                        inventoryObject.setItemsMap(itemsHashMap);
                    }

                    if (inventoryJSON.has("uniqueItems")) {
                        JSONArray uniqueItemsJSON = inventoryJSON.getJSONArray("uniqueItems");

                        ConcurrentHashMap<String, UniquePlayerItem> uniqueItemsHashMap = new ConcurrentHashMap<>();

                        JsonParser jsonParser = new JsonParser();

                        for (int i = 0; i < uniqueItemsJSON.length(); i++) {
                            UniquePlayerItem uniquePlayerItem = gson.fromJson(uniqueItemsJSON.getJSONObject(i).toString(), UniquePlayerItem.class);
                            if (uniqueItemsJSON.getJSONObject(i).has("uniqueProperties")) {
                                JsonObject properties = (JsonObject) jsonParser.parse(uniqueItemsJSON.getJSONObject(i).getJSONObject("uniqueProperties").toString());
                                uniquePlayerItem.setUniqueProperties(properties);
                            }

                            uniqueItemsHashMap.put(uniquePlayerItem.getUniqueId(), uniquePlayerItem);
                        }

                        inventoryObject.setUniqueItemsMap(uniqueItemsHashMap);
                    }

                    if (inventoryJSON.has("offset")) {
                        inventoryObject.setOffset(inventoryJSON.getLong("offset"));
                    }

                    if (inventoryJSON.has("logic")) {
                        inventoryObject.setLogic(inventoryJSON.getString("logic"));
                    }
                }
            }

            setName(responseEvent.getName());
            setType(responseEvent.getType());
            setAction(responseEvent.getAction());
            setMessage(responseEvent.getMessage());
            this.responseData = responseEvent.responseData;
        } catch (Exception e) {
            e.printStackTrace();
        }
    }

    @Override
    public void processData(Context context) {
        LoggingUtil.d("Processing data for PlayerDataResponseEvent");

        String action = getAction().toLowerCase().trim();

        switch (action) {
            case "update":
                UserDataManager.getInstance(context).getPlayerDataManager().processPlayerDataUpdate(walletObject, inventoryObject);
                break;
            case "syncerror":
                UserDataManager.getInstance(context).processSyncError();
                break;
            case "dropped":
                UserDataManager.getInstance(context).processDroppedResponse(getMessage());
                break;
            case "lockerror":
                UserDataManager.getInstance(context).processLockError();
                break;
        }
    }
}
