package io.gamedock.sdk.events;

import android.annotation.SuppressLint;
import android.content.Context;
import android.os.Build;
import android.os.Bundle;
import android.os.Handler;
import android.provider.Settings;

import com.google.firebase.analytics.FirebaseAnalytics;
import com.google.gson.JsonElement;
import com.google.gson.JsonSyntaxException;
import com.google.gson.reflect.TypeToken;

import org.json.JSONArray;
import org.json.JSONException;
import org.json.JSONObject;

import java.io.IOException;
import java.util.ArrayList;
import java.util.Calendar;
import java.util.Iterator;
import java.util.Locale;
import java.util.Map;
import java.util.TimeZone;
import java.util.UUID;

import io.gamedock.sdk.BuildConfig;
import io.gamedock.sdk.GamedockSDK;
import io.gamedock.sdk.ads.AdEvents;
import io.gamedock.sdk.config.internal.GamedockConfigManager;
import io.gamedock.sdk.events.internal.AssetBundlesEvent;
import io.gamedock.sdk.events.internal.ConfigEvent;
import io.gamedock.sdk.events.internal.GameDataEvent;
import io.gamedock.sdk.events.internal.HeartbeatEvent;
import io.gamedock.sdk.events.internal.IAPEvent;
import io.gamedock.sdk.events.internal.InitializeSDKEvent;
import io.gamedock.sdk.events.internal.LocalizationEvent;
import io.gamedock.sdk.events.internal.MissionEvent;
import io.gamedock.sdk.events.internal.MoreAppsEvent;
import io.gamedock.sdk.events.internal.OverlayEvent;
import io.gamedock.sdk.events.internal.PromotionEvent;
import io.gamedock.sdk.events.internal.RewardEvent;
import io.gamedock.sdk.events.internal.ServerDataEvent;
import io.gamedock.sdk.events.internal.SocialLoginEvent;
import io.gamedock.sdk.events.internal.TierEvent;
import io.gamedock.sdk.events.internal.UserDataEvent;
import io.gamedock.sdk.events.response.ResponseEvent;
import io.gamedock.sdk.network.NetworkJob;
import io.gamedock.sdk.userdata.UserDataManager;
import io.gamedock.sdk.utils.device.DeviceUtil;
import io.gamedock.sdk.utils.gcm.GCMUtils;
import io.gamedock.sdk.utils.logging.LoggingUtil;
import io.gamedock.sdk.utils.other.LimitedSizeQueue;
import io.gamedock.sdk.utils.session.SessionUtil;
import io.gamedock.sdk.utils.storage.StorageUtil;
import io.gamedock.sdk.utils.userid.UserIDGenerator;

/**
 * Utility class that handles all the logic regarding the event history of the SDK.
 */
public class EventManager {

    private static final Object lock = new Object();

    private static volatile EventManager mInstance = null;
    private Context context;

    private LimitedSizeQueue<Event> recentSentEvents;
    private static int persistenceListBucketSize;

    private LimitedSizeQueue<Event> recentIAPEvents;

    private ArrayList<String> whitelistedEvents = new ArrayList<>();

    private EventManager(Context context) {
        this.context = context;
        recentSentEvents = retrieveRecentEvents();
        persistenceListBucketSize = getPersistenceListSize();

        recentIAPEvents = new LimitedSizeQueue<>(50);
    }

    public static EventManager getInstance(Context context) {
        if (mInstance == null) {
            synchronized (lock) {
                if (mInstance == null) {
                    mInstance = new EventManager(context);
                }
            }
        }
        return mInstance;
    }

    /**
     * Main SDK method that tracks the events to the Gamedock server.
     * Adds the necessary default parameters to the event.
     *
     * @param event          The event that needs to be sent to the server.
     * @param actionListener The action listener needed in case you want to override the normal response processing flow.
     */
    public void trackEvent(Event event, final EventActionListener actionListener) {
        if (GamedockSDK.getInstance(context).isGameUnpublished()) {
            LoggingUtil.e("Event " + event.getName() + " will not be sent because the game with package: " + GamedockSDK.getInstance(context).getApplicationPackageName() + " has been archived. Please request to un-archive it with the administrators of this product.");
            trackFirebaseEvent(event);
            return;
        }

        if (!whitelistedEvents.isEmpty() && !whitelistedEvents.contains(event.getName())) {
            LoggingUtil.e("Event " + event.getName() + " will not be sent because it has not been whitelisted by the Gamedock Backend. Visit the Gamedock Console to whitelist your event or contact a Gamedock representative.");
            trackFirebaseEvent(event);
            return;
        }

        try {
            boolean isPrivacyPolicyAccepted = GamedockSDK.getInstance(context).getStorageUtil().getBoolean(StorageUtil.Keys.PrivacyPolicyStatus, false);
            if (isPrivacyPolicyAccepted) {
                if (DeviceUtil.isAlphaNumeric(event.getName())) {
                    LoggingUtil.d("Tracking event: " + event.toString());

                    event = addDefaultParameters(event);

                    Handler handler;
                    if (GamedockSDK.getInstance(context).handler != null) {
                        handler = GamedockSDK.getInstance(context).handler;
                    } else if (GamedockSDK.getInstance(context).mainThreadHandler != null) {
                        handler = GamedockSDK.getInstance(context).mainThreadHandler;
                    } else {
                        LoggingUtil.e("Error sending event due to sending thread not being present: " + event.getName());
                        return;
                    }

                    final Event finalEvent = event;
                    handler.post(new Runnable() {
                        @Override
                        public void run() {
                            //Create a new Data Sender Job that handles the sending of the event
                            new NetworkJob().addJob(context, finalEvent, new EventActionListener() {
                                @Override
                                public void onResponse(ResponseEvent responseEvent) {
                                    String listenerId = UUID.randomUUID().toString();

                                    if (GamedockSDK.getInstance(context).actionListenerList.size() > 100) {

                                        int n = 50;
                                        int c = 0;
                                        Iterator<String> it = GamedockSDK.getInstance(context).actionListenerList.keySet().iterator();

                                        while (it.hasNext()) {
                                            if (c == n) break;
                                            c++;
                                            it.next();
                                            it.remove();
                                        }
                                    }
                                    GamedockSDK.getInstance(context).actionListenerList.put(listenerId, actionListener);

                                    GamedockSDK.getInstance(context).processResponseEvent(responseEvent, listenerId);
                                }
                            });
                        }
                    });
                } else {
                    LoggingUtil.e("Event name is invalid! Please only use a-z, A-Z, 0-9, -, _ characters for your event name!");
                }

                trackFirebaseEvent(event);

            } else {
                LoggingUtil.e("Gamedock SDK not initialised! Event with name: " + event.getName() + " will not be sent!");
            }
        } catch (Exception e) {
            e.printStackTrace();
        }
    }

    private void trackFirebaseEvent(Event event) {
        try {
            if (GamedockSDK.getInstance(context).firebaseInitialised && !isReservedEvent(event)) {
                Bundle firebaseParams = new Bundle();

                for (Map.Entry<String, JsonElement> entry : event.customData.entrySet()) {
                    if (entry.getValue().isJsonNull()) {
                        firebaseParams.putString(entry.getKey(), "null");
                    }

                    if (entry.getValue().isJsonPrimitive()) {
                        if (entry.getValue().getAsJsonPrimitive().isBoolean()) {
                            firebaseParams.putBoolean(entry.getKey(), entry.getValue().getAsJsonPrimitive().getAsBoolean());
                        } else if (entry.getValue().getAsJsonPrimitive().isString() && entry.getValue().getAsJsonPrimitive().getAsString().length() < 100) {
                            firebaseParams.putString(entry.getKey(), entry.getValue().getAsJsonPrimitive().getAsString());
                        } else if (entry.getValue().getAsJsonPrimitive().isNumber()) {
                            firebaseParams.putLong(entry.getKey(), entry.getValue().getAsJsonPrimitive().getAsLong());
                        }
                    }

                    if ((entry.getValue().isJsonArray() && entry.getValue().getAsJsonArray().size() > 0) || entry.getValue().isJsonObject()) {
                        Bundle values = getFirebaseBundleFromJson(entry.getValue());
                        firebaseParams.putBundle(entry.getKey(), values);
                    }
                }

                FirebaseAnalytics.getInstance(context).logEvent(event.getName(), firebaseParams);

                LoggingUtil.d("GamedockSDK Firebase Event: ###" + event.getName() + "### With Params: " + event.customData.toString());
            }
        } catch (Exception | NoClassDefFoundError e) {
            e.printStackTrace();
            LoggingUtil.e("Gamedock Firebase Module not included! If you want to use Firebase please include the gamedock-sdk-firebase dependency");
        }
    }


    /**
     * Method that is called internally if the event has been already queued but still needs to be sent.
     *
     * @param queuedEvent    The event that needs to be sent to the server.
     * @param actionListener The action listener needed in case you want to override the normal response processing flow.
     */
    public void trackEventQueued(Event queuedEvent, EventActionListener actionListener) {
        try {
            if (!(queuedEvent instanceof HeartbeatEvent)) {
                queuedEvent.setQueued(true);
                new NetworkJob().addJob(context, queuedEvent, actionListener);
            } else {
                LoggingUtil.i("Event cancelled: " + queuedEvent.toString());
            }
        } catch (Exception e) {
            e.printStackTrace();
        }
    }

    /**
     * Method that is used mainly by Unity in order to track custom events.
     * Also holds logic if the custom events are of specific nature such as IAP or tracking updatePlayerData.
     *
     * @param eventName  The name of the custom event
     * @param parameters Parameters that need to be added to the custom event
     */
    public void trackSpecialEvent(String eventName, JSONObject parameters, final EventActionListener eventActionListener) {
        Event event = new Event(context);
        event.setName(eventName);

        try {

            for (int i = 0; i < parameters.length(); i++) {
                String key = parameters.names().getString(i);

                if (parameters.get(key) instanceof String) {
                    event.addCustomData(key, parameters.getString(key));
                } else if (parameters.get(key) instanceof Integer) {
                    event.addCustomData(key, parameters.getInt(key));
                } else if (parameters.get(key) instanceof Boolean) {
                    event.addCustomData(key, parameters.getBoolean(key));
                } else if (parameters.get(key) instanceof Double) {
                    event.addCustomData(key, parameters.getDouble(key));
                } else if (parameters.get(key) instanceof Long) {
                    event.addCustomData(key, parameters.getLong(key));
                } else if (parameters.get(key) instanceof JSONObject) {
                    event.addCustomData(key, parameters.getJSONObject(key));
                } else if (parameters.get(key) instanceof JSONArray) {
                    event.addCustomData(key, parameters.getJSONArray(key));
                } else {
                    String invalid = "INVALID PARAMETER TYPE";
                    event.addCustomData(key, invalid);
                }
            }

            // If this is an iapPurchasedEvent then get the SKU and request the Price and Currency from the Google Play Store using the Google IAB library.
            if (eventName.toLowerCase().trim().equals("iappurchased")) {
                IAPEvent iapEvent = new IAPEvent(context);
                iapEvent.setIAPPurchasedEvent();
                iapEvent.customData = event.customData;

                String SKU = iapEvent.getCustomDataAsString("skuId");
                String transactionId = iapEvent.getCustomDataAsString("transactionId");

                if (SKU != null && transactionId != null) {
                    LoggingUtil.d("iappurchased detected, retrieving price and currency for SKU \"" + SKU + "\"");

                    GamedockSDK.getInstance(context).processIAPPurchaseEvent(iapEvent, SKU);
                } else {
                    LoggingUtil.d("iappurchased detected in test mode");
                    trackEvent(iapEvent, new EventActionListener() {
                        @Override
                        public void onResponse(ResponseEvent responseEvent) {
                            if (eventActionListener != null) {
                                eventActionListener.onResponse(responseEvent);
                            }

                            GamedockSDK.getInstance(context).sendDataToUnity("OnResponseReceived", responseEvent.toJSONString(false));
                        }
                    });
                }

            } else if (eventName.toLowerCase().trim().equals("updateplayerdata")) {

                if (event.customData.has("wallet")) {
                    try {
                        JSONObject jsonObject = new JSONObject(event.getCustomDataAsString("wallet"));
                        event.customData.remove("wallet");
                        event.addCustomData("wallet", jsonObject);
                    } catch (JSONException e) {
                        e.printStackTrace();
                    }
                }

                if (event.customData.has("inventory")) {
                    try {
                        JSONObject jsonObject = new JSONObject(event.getCustomDataAsString("inventory"));
                        event.customData.remove("inventory");
                        event.addCustomData("inventory", jsonObject);
                    } catch (JSONException e) {
                        e.printStackTrace();
                    }
                }

                boolean trackingOnly = event.getCustomDataAsBoolean("trackingOnly");
                event.addCustomData("trackingOnly", trackingOnly);

                trackEvent(event, eventActionListener);

            } else if (eventName.toLowerCase().trim().equals("requestrewardvideo")) {
                LoggingUtil.w("Ad part is removed from Gamedock SDK");
            }
             else {
                trackEvent(event, new EventActionListener() {
                    @Override
                    public void onResponse(ResponseEvent responseEvent) {
                        if (eventActionListener != null) {
                            eventActionListener.onResponse(responseEvent);
                        }
                        GamedockSDK.getInstance(context).sendDataToUnity("OnResponseReceived", responseEvent.toJSONString(false));

                        /*
                        if (responseEvent.getType().equals("advertisement") && responseEvent.getAction().equals("show")){
                            GamedockSDK.getInstance(context).sendDataToUnity("OnAdTrigger", responseEvent.toJSONString(false));
                        }
                        else{


                        }
                        */

                    }
                });
            }
        } catch (JSONException e) {
            e.printStackTrace();
        }
    }

    /**
     * Method that adds the default parameters that are included which each event.
     *
     * @param event The event to which the parameters will be added to.
     * @return The event that needs to be sent to the server.
     */
    @SuppressLint("HardwareIds")
    private Event addDefaultParameters(Event event) {
        //Event id
        event.setEventId(EventManager.getInstance(context).getEventId());
        event.addData("id", event.getEventId());

        //Event ts
        event.addData("ts", event.getTimestamp());

        //Device Id
        if (!GamedockSDK.getInstance(context).isCoppaEnabled()) {
            event.addData("deviceId", UserIDGenerator.getUniqueDeviceId(context));
        }

        //User Id
        event.addData("uid", UserIDGenerator.getGamedockUserId(context));

        //GCM Id (used for push notifications)
        event.addData("gcmId", GCMUtils.getGCMID(context));

        //Device language
        Locale locale;
        if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.N) {
            locale = context.getResources().getConfiguration().getLocales().get(0);
        } else {
            locale = context.getResources().getConfiguration().locale;
        }
        event.addData("locale", locale.toString());

        //application version
        event.addData("appVersion", GamedockSDK.getInstance(context).appVersion);

        //SDK version
        event.addData("apiVersion", BuildConfig.VERSION_NAME);

        //Plugin Name and Version
        String pluginName = GamedockSDK.getInstance(context).getStorageUtil().getString(StorageUtil.Keys.PluginName, null);
        String pluginVersion = GamedockSDK.getInstance(context).getStorageUtil().getString(StorageUtil.Keys.PluginVersion, null);
        if (pluginName != null && pluginVersion != null) {
            event.addData("pluginName", pluginName);
            event.addData("pluginVersion", pluginVersion);
        }

        //Operating System
        event.addData("os", "Android");

        //OS Version
        event.addData("osVersion", Build.VERSION.RELEASE);

        //Device Model
        event.addData("deviceModel", Build.MANUFACTURER + " " + Build.MODEL);

        //Package name
        event.addData("packageName", GamedockSDK.getInstance(context).getApplicationPackageName());

        //Timezone Offset
        Calendar cal = Calendar.getInstance(TimeZone.getDefault());
        String tzOffset = "" + ((cal.get(Calendar.ZONE_OFFSET) + cal.get(Calendar.DST_OFFSET)) / 60000);
        event.addData("timezoneOffset", tzOffset);

        //Total Time Opened
        event.addData("tto", SessionUtil.getTotalTimeOpen(context));

        //Session Id
        event.addData("sessionId", SessionUtil.getSessionId(context));

        //Session Time
        event.addData("sessionDuration", SessionUtil.getSessionDuration());

        //Store
        event.addData("store", GamedockSDK.getInstance(context).getStorageUtil().getString(StorageUtil.Keys.Store, null));

        //Google Advertising Id or Android Id
        String googleAdvertisingId = GamedockConfigManager.getInstance(context).getGoogleAdvertisingId();
        if (googleAdvertisingId != null && !GamedockSDK.getInstance(context).isCoppaEnabled()) {
            event.addData("googleAdvertisingId", googleAdvertisingId);
        }

        try {
            if (!GamedockSDK.getInstance(context).isCoppaEnabled()) {
                event.addData("androidId", Settings.Secure.getString(context.getContentResolver(), Settings.Secure.ANDROID_ID));
            }
        } catch (NullPointerException ignore) {
        }

        //Social User Id
        if (UserDataManager.getInstance(context).getUserData() != null) {
            String provider = UserDataManager.getInstance(context).getUserData().getProvider();
            String userId = UserDataManager.getInstance(context).getUserData().getUserID();
            if (provider != null && userId != null) {
                try {
                    JSONObject jsonObject = new JSONObject();
                    jsonObject.put("provider", provider);
                    jsonObject.put("userId", userId);

                    event.addData("externalUserId", jsonObject);
                } catch (JSONException e) {
                    e.printStackTrace();
                }
            }
        }

        //External User Ids
        try {
            JSONArray externalIdsList = new JSONArray();
            for (Map.Entry<String, String> entry : GamedockSDK.getInstance(context).externalIds.entrySet()) {
                JSONObject externalIdObject = new JSONObject();
                externalIdObject.put("provider", entry.getKey());
                externalIdObject.put("id", entry.getValue());

                externalIdsList.put(externalIdObject);
            }
            event.addData("externalIds", externalIdsList);
        } catch (JSONException ignore) {
        }

        return event;
    }

    /**
     * Method used for processing all the response events received from the Gamedock server.
     * Decides how each response will be processed and to which processing util class it will be sent to.
     *
     * @param responseEvent The response event received from the Gamedock server.
     * @param listenerId    The listener id associated with the actionListener attached to the initial sent event.
     */
    public void processResponseEvent(ResponseEvent responseEvent, final String listenerId) {
        EventActionListener actionListener = null;

        if (listenerId != null) {
            actionListener = GamedockSDK.getInstance(context).actionListenerList.get(listenerId);
        }

        if (responseEvent != null) {

            LoggingUtil.d("Event Received: " + responseEvent);

            if ((actionListener != null && !actionListener.skipProcessing) || actionListener == null) {
                responseEvent.processData(context);

                if (actionListener != null) {
                    actionListener.onResponse(responseEvent);
                }
                if (GamedockSDK.getInstance(context).globalEventActionListener != null) {
                    GamedockSDK.getInstance(context).globalEventActionListener.onResponse(responseEvent);
                }

                GamedockSDK.getInstance(context).actionListenerList.remove(listenerId);
            } else {
                actionListener.onResponse(responseEvent);

                if (GamedockSDK.getInstance(context).globalEventActionListener != null) {
                    GamedockSDK.getInstance(context).globalEventActionListener.onResponse(responseEvent);
                }

                GamedockSDK.getInstance(context).actionListenerList.remove(listenerId);
            }
        }
    }

    private Bundle getFirebaseBundleFromJson(JsonElement json) {
        Bundle bundle = new Bundle();

        if (json.isJsonObject()) {
            for (Map.Entry<String, JsonElement> entry : json.getAsJsonObject().entrySet()) {
                if (entry.getValue().isJsonNull()) {
                    bundle.putString(entry.getKey(), "null");
                }

                if (entry.getValue().isJsonPrimitive()) {
                    if (entry.getValue().getAsJsonPrimitive().isBoolean()) {
                        bundle.putBoolean(entry.getKey(), entry.getValue().getAsJsonPrimitive().getAsBoolean());
                    } else if (entry.getValue().getAsJsonPrimitive().isString() && entry.getValue().getAsJsonPrimitive().getAsString().length() < 100) {
                        bundle.putString(entry.getKey(), entry.getValue().getAsJsonPrimitive().getAsString());
                    } else if (entry.getValue().getAsJsonPrimitive().isNumber()) {
                        bundle.putLong(entry.getKey(), entry.getValue().getAsJsonPrimitive().getAsLong());
                    }
                }

                if (entry.getValue().isJsonArray() || entry.getValue().isJsonObject()) {
                    Bundle values = getFirebaseBundleFromJson(entry.getValue());
                    bundle.putBundle(entry.getKey(), values);
                }
            }
        } else if (json.isJsonArray()) {
            int keyCount = 0;
            for (JsonElement entry : json.getAsJsonArray()) {
                String entryHash = String.valueOf("key_" + keyCount);
                if (entry.isJsonNull()) {
                    bundle.putString(entryHash, "null");
                }

                if (entry.isJsonPrimitive()) {
                    if (entry.getAsJsonPrimitive().isBoolean()) {
                        bundle.putBoolean(entryHash, entry.getAsJsonPrimitive().getAsBoolean());
                    } else if (entry.getAsJsonPrimitive().isString() && entry.getAsJsonPrimitive().getAsString().length() < 100) {
                        bundle.putString(entryHash, entry.getAsJsonPrimitive().getAsString());
                    } else if (entry.getAsJsonPrimitive().isNumber()) {
                        bundle.putLong(entryHash, entry.getAsJsonPrimitive().getAsLong());
                    }
                }

                if (entry.isJsonArray() || entry.isJsonObject()) {
                    Bundle values = getFirebaseBundleFromJson(entry);
                    bundle.putBundle(entryHash, values);
                }

                keyCount++;
            }
        }


        return bundle;
    }

    /**
     * Retrieves the 100 most recent events sent by the SDK.
     *
     * @return Returns a list of events that have been sent recently.
     */
    private LimitedSizeQueue<Event> retrieveRecentEvents() {
        LimitedSizeQueue<Event> recentSentEvents = new LimitedSizeQueue<>(100);

        String recentEvents = GamedockSDK.getInstance(context).getStorageUtil().getString(StorageUtil.Keys.SpilSDKRecentEventList, null);

        if (recentEvents != null) {
            try {
                recentSentEvents.list = GamedockSDK.getInstance(context).getGson().fromJson(recentEvents, new TypeToken<ArrayList<Event>>() {
                }.getType());
            } catch (JsonSyntaxException e) {
                recentSentEvents.list = new ArrayList<>();
            }
        } else {
            recentSentEvents.list = new ArrayList<>();
        }

        return recentSentEvents;
    }

    /**
     * Saves the recent events list to the shared preferences.
     */
    public synchronized void saveRecentEvents() {
        GamedockSDK.getInstance(context).getStorageUtil().putString(StorageUtil.Keys.SpilSDKRecentEventList, GamedockSDK.getInstance(context).getGson().toJson(recentSentEvents.list));
    }

    /**
     * Method that retrieves the recent event list.
     *
     * @return The recent event list contained in a {@link LimitedSizeQueue}.
     */
    public synchronized LimitedSizeQueue<Event> getRecentSentEvents() {
        return recentSentEvents;
    }

    /**
     * Method that adds to the recent event list
     *
     * @param event The event that has to be added.
     */
    public synchronized void addToRecentSentEvents(Event event) {
        getRecentSentEvents().add(event);
    }

    /**
     * Method that returns the persistence event list bucket at the specified position.
     *
     * @param position The position from which the list bucket has to be retrieve.
     * @return The stored recent event list bucket or a new list.
     */
    public synchronized ArrayList<Event> getPersistenceEventsList(int position) {
        ArrayList<Event> persistenceEventDataList;

        try {
            String persistenceEvents = GamedockSDK.getInstance(context).getStorageUtil().getString((StorageUtil.Keys.SpilSDKPersistenceSpilEvents + position), null);

            if (persistenceEvents != null) {
                try {
                    persistenceEventDataList = GamedockSDK.getInstance(context).getGson().fromJson(persistenceEvents, new TypeToken<ArrayList<Event>>() {
                    }.getType());
                    return persistenceEventDataList;
                } catch (JsonSyntaxException e) {
                    persistenceEventDataList = new ArrayList<>();
                    return persistenceEventDataList;
                }
            } else {
                persistenceEventDataList = new ArrayList<>();
                return persistenceEventDataList;
            }
        } catch (Error | Exception e) {
            e.printStackTrace();
            persistenceEventDataList = new ArrayList<>();
            GamedockSDK.getInstance(context).getStorageUtil().remove((StorageUtil.Keys.SpilSDKPersistenceSpilEvents + position));
            return persistenceEventDataList;
        }
    }

    /**
     * Saves the persistence event list to the shared preferences.
     */
    public synchronized void savePersistenceEventsList(ArrayList<Event> persistenceEventDataList, int position) {
        try {
            String persistenceListString = GamedockSDK.getInstance(context).getGson().toJson(persistenceEventDataList);
            GamedockSDK.getInstance(context).getStorageUtil().putString((StorageUtil.Keys.SpilSDKPersistenceSpilEvents + position), persistenceListString);
        } catch (Error | Exception e) {
            e.printStackTrace();
            persistenceEventDataList.clear();
            String persistenceListString = GamedockSDK.getInstance(context).getGson().toJson(persistenceEventDataList);
            GamedockSDK.getInstance(context).getStorageUtil().putString((StorageUtil.Keys.SpilSDKPersistenceSpilEvents + position), persistenceListString);
        }
    }

    /**
     * Method that clears the persistence event list at the given position.
     *
     * @param persistenceEventDataList The event list that has to be cleared locally and in storage.
     * @param position                 The position at which the list bucket is located.
     */
    public synchronized void clearPersistenceEventDataList(ArrayList<Event> persistenceEventDataList, int position) {
        persistenceEventDataList.clear();
        GamedockSDK.getInstance(context).getStorageUtil().remove((StorageUtil.Keys.SpilSDKPersistenceSpilEvents + position));
        if (persistenceListBucketSize > 1) {
            persistenceListBucketSize--;
            GamedockSDK.getInstance(context).getStorageUtil().putInt(StorageUtil.Keys.SpilSDKPersistenceSize, persistenceListBucketSize);
        }
    }

    /**
     * Method that adds an event to the current list bucket.
     * If the bucket reaches 50 events stored it is saved in storage and a new bucket is created.
     *
     * @param event The event that has to be stored in the list.
     */
    public synchronized void addToPersistenceEventDataList(Event event) {
        ArrayList<Event> persistenceEventDataList = getPersistenceEventsList(persistenceListBucketSize);
        if (persistenceEventDataList.size() < 51) {
            persistenceEventDataList.add(event);
            savePersistenceEventsList(persistenceEventDataList, persistenceListBucketSize);
        } else {
            savePersistenceEventsList(persistenceEventDataList, persistenceListBucketSize);
            persistenceListBucketSize++;
            GamedockSDK.getInstance(context).getStorageUtil().putInt(StorageUtil.Keys.SpilSDKPersistenceSize, persistenceListBucketSize);
            persistenceEventDataList = new ArrayList<>();
            persistenceEventDataList.add(event);
            savePersistenceEventsList(persistenceEventDataList, persistenceListBucketSize);
        }
    }

    /**
     * Retrieves the number of buckets stored on the disk.
     *
     * @return
     */
    public synchronized int getPersistenceListSize() {
        try {
            return GamedockSDK.getInstance(context).getStorageUtil().getInt(StorageUtil.Keys.SpilSDKPersistenceSize, 1);
        } catch (Error | Exception e) {
            return 1;
        }
    }

    /**
     * Method that retrieves the recent IAP event list or creates a new one.
     *
     * @return The recent IAP event list.
     */
    public LimitedSizeQueue<Event> getRecentIAPEvents() {
        if (recentIAPEvents.list == null) {
            recentIAPEvents.list = new ArrayList<>();
            return recentIAPEvents;
        }

        return recentIAPEvents;
    }

    /**
     * Method that sets the recent IAP list.
     *
     * @param recentIAPEvents The recent IAP list that has to be set.
     */
    public void setRecentIAPEvents(LimitedSizeQueue<Event> recentIAPEvents) {
        this.recentIAPEvents = recentIAPEvents;
    }

    public long getEventId() {
        long eventId = GamedockSDK.getInstance(context).getStorageUtil().getLong(StorageUtil.Keys.EventId, 0);
        GamedockSDK.getInstance(context).getStorageUtil().putLong(StorageUtil.Keys.EventId, eventId + 1);

        return eventId;
    }

    public long getSendId() {
        long sendId = GamedockSDK.getInstance(context).getStorageUtil().getLong(StorageUtil.Keys.SendId, 0);
        GamedockSDK.getInstance(context).getStorageUtil().putLong(StorageUtil.Keys.SendId, sendId + 1);

        return sendId;
    }

    /**
     * Events that will not be sent to Firebase
     * @param event
     * @return
     */
    public static boolean isReservedEvent(Event event) {
        String name = event.getName();

        return  name.equals(AssetBundlesEvent.RequestAssetBundles) ||
                name.equals(ConfigEvent.RequestConfig) ||
                name.equals(GameDataEvent.RequestGameData) || name.equals(GameDataEvent.SetCurrencyLimit) || name.equals(GameDataEvent.SetItemLimit) ||
                name.equals(HeartbeatEvent.HeartBeat) ||
                name.equals(IAPEvent.ValidateSubscription) ||
                name.equals(InitializeSDKEvent.InitializeSDK) || name.equals(InitializeSDKEvent.UserChanged) ||
                name.equals(MissionEvent.RequestMissionConfig) || name.equals(MissionEvent.UpdateUserMissionData) ||
                name.equals(MoreAppsEvent.RequestMoreApps) ||
                name.equals(OverlayEvent.RequestSplashscreen) || name.equals(OverlayEvent.RequestDailyBonus) || name.equals(OverlayEvent.DroppedSplashscreen) ||
                name.equals(PromotionEvent.RequestPromotions) || name.equals(PromotionEvent.ShowPromotionScreen) || name.equals(PromotionEvent.BoughtPromotion) ||
                name.equals(RewardEvent.ClaimTokenEvent) ||
                name.equals(ServerDataEvent.RequestServerTime) ||
                name.equals(SocialLoginEvent.UserLogin) || name.equals(SocialLoginEvent.UserLogout) || name.equals(SocialLoginEvent.UserPlayAsGuest) ||
                name.equals(TierEvent.RequestTieredEvent) || name.equals(TierEvent.UpdateTierProgress) || name.equals(TierEvent.ClaimTierReward) || name.equals(TierEvent.ShowTierProgress) ||
                name.equals(UserDataEvent.RequestUserData) || name.equals(UserDataEvent.MergeUserData) || name.equals(UserDataEvent.RequestOtherUsersGameState) || name.equals(UserDataEvent.UpdateGameState) || name.equals(UserDataEvent.UpdatePlayerData) ||
                name.equals(AdEvents.AdvertisementInitialized) ||
                name.equals(LocalizationEvent.RequestLocalization) || name.equals(LocalizationEvent.LocalizationKeyNotFound) ||
                name.equals("install") || name.equals("update") || name.equals("privacyChanged") ||
                name.equals("sessionStart") || name.equals("sessionStop") ||
                name.equals("error");
    }




    public ArrayList<String> getWhitelistedEvents() {
        return whitelistedEvents;
    }

    public void resetEventManager() {
        mInstance = null;
    }
}
