package io.gamedock.sdk.events;

import android.content.Context;

import com.google.gson.JsonArray;
import com.google.gson.JsonObject;
import com.google.gson.JsonParser;

import org.json.JSONArray;
import org.json.JSONObject;

import java.util.Locale;

import io.gamedock.sdk.GamedockSDK;
import io.gamedock.sdk.utils.logging.LoggingUtil;

/**
 * Class that holds all the attributes of a event.
 * The event id and timestamp are created automatically.
 */
public class Event {

    private static String TAG = "Event";
    private long eventId;
    private String name;
    public JsonObject data;
    public JsonObject customData;
    private boolean queued;
    private long timestamp;
    private String retryReason;
    private JsonParser jsonParser;
    private String spilToken;

    private boolean isNetworkErrorHandled = false;

    public Event() {

    }

    public Event(Context context) {
        data = new JsonObject();
        customData = new JsonObject();
        timestamp = System.currentTimeMillis();
        queued = false;
        jsonParser = new JsonParser();
        retryReason = "";
        spilToken = GamedockSDK.getInstance(context).getGamedockToken();
    }

    public static String getTAG() {
        return TAG;
    }

    public static void setTAG(String TAG) {
        Event.TAG = TAG;
    }

    public String toString() {
        String logCustomData = "";
        if (customData != null) {
            logCustomData = customData.toString();
        }
        return String.format(Locale.ENGLISH, "Event: name=%s, id=%s, queued=%b, ts=%d, retryReason=%s, customData=%s",
                name, eventId, queued, timestamp, retryReason, logCustomData);
    }

    /**
     * Method that adds data based on a {@link String} value.
     *
     * @param key
     * @param value
     */
    public void addData(String key, String value) {
        try {
            if (data == null) data = new JsonObject();
            data.addProperty(key, value);
        } catch (Exception e) {
            LoggingUtil.e("error adding data " + e.getMessage());
        }
    }

    /**
     * Method that adds data based on a {@code boolean} value
     *
     * @param key
     * @param value
     */
    public void addData(String key, boolean value) {
        try {
            if (data == null) data = new JsonObject();
            data.addProperty(key, value);
        } catch (Exception e) {
            LoggingUtil.e("error adding data " + e.getMessage());
        }
    }

    /**
     * Method that adds data based on a {@code long} value
     *
     * @param key
     * @param value
     */
    public void addData(String key, long value) {
        try {
            if (data == null) data = new JsonObject();
            data.addProperty(key, value);
        } catch (Exception e) {
            LoggingUtil.e("error adding data " + e.getMessage());
        }
    }

    /**
     * Method that adds data based on a {@link JSONObject}
     *
     * @param key
     * @param value
     */
    public void addData(String key, JSONObject value) {
        try {
            if (data == null) data = new JsonObject();
            JsonObject finalJson = (JsonObject) jsonParser.parse(value.toString());
            data.add(key, finalJson);
        } catch (Exception e) {
            LoggingUtil.e("error adding data " + e.getMessage());
        }
    }

    /**
     * Method that adds JSON Array data to the data section of the event.
     *
     * @param key
     * @param value
     */
    public void addData(String key, JSONArray value) {
        try {
            if (data == null) {
                data = new JsonObject();
            }
            JsonArray finalJson = (JsonArray) jsonParser.parse(value.toString());
            data.add(key, finalJson);
        } catch (Exception e) {
            LoggingUtil.e("error adding data " + e.getMessage());
        }
    }

    /**
     * Method that adds {@link String} data to the custom data section of the event.
     *
     * @param key
     * @param value
     */
    public void addCustomData(String key, String value) {
        try {
            if (customData == null) customData = new JsonObject();
            customData.addProperty(key, value);
        } catch (Exception e) {
            LoggingUtil.e("error adding data " + e.getMessage());
        }
    }

    /**
     * Method that adds {@code int} data to the custom data section of the event.
     *
     * @param key
     * @param value
     */
    public void addCustomData(String key, int value) {
        try {
            if (customData == null) customData = new JsonObject();
            customData.addProperty(key, value);
        } catch (Exception e) {
            LoggingUtil.e("error adding data " + e.getMessage());
        }
    }

    /**
     * Method that adds {@code boolean} data to the custom data section of the event.
     *
     * @param key
     * @param value
     */
    public void addCustomData(String key, boolean value) {
        try {
            if (customData == null) customData = new JsonObject();
            customData.addProperty(key, value);
        } catch (Exception e) {
            LoggingUtil.e("error adding data " + e.getMessage());
        }
    }

    /**
     * Method that adds {@code double} data to the custom data section of the event.
     *
     * @param key
     * @param value
     */
    public void addCustomData(String key, double value) {
        try {
            if (customData == null) customData = new JsonObject();
            customData.addProperty(key, value);
        } catch (Exception e) {
            LoggingUtil.e("error adding data " + e.getMessage());
        }
    }

    /**
     * Method that adds {@code long} data to the custom data section of the event.
     *
     * @param key
     * @param value
     */
    public void addCustomData(String key, long value) {
        try {
            if (customData == null) customData = new JsonObject();
            customData.addProperty(key, value);
        } catch (Exception e) {
            LoggingUtil.e("error adding data " + e.getMessage());
        }
    }

    /**
     * Method that adds JSON Object data to the custom data section of the event.
     *
     * @param key
     * @param value
     */
    public void addCustomData(String key, JSONObject value) {
        try {
            if (customData == null) customData = new JsonObject();

            JsonObject finalJson = (JsonObject) jsonParser.parse(value.toString());
            customData.add(key, finalJson);
        } catch (Exception e) {
            LoggingUtil.e("error adding data " + e.getMessage());
        }
    }

    /**
     * Method that adds JSON Array data to the custom data section of the event.
     *
     * @param key
     * @param value
     */
    public void addCustomData(String key, JSONArray value) {
        try {
            if (customData == null) {
                customData = new JsonObject();
            }
            JsonArray finalJson = (JsonArray) jsonParser.parse(value.toString());
            customData.add(key, finalJson);
        } catch (Exception e) {
            LoggingUtil.e("error adding data " + e.getMessage());
        }
    }

    /**
     * Method that adds JsonObject data to the custom data section of the event.
     *
     * @param key
     * @param value
     */
    public void addCustomData(String key, JsonObject value) {
        try {
            if (customData == null) {
                customData = new JsonObject();
            }
            customData.add(key, value);
        } catch (Exception e) {
            LoggingUtil.e("error adding data " + e.getMessage());
        }
    }

    /**
     * Method that adds JsonArray data to the custom data section of the event.
     *
     * @param key
     * @param value
     */
    public void addCustomData(String key, JsonArray value) {
        try {
            if (customData == null) {
                customData = new JsonObject();
            }
            customData.add(key, value);
        } catch (Exception e) {
            LoggingUtil.e("error adding data " + e.getMessage());
        }
    }

    /**
     * Method that retrieves a specific value from the data part of the event, given a key.
     *
     * @param key The key for which the value has to be retrieved.
     * @return The value from the data part of the event.
     */
    public String getData(String key) {
        //LoggingUtil.v("Called Event.getData(String key)");

        try {

            if (data == null || !data.has(key)) return null;

            return data.get(key).getAsString();
        } catch (Exception e) {
            LoggingUtil.e("error getting data " + e.getMessage());
        }
        return null;
    }

    /**
     * Method that retrieves a specific value from the custom part of the event, given a key.
     *
     * @param key The key for which the value has to be retrieved.
     * @return The value from the data part of the event in {@link String} format.
     */
    public String getCustomDataAsString(String key) {
        //LoggingUtil.v("Called Event.getCustomData(String key)");

        try {

            if (customData == null || !customData.has(key)) return null;

            return customData.get(key).getAsString();
        } catch (Exception e) {
            LoggingUtil.e("error getting data " + e.getMessage());
        }
        return null;
    }

    /**
     * Method that retrieves a specific value from the custom part of the event, given a key.
     *
     * @param key The key for which the value has to be retrieved.
     * @return The value from the data part of the event in {@code int} format.
     */
    public int getCustomDataAsInt(String key) {
        try {
            if (customData == null || !customData.has(key)) return 0;

            return customData.get(key).getAsInt();
        } catch (Exception e) {
            LoggingUtil.e("error getting data " + e.getMessage());
        }
        return 0;
    }

    /**
     * Method that retrieves a specific value from the custom part of the event, given a key.
     *
     * @param key The key for which the value has to be retrieved.
     * @return The value from the data part of the event in {@code boolean} format.
     */
    public boolean getCustomDataAsBoolean(String key) {
        try {
            return !(customData == null || !customData.has(key)) && customData.get(key).getAsBoolean();
        } catch (Exception e) {
            LoggingUtil.e("error getting data " + e.getMessage());
        }
        return false;
    }

    // Getters and setters

    public String getName() {
        return name;
    }

    public void setName(String name) {
        this.name = name;
    }

    public long getEventId() {
        return eventId;
    }

    public void setEventId(long eventId) {
        this.eventId = eventId;
    }

    public boolean isQueued() {
        return queued;
    }

    public void setQueued(boolean queued) {
        this.queued = queued;
    }

    public long getTimestamp() {
        return timestamp;
    }

    public void setTimestamp(long timestamp) {
        this.timestamp = timestamp;
    }

    public String getRetryReason() {
        return retryReason;
    }

    public boolean hasRetryReason() {
        return !"".equals(retryReason);
    }

    public String getSpilToken() {
        return spilToken;
    }

    public void setRetryReason(String reason) {
        LoggingUtil.d("Event setRetryReason: " + reason);
        this.retryReason = reason;
    }

    public boolean isNetworkErrorHandled() {
        return isNetworkErrorHandled;
    }

    public void setNetworkErrorHandled(boolean networkErrorHandled) {
        isNetworkErrorHandled = networkErrorHandled;
    }

    /**
     * Dummy method.
     * Check implementation of this method in each Event object
     *
     * @param context The activity context.
     */
    public void handleNetworkError(Context context) {

    }
}
