package io.gamedock.sdk.dailybonus;

import com.unity3d.player.UnityPlayer;

import org.json.JSONException;
import org.json.JSONObject;

import io.gamedock.sdk.utils.error.ErrorCodes;
import io.gamedock.sdk.utils.logging.LoggingUtil;

/**
 * Class that handles the Daily Bonus Callbacks.
 * Notifies the application when the Daily Bonus has been received and processed by the SDK.
 * Informs the application if an error occurred while processing the Game State values.
 */
public class DailyBonusCallbacks {

    private OnDailyBonusListener dailyBonusListener;

    public DailyBonusCallbacks() {
        dailyBonusListener = null;
    }

    public DailyBonusCallbacks(OnDailyBonusListener dailyBonusListener) {
        this.dailyBonusListener = dailyBonusListener;
    }

    /**
     * Method called when the Daily Bonus Web View has been opened.
     */
    public void dailyBonusOpen() {
        if (dailyBonusListener != null) {
            dailyBonusListener.DailyBonusOpen();
        } else {
            try {
                UnityPlayer.UnitySendMessage("GamedockSDK", "DailyBonusOpen", "");
            } catch (NoClassDefFoundError e) {
                LoggingUtil.w("You need to register the DailyBonusListener in order to receive information or something went wrong with Unity");
            }
        }
    }

    /**
     * Method called when the Daily Bonus Web View has been closed.
     */
    public void dailyBonusClosed() {
        if (dailyBonusListener != null) {
            dailyBonusListener.DailyBonusClosed();
        } else {
            try {
                UnityPlayer.UnitySendMessage("GamedockSDK", "DailyBonusClosed", "");
            } catch (NoClassDefFoundError e) {
                LoggingUtil.w("You need to register the DailyBonusListener in order to receive information or something went wrong with Unity");
            }
        }
    }

    /**
     * Method called when the Daily Bonus can be shown.
     */
    public void dailyBonusAvailable(String type) {
        if (dailyBonusListener != null) {
            dailyBonusListener.DailyBonusAvailable(type);
        } else {
            try {
                if (type == null) {
                    type = "";
                }
                UnityPlayer.UnitySendMessage("GamedockSDK", "DailyBonusAvailable", type);
            } catch (NoClassDefFoundError e) {
                LoggingUtil.w("You need to register the DailyBonusListener in order to receive information or something went wrong with Unity");
            }
        }
    }

    /**
     * Method called when the Daily Bonus Web View cannot be loaded.
     */
    public void dailyBonusNotAvailable() {
        if (dailyBonusListener != null) {
            dailyBonusListener.DailyBonusNotAvailable();
        } else {
            try {
                UnityPlayer.UnitySendMessage("GamedockSDK", "DailyBonusNotAvailable", "");
            } catch (NoClassDefFoundError e) {
                LoggingUtil.w("You need to register the DailyBonusListener in order to receive information or something went wrong with Unity");
            }
        }
    }

    /**
     * Method that is called when an error occurred while processing the Daily Bonus data.
     * Uses the {@link ErrorCodes} class do define the error.
     */
    public void dailyBonusError(ErrorCodes error) {
        if (dailyBonusListener != null) {
            dailyBonusListener.DailyBonusError(error);
        } else {
            try {
                JSONObject jsonObject = new JSONObject();
                jsonObject.put("id", error.getId());
                jsonObject.put("name", error.getName());
                jsonObject.put("message", error.getMessage());

                UnityPlayer.UnitySendMessage("GamedockSDK", "DailyBonusError", jsonObject.toString());
            } catch (NoClassDefFoundError | JSONException e) {
                LoggingUtil.w("You need to register the DailyBonusListener in order to receive information or something went wrong with Unity");
            }
        }
    }

    /**
     * Method used to send the external rewards towards Unity. Rewards that cannot be processed by the SDK.
     *
     * @param rewardData JSON array of reward data.
     */
    public void dailyBonusReward(String rewardData) {
        if (dailyBonusListener != null) {
            dailyBonusListener.DailyBonusReward(rewardData);
        } else {
            try {
                JSONObject jsonObject = new JSONObject();
                jsonObject.put("data", rewardData);

                UnityPlayer.UnitySendMessage("GamedockSDK", "DailyBonusReward", jsonObject.toString());
            } catch (NoClassDefFoundError | JSONException e) {
                LoggingUtil.w("You need to register the DailyBonusListener in order to receive information or something went wrong with Unity");
            }
        }
    }
}
