package io.gamedock.sdk.config.firebase;

import android.content.Context;

import com.google.android.gms.tasks.OnCompleteListener;
import com.google.android.gms.tasks.RuntimeExecutionException;
import com.google.android.gms.tasks.Task;
import com.google.firebase.remoteconfig.FirebaseRemoteConfig;

import org.json.JSONException;
import org.json.JSONObject;

import java.util.HashMap;
import java.util.Map;

import io.gamedock.sdk.GamedockSDK;
import io.gamedock.sdk.utils.logging.LoggingUtil;

/**
 * Class that handles the Firebase Remote Config implementation.
 */
public class FirebaseRemoteConfigManager {
    public static void setDefaultParameterValues(String defaultValues) {
        try {
            JSONObject defaultValuesJSON = new JSONObject(defaultValues);
            Map<String, Object> defaultValuesMap = new HashMap<>();

            for (int i = 0; i < defaultValuesJSON.names().length(); i++) {
                String key = defaultValuesJSON.names().getString(i);
                Object value = null;
                if (defaultValuesJSON.get(key) instanceof JSONObject) {
                    value = defaultValuesJSON.getJSONObject(key).toString();
                } else {
                    value = defaultValuesJSON.get(key);
                }
                defaultValuesMap.put(key, value);
            }

            FirebaseRemoteConfig.getInstance().setDefaultsAsync(defaultValuesMap);
        } catch (NoClassDefFoundError | JSONException e) {
            e.printStackTrace();
        }
    }

    /**
     * Method used to fetch the Firebase remote configuration from the Firebase backend.
     *
     * @param context                The context of the Activity.
     * @param cacheExpirationSeconds The cache expiration value that Firebase takes into account on. It is used to determine if a server request will be done or the local values will be provided.
     */
    public static void fetchFirebaseRemoteConfig(final Context context, long cacheExpirationSeconds) {
        if (cacheExpirationSeconds == 0) {
            cacheExpirationSeconds = 43200;
        }

        try {
            FirebaseRemoteConfig.getInstance().fetchAndActivate().addOnCompleteListener(new OnCompleteListener<Boolean>() {
                @Override
                public void onComplete(@androidx.annotation.NonNull Task<Boolean> task) {
                    try {
                        if (task.isSuccessful()) {
                            GamedockSDK.getInstance(context).getConfigDataCallbacks().firebaseRemoteConfigUpdated();
                        } else {
                            if (task.getException() != null) {
                                LoggingUtil.d(task.getException().toString());
                            }
                        }
                    } catch (RuntimeExecutionException e) {
                        e.printStackTrace();
                    }
                }
            });
        } catch (NoClassDefFoundError | Exception e) {
            e.printStackTrace();
        }
    }

    public static boolean getBoolean(String key, String namespace) {
        if (namespace != null && !namespace.equals("null")) {
            return FirebaseRemoteConfig.getInstance().getBoolean(key);
        } else {
            return FirebaseRemoteConfig.getInstance().getBoolean(key);
        }
    }

    public static double getDouble(String key, String namespace) {
        if (namespace != null && !namespace.equals("null")) {
            return FirebaseRemoteConfig.getInstance().getDouble(key);
        } else {
            return FirebaseRemoteConfig.getInstance().getDouble(key);
        }
    }

    public static long getLong(String key, String namespace) {
        if (namespace != null && !namespace.equals("null")) {
            return FirebaseRemoteConfig.getInstance().getLong(key);
        } else {
            return FirebaseRemoteConfig.getInstance().getLong(key);
        }
    }

    public static String getString(String key, String namespace) {
        if (namespace != null && !namespace.equals("null")) {
            return FirebaseRemoteConfig.getInstance().getString(key);
        } else {
            return FirebaseRemoteConfig.getInstance().getString(key);
        }
    }
}
