package io.gamedock.sdk.config;

import android.content.Context;

import androidx.annotation.VisibleForTesting;

import org.json.JSONException;
import org.json.JSONObject;

import java.io.IOException;
import java.io.InputStream;

import io.gamedock.sdk.GamedockSDK;
import io.gamedock.sdk.events.internal.ConfigEvent;
import io.gamedock.sdk.utils.assets.FileAssetsUtil;
import io.gamedock.sdk.utils.features.FeaturesUtil;
import io.gamedock.sdk.utils.logging.LoggingUtil;
import io.gamedock.sdk.utils.storage.StorageUtil;

/**
 * Utility class that handles all the logic for the configuration management part of the SDK.
 * It processes the config from a JSON format and saves it in the shared preferences for later retrieval via Unity/Native App.
 * Loads a default configuration file stored in the assets folder.
 */
public class ConfigManager {
    private static final Object lock = new Object();

    private static volatile ConfigManager mInstance = null;
    private Context context;

    private final FileAssetsUtil fileAssetsUtil;

    public static final String FEATURE_NAME = "config";

    private ConfigManager(Context context, FileAssetsUtil fileAssetsUtil) {
        this.context = context;
        this.fileAssetsUtil = fileAssetsUtil;
    }

    public static ConfigManager getInstance(Context context) {
        if (mInstance == null) {
            synchronized (lock) {
                if (mInstance == null) {
                    mInstance = new ConfigManager(context, GamedockSDK.getFileAssetsUtil());
                }
            }
        }
        return mInstance;
    }

    /**
     * Method that is called internally when a response has been received from the server.
     * Processes the JSON response from the Gamedock backend.
     *
     * @param configJSON The received JSON configuration.
     */
    public void processConfigJSON(String configJSON) {
        if (configJSON != null) {
            GamedockSDK.getInstance(context).getStorageUtil().putString(StorageUtil.Keys.GameConfig, configJSON);
            GamedockSDK.getInstance(context).getConfigDataCallbacks().configDataUpdated();
        }
    }

    /**
     * Method that requests the configuration from the Gamedock backend.
     */
    public void requestConfig() {
        if (!FeaturesUtil.isFeatureEnabled(FEATURE_NAME)) {
            return;
        }

        ConfigEvent configEvent = new ConfigEvent(context);
        configEvent.setRequestConfig();

        GamedockSDK.getInstance(context).trackEvent(configEvent, null);
    }

    /**
     * Method that retrieves a configuration value given the requested key.
     * The method checks if a configuration has been saved so far (received from the server) and if not it initialises the local configuration file.
     *
     * @param key The key for which the value has to retrieve for.
     * @return The requested configuration value. The return value is a String or a JSON string.
     */
    public String getConfigValue(String key) {
        if (!FeaturesUtil.isFeatureEnabled(FEATURE_NAME)) {
            return "";
        }

        StorageUtil storageUtil = GamedockSDK.getInstance(context).getStorageUtil();
        String config = storageUtil.getString(StorageUtil.Keys.GameConfig, null);

        if (config != null) {
            try {
                JSONObject object = new JSONObject(config);

                if (key.equals("androidSdkConfig") && !object.has("androidSdkConfig")) {
                    return loadSDKConfigFromAssets(context, fileAssetsUtil);
                }

                return object.getString(key);
            } catch (JSONException e) {
                LoggingUtil.w("Error retrieving values for key: " + key);
            }
        } else {
            config = loadGameConfigFromAssets(context);
            if (config != null) {
                storageUtil.putString(StorageUtil.Keys.GameConfig, config);
                try {
                    JSONObject object = new JSONObject(config);
                    return object.getString(key);
                } catch (JSONException e) {
                    LoggingUtil.w("Error retrieving values for key: " + key);
                }
            }
        }

        return null;
    }

    /**
     * Method that returns the full configuration JSON in a String format.
     * The method checks if a configuration has been saved so far (received from the server) and if not it initialises the local configuration file.
     *
     * @return The String version of the JSON configuration.
     */
    public String getConfigAll(boolean withSdkConfig) {
        if (!FeaturesUtil.isFeatureEnabled(FEATURE_NAME)) {
            return "{}";
        }

        StorageUtil storageUtil = GamedockSDK.getInstance(context).getStorageUtil();
        String config = storageUtil.getString(StorageUtil.Keys.GameConfig, null);

        if (config != null) {
            try {
                JSONObject configJSON = new JSONObject(config);

                if (!withSdkConfig) {
                    if (configJSON.has("androidSdkConfig")) {
                        configJSON.remove("androidSdkConfig");
                    }

                    if (configJSON.has("iosSdkConfig")) {
                        configJSON.remove("iosSdkConfig");
                    }
                }
                return configJSON.toString();
            } catch (JSONException e) {
                e.printStackTrace();
            }
        } else {
            config = loadGameConfigFromAssets(context);
            if (config != null) {
                storageUtil.putString(StorageUtil.Keys.GameConfig, config);
                try {
                    JSONObject configJSON = new JSONObject(config);

                    if (!withSdkConfig) {
                        if (configJSON.has("androidSdkConfig")) {
                            configJSON.remove("androidSdkConfig");
                        }

                        if (configJSON.has("iosSdkConfig")) {
                            configJSON.remove("iosSdkConfig");
                        }
                    }
                    return configJSON.toString();
                } catch (JSONException e) {
                    e.printStackTrace();
                }
            }
        }

        return null;
    }

    /**
     * Method that is used internally to load the configuration JSON from the file stored locally.
     * The name of the file is "defaultGameConfig.json".
     * This method does not load the SDK specific configuration.
     *
     * @param context The activity context.
     * @return The String version of the local JSON configuration.
     */
    private String loadGameConfigFromAssets(Context context) {
        String json = null;

        try {

            //InputStream is = context.getAssets().open("defaultGameConfigAndroid.json");
            InputStream is = context.getAssets().open("defaultGameConfig.json");
            int size = is.available();
            byte[] buffer = new byte[size];
            is.read(buffer);
            is.close();
            json = new String(buffer, "UTF-8");

            JSONObject temp = new JSONObject(json);

            json = temp.toString();

        } catch (IOException | JSONException ex) {
            ex.printStackTrace();
        }

        return json;
    }

    /**
     * Method that is used internally to load the configuration JSON from the file stored locally.
     * The name of the file is "defaultGameConfig.json".
     * This method loads only the SDK specific configuration.
     *
     * @param context The activity context.
     * @return The String version of the SDK JSON configuration.
     */
    @VisibleForTesting
    private String loadSDKConfigFromAssets(Context context, FileAssetsUtil fileAssetsUtil) {
        String json = null;

        try {
            InputStream is = fileAssetsUtil.getGameConfigFileAsset(context);
            int size = is.available();
            byte[] buffer = new byte[size];
            is.read(buffer);
            is.close();
            json = new String(buffer, "UTF-8");

            JSONObject temp = new JSONObject(json);

            if (temp.has("androidSdkConfig")) {
                json = temp.getString("androidSdkConfig");
            } else {
                json = null;
            }

        } catch (IOException | JSONException ex) {
            ex.printStackTrace();
        }

        return json;
    }

    public void resetConfig() {
        GamedockSDK.getInstance(context).getStorageUtil().remove(StorageUtil.Keys.GameConfig);
    }
}
