package io.gamedock.sdk.config;


import com.unity3d.player.UnityPlayer;

import org.json.JSONException;
import org.json.JSONObject;

import io.gamedock.sdk.utils.error.ErrorCodes;
import io.gamedock.sdk.utils.logging.LoggingUtil;

/**
 * Class that handles the Config Callbacks.
 * Notifies the application when the Config has been received and processed by the SDK.
 */
public class ConfigDataCallbacks {

    private OnConfigDataListener configDataListener;

    public ConfigDataCallbacks() {
        configDataListener = null;
    }

    public ConfigDataCallbacks(OnConfigDataListener configDataListener) {
        this.configDataListener = configDataListener;
    }

    /**
     * Method that is called when the configuration has been updated by the SDK.
     */
    public void configDataUpdated() {
        if (configDataListener != null) {
            configDataListener.ConfigDataUpdated();
        } else {
            try {
                UnityPlayer.UnitySendMessage("GamedockSDK", "ConfigUpdated", "");
            } catch (NoClassDefFoundError e) {
                LoggingUtil.w("Tried to send information to Unity but UnityPlayer could not be found (NoClassDefFoundError). This can happen by design and is not necessarily a problem.");
            }
        }
    }

    /**
     * Method that is called if an error has occured while trying to retrieve the game configuration.
     *
     * @param error The error title and error message describing the problem.
     */
    public void configError(ErrorCodes error) {
        if (configDataListener != null) {
            configDataListener.ConfigError(error);
        } else {
            try {
                JSONObject jsonObject = new JSONObject();
                jsonObject.put("id", error.getId());
                jsonObject.put("name", error.getName());
                jsonObject.put("message", error.getMessage());

                UnityPlayer.UnitySendMessage("GamedockSDK", "ConfigError", jsonObject.toString());
            } catch (JSONException | NoClassDefFoundError e) {
                LoggingUtil.w("Tried to send information to Unity but UnityPlayer could not be found (NoClassDefFoundError). This can happen by design and is not necessarily a problem.");
            }
        }
    }

    /**
     * Method called when the Firebase Remote configuration has been updated from the backend.
     */
    public void firebaseRemoteConfigUpdated() {
        if (configDataListener != null) {
            configDataListener.FirebaseRemoteConfigUpdated();
        } else {
            try {
                UnityPlayer.UnitySendMessage("GamedockSDK", "FirebaseRemoteConfigUpdated", "");
            } catch (NoClassDefFoundError e) {
                LoggingUtil.w("Tried to send information to Unity but UnityPlayer could not be found (NoClassDefFoundError). This can happen by design and is not necessarily a problem.");
            }
        }
    }

}
