package io.gamedock.sdk.utils.permissions;

import android.app.Activity;
import android.content.Context;
import android.content.Intent;
import android.content.pm.PackageManager;
import android.util.Log;

import androidx.annotation.MainThread;
import androidx.annotation.NonNull;
import androidx.core.app.ActivityCompat;
import androidx.core.content.ContextCompat;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import io.gamedock.sdk.GamedockSDK;
import io.gamedock.sdk.R;
import io.gamedock.sdk.utils.dialog.MaterialDialog;
import io.gamedock.sdk.utils.dialog.MaterialStyledDialog;
import io.gamedock.sdk.utils.dialog.internal.DialogAction;
import io.gamedock.sdk.utils.logging.LoggingUtil;

/**
 * A class to make permission-management easier. Provides methods to conveniently request permissions anywhere in your
 * app.
 */
public class PermissionBuilder {

    private static final String TAG = "PermissionBuilder";

    private RequestData requestData;

    /**
     * A map to keep track of our outstanding permission requests. The key is the request code sent when we call
     * {@link ActivityCompat#requestPermissions(Activity, String[], int)}. The value is the {@link PermissionBuilder.RequestData}
     * bundle that holds all of the request information.
     */
    private Map<Integer, RequestData> mCodesToRequests;

    /**
     * This is just a value we increment to generate new request codes for use with
     * {@link ActivityCompat#requestPermissions(Activity, String[], int)}.
     */
    private int mActiveRequestCode = 1;

    /**
     * The singleton instance.
     */
    private static PermissionBuilder mInstance = new PermissionBuilder();


    // =====================================================================
    // Creation
    // =====================================================================

    /**
     * @return An instance of {@link PermissionBuilder} to help you manage your permissions.
     */
    public static PermissionBuilder getInstance() {
        return mInstance;
    }

    /**
     * Implementing a singleton pattern, so this is private.
     */
    private PermissionBuilder() {
        mCodesToRequests = new HashMap<>();
    }


    // =====================================================================
    // Public
    // =====================================================================

    /**
     * Request one or more permissions from the system.
     *
     * @param callback    A callback that will be triggered when the results of your permission request are available.
     * @param permissions A list of permission constants that you are requesting. Use constants from
     *                    {@link android.Manifest.permission}.
     */
    @MainThread
    public void requestPermissions(Context callingContext, @NonNull IOnPermissionResult callback, boolean showRationale, String... permissions) {

        Activity activity = checkActivity(callingContext);

        requestData = new RequestData(callback, permissions);

        // Mark any permissions that are already granted
        for (String permission : permissions) {
            if (activity != null && ContextCompat.checkSelfPermission(activity, permission) == PackageManager.PERMISSION_GRANTED) {
                requestData.resultSet.grantPermissions(permission);
            }
        }

        // If we had all of them, yay! No need to do anything else.
        if (requestData.resultSet.areAllPermissionsGranted()) {
            requestData.onResultListener.onPermissionResult(requestData.resultSet);
        } else {
            // If we have some unsatisfied ones, let's first see if they can be satisfied by an active request. If it
            // can, we'll re-wire the callback of the active request to also trigger this new one.
            boolean linkedToExisting = linkToExistingRequestIfPossible(requestData);

            // If there was no existing request that can satisfy this one, then let's make a new permission request to
            // the system
            if (!linkedToExisting) {
                // Mark the request as active
                final int requestCode = markRequestAsActive(requestData);

                // First check if there's any permissions for which we need to provide a rationale for using
                String[] permissionsThatNeedRationale = requestData.resultSet.getPermissionsThatNeedRationale(activity);

                // If there are some that need a rationale, show that rationale, then continue with the request
                final Context context = checkActivity(callingContext);

                if (context != null && !PermissionUtil.getPermissionRequestDeny(context)) {
                    permissionsThatNeedRationale = permissions;
                }

                if (permissionsThatNeedRationale.length > 0 && showRationale) {
                    requestData.onResultListener.onRationaleRequested(new IOnRationaleProvided() {
                        @Override
                        public void onRationaleProvided() {
                            makePermissionRequest(context, requestCode);
                        }
                    }, permissionsThatNeedRationale);
                } else {
                    makePermissionRequest(context, requestCode);
                }
            }
        }
    }

    /**
     * This method needs to be called by your activity's {@link Activity#onRequestPermissionsResult(int, String[], int[])}.
     * Simply forward the results of that method here.
     * <p>
     *
     * @param requestCode  The request code given to you by {@link Activity#onRequestPermissionsResult(int, String[], int[])}.
     * @param permissions  The permissions given to you by {@link Activity#onRequestPermissionsResult(int, String[], int[])}.
     * @param grantResults The grant results given to you by {@link Activity#onRequestPermissionsResult(int, String[], int[])}.
     */
    @MainThread
    public void onRequestPermissionResult(Context context, int requestCode, String[] permissions, int[] grantResults) {
        Activity activity = checkActivity(context);
        if (activity != null && mCodesToRequests.containsKey(requestCode)) {
            RequestData requestData = mCodesToRequests.get(requestCode);
            requestData.resultSet.parsePermissionResults(permissions, grantResults, activity);
            requestData.onResultListener.onPermissionResult(requestData.resultSet);
            mCodesToRequests.remove(requestCode);
        } else {
            Log.w(TAG, "onRequestPermissionResult() was given an unrecognized request code.");
        }
    }

    /**
     * A helper to show your rationale in a {@link android.app.DialogFragment} when implementing
     * {@link IOnRationaleProvided#onRationaleProvided()}. Automatically invokes the rationale callback when the user
     * dismisses the dialog.
     *
     * @param title             The title of the dialog. If null, there will be no title.
     * @param message           The message displayed in the dialog.
     * @param rationaleCallback The callback to be trigger
     */
    @MainThread
    public void showRationaleInDialog(Context callingContext, @NonNull String title, @NonNull String message, boolean isDenyRationale, boolean withSettings, boolean withAskAgain, final IOnRationaleProvided rationaleCallback) {
        if (isDenyRationale) {
            Context context = checkActivity(callingContext);
            if (context != null) {
                GamedockSDK.getInstance(context).isShowingChildActivity = true;

                Intent intent = new Intent(context, PermissionRationaleActivity.class);
                intent.putExtra("dialogTitle", title);
                intent.putExtra("dialogMessage", message);
                intent.putExtra("dialogWithSettings", withSettings);
                intent.putExtra("dialogWithAskAgain", withAskAgain);
                intent.addFlags(Intent.FLAG_ACTIVITY_NEW_TASK);

                context.startActivity(intent);
            }
        } else {
            Context context = checkActivity(callingContext);
            if (context != null) {
                int permissionHeader;
                int resourceId = context.getResources().getIdentifier("permission_header_custom", "drawable", context.getPackageName());

                if (resourceId != 0) {
                    permissionHeader = resourceId;
                } else {
                    permissionHeader = R.drawable.permission_header;
                }

                MaterialStyledDialog.Builder builder = new MaterialStyledDialog.Builder(context)
                        .setTitle(title)
                        .setDescription(message)
                        .setHeaderDrawable(permissionHeader)
                        .autoDismiss(false)
                        .withDialogAnimation(true)
                        .setPositiveText(android.R.string.ok)
                        .onPositive(new MaterialDialog.SingleButtonCallback() {
                            @Override
                            public void onClick(@NonNull MaterialDialog materialDialog, @NonNull DialogAction dialogAction) {
                                rationaleCallback.onRationaleProvided();
                                materialDialog.dismiss();
                            }
                        });

                builder.show();
            }
        }
    }

    // =====================================================================
    // Private
    // =====================================================================

    /**
     * Checks to see if there are any active requests that are already requesting a superset of the permissions this
     * new request is asking for. If so, this will wire up this new request's callback to be triggered when the
     * existing request is completed and return true. Otherwise, this does nothing and returns false.
     *
     * @param newRequest The new request that is about to be made.
     * @return True if a request was linked, otherwise false.
     */
    private boolean linkToExistingRequestIfPossible(final RequestData newRequest) {
        boolean found = false;

        // Go through all outstanding requests
        for (final RequestData activeRequest : mCodesToRequests.values()) {
            // If we find one that can satisfy all of the new request's permissions, we re-wire the active one's
            // callback to also call this new one's callback
            if (activeRequest.resultSet.containsAllUngrantedPermissions(newRequest.resultSet)) {
                final IOnPermissionResult originalOnResultListener = activeRequest.onResultListener;
                activeRequest.onResultListener = new IOnPermissionResult() {
                    @Override
                    public void onPermissionResult(ResultSet resultSet) {
                        // First, call the active one's callback. It was added before this new one.
                        originalOnResultListener.onPermissionResult(resultSet);

                        // Next, copy over the results to the new one's resultSet
                        String[] unsatisfied = newRequest.resultSet.getUngrantedPermissions();
                        for (String permission : unsatisfied) {
                            newRequest.resultSet.requestResults.put(permission, resultSet.requestResults.get(permission));
                        }

                        // Finally, trigger the new one's callback
                        newRequest.onResultListener.onPermissionResult(newRequest.resultSet);
                    }

                    @Override
                    public void onRationaleRequested(IOnRationaleProvided callback, String... permissions) {
                        activeRequest.onResultListener.onRationaleRequested(callback, permissions);
                    }
                };
                found = true;
                break;
            }
        }

        return found;
    }

    /**
     * Puts the RequestData in the map of requests and gives back the request code.
     *
     * @return The request code generated for this request.
     */
    private int markRequestAsActive(RequestData requestData) {
        int requestCode = mActiveRequestCode++;
        mCodesToRequests.put(requestCode, requestData);
        return requestCode;
    }

    /**
     * Makes the permission request for the request that matches the provided request code.
     *
     * @param requestCode The request code of the request you want to run.
     */
    private void makePermissionRequest(Context callingContext, int requestCode) {
        Activity activity = checkActivity(callingContext);

        if (activity != null) {
            RequestData requestData = mCodesToRequests.get(requestCode);

            if(requestData != null && requestData.resultSet != null) {
                ActivityCompat.requestPermissions(activity, requestData.resultSet.getUngrantedPermissions(), requestCode);
            } else {
                LoggingUtil.e("Could not request permissions!");
            }
        } else {
            LoggingUtil.e("Could not request permissions!");
        }
    }

    /**
     * Ensures that our WeakReference to the Activity is still valid. If it isn't, throw an exception saying that the
     * Activity needs to be set.
     */
    private Activity checkActivity(Context context) {
        if(PermissionRationaleActivity.activity != null) {
            return PermissionRationaleActivity.activity;
        } else {
            if (context != null && context instanceof Activity) {
                return (Activity) context;
            }
        }

        return null;
    }


    // =====================================================================
    // Inner Classes
    // =====================================================================

    /**
     * A callback interface for receiving the results of a permission request.
     */
    public interface IOnPermissionResult {
        /**
         * Invoked when the results of your permission request are ready.
         *
         * @param resultSet An object holding the result of your permission request.
         */
        void onPermissionResult(ResultSet resultSet);

        /**
         * Called when the system recommends that you provide a rationale for a permission. This typically happens when
         * a user denies a permission, but they you request it again.
         *
         * @param callback    A callback to be triggered when you are finished showing the user the rationale.
         * @param permissions The list of permissions for which the system recommends you provide a rationale.
         */
        void onRationaleRequested(IOnRationaleProvided callback, String... permissions);
    }

    /**
     * Simple callback to let Permiso know that you have finished providing the user a rationale for a set of permissions.
     * For easy handling of this callback, consider using
     */
    public interface IOnRationaleProvided {
        /**
         * Invoke this method when you are done providing a rationale to the user in
         * {@link IOnPermissionResult#onRationaleRequested(IOnRationaleProvided, String...)}. The permission request
         * will not be made until this method is invoked.
         */
        void onRationaleProvided();
    }

    private static class RequestData {
        IOnPermissionResult onResultListener;
        ResultSet resultSet;

        public RequestData(@NonNull IOnPermissionResult onResultListener, String... permissions) {
            this.onResultListener = onResultListener;
            resultSet = new ResultSet(permissions);
        }
    }

    /**
     * A class representing the results of a permission request.
     */
    public static class ResultSet {

        private Map<String, Result> requestResults;

        private ResultSet(String... permissions) {
            requestResults = new HashMap<>(permissions.length);
            for (String permission : permissions) {
                requestResults.put(permission, Result.DENIED);
            }
        }

        /**
         * Checks if a permission was granted during your permission request.
         *
         * @param permission The permission you are inquiring about. This should be a constant from {@link android.Manifest.permission}.
         * @return True if the permission was granted, otherwise false.
         */
        public boolean isPermissionGranted(String permission) {
            return requestResults.containsKey(permission) && requestResults.get(permission) == Result.GRANTED;
        }

        /**
         * Determines if all permissions in the request were granted.
         *
         * @return True if all permissions in the request were granted, otherwise false.
         */
        public boolean areAllPermissionsGranted() {
            return !requestResults.containsValue(Result.DENIED) && !requestResults.containsValue(Result.PERMANENTLY_DENIED);
        }

        /**
         * Checks if a permission was permanently denied by the user (i.e. they denied and selected "Dont Ask Again".)
         *
         * @param permission The permission you are inquiring about. This should be a constant from {@link android.Manifest.permission}.
         * @return True if the permission was permanently denied, otherwise false.
         */
        public boolean isPermissionPermanentlyDenied(String permission) {
            return requestResults.containsKey(permission) && requestResults.get(permission) == Result.PERMANENTLY_DENIED;
        }

        /**
         * Returns a map representation of this result set. Useful if you'd like to do more complicated operations
         * with the results.
         *
         * @return A mapping of permission constants to {@link Result}.
         */
        public Map<String, Result> toMap() {
            return new HashMap<>(requestResults);
        }

        private void grantPermissions(String... permissions) {
            for (String permission : permissions) {
                requestResults.put(permission, Result.GRANTED);
            }
        }

        private void parsePermissionResults(String[] permissions, int[] grantResults, Activity activity) {
            for (int i = 0; i < permissions.length; i++) {
                if (grantResults[i] == PackageManager.PERMISSION_GRANTED) {
                    requestResults.put(permissions[i], Result.GRANTED);
                } else if (!ActivityCompat.shouldShowRequestPermissionRationale(activity, permissions[i])) {
                    requestResults.put(permissions[i], Result.PERMANENTLY_DENIED);
                } else {
                    requestResults.put(permissions[i], Result.DENIED);
                }
            }
        }

        private String[] getUngrantedPermissions() {
            List<String> ungrantedList = new ArrayList<>(requestResults.size());
            for (Map.Entry<String, Result> requestResultsEntry : requestResults.entrySet()) {
                Result result = requestResultsEntry.getValue();
                if (result == Result.DENIED || result == Result.PERMANENTLY_DENIED) {
                    ungrantedList.add(requestResultsEntry.getKey());
                }
            }
            return ungrantedList.toArray(new String[ungrantedList.size()]);
        }

        private boolean containsAllUngrantedPermissions(ResultSet set) {
            List<String> ungranted = Arrays.asList(set.getUngrantedPermissions());
            return requestResults.keySet().containsAll(ungranted);
        }

        private String[] getPermissionsThatNeedRationale(Activity activity) {
            String[] ungranted = getUngrantedPermissions();
            List<String> shouldShowRationale = new ArrayList<>(ungranted.length);
            for (String permission : ungranted) {
                if (ActivityCompat.shouldShowRequestPermissionRationale(activity, permission)) {
                    shouldShowRationale.add(permission);
                }
            }
            return shouldShowRationale.toArray(new String[shouldShowRationale.size()]);
        }
    }

    /**
     * Describes the result of a permission request.
     */
    public enum Result {
        /**
         * The permission was granted.
         */
        GRANTED,

        /**
         * The permission was denied, but not permanently.
         */
        DENIED,

        /**
         * The permission was permanently denied.
         */
        PERMANENTLY_DENIED
    }
}