package io.gamedock.sdk.utils.performance;

import android.content.Context;

import java.util.ArrayList;
import java.util.Locale;

import io.gamedock.sdk.GamedockSDK;
import io.gamedock.sdk.models.performance.PerformanceMetric;
import io.gamedock.sdk.utils.storage.StorageUtil;

/**
 * Utility class used to retrieve, popluate and manage stat metrics from the SDK.
 */
public class PerformanceUtil {

    private static final Object lock = new Object();

    private static volatile PerformanceUtil mInstance = null;
    private Context context;

    private ArrayList<Float> memoryUsageHistory = new ArrayList<>();
    private ArrayList<Float> cpuUsageHistory = new ArrayList<>();
    private ArrayList<Float> networkSentUsageHistory = new ArrayList<>();
    private ArrayList<Float> networkReceivedUsageHistory = new ArrayList<>();
    private ArrayList<Float> fpsUsageHistory = new ArrayList<>();

    private PerformanceUtil(Context context) {
        this.context = context;
    }

    public static PerformanceUtil getInstance(Context context) {
        if (mInstance == null) {
            synchronized (lock) {
                if (mInstance == null) {
                    mInstance = new PerformanceUtil(context);
                }
            }
        }
        return mInstance;
    }

    /**
     * Method used to periodically retrieve stats regarding Memory, CPU and Network traffic.
     */
    public void recordPerformanceStats() {
        //Memory recording
        memoryUsageHistory.add((float) MemoryInfo.getMemoryUsage(context));

        //Cpu recording
        cpuUsageHistory.add(CpuInfo.getCpuUsage());

        //NetworkSent recording
        long currentRecordedNetworkSent = NetworkInfo.getNetworkTrafficSent(context);
        float previousRecordedNetworkSent = GamedockSDK.getInstance(context).getStorageUtil().getFloat(StorageUtil.Keys.PreviouslyNetworkSent, 0);
        //Check if device was rebooted and if so start the tracking from 0
        if (currentRecordedNetworkSent < previousRecordedNetworkSent) {
            previousRecordedNetworkSent = 0;
        }
        networkSentUsageHistory.add((float) (currentRecordedNetworkSent - previousRecordedNetworkSent));
        GamedockSDK.getInstance(context).getStorageUtil().putFloat(StorageUtil.Keys.PreviouslyNetworkSent, currentRecordedNetworkSent);

        //NetworkReceived recording
        long currentRecordedNetworkReceived = NetworkInfo.getNetworkTrafficReceived(context);
        float previousRecordedNetworkReceived = GamedockSDK.getInstance(context).getStorageUtil().getFloat(StorageUtil.Keys.PreviouslyNetworkReceived, 0);
        //Check if device was rebooted and if so start the tracking from 0
        if (currentRecordedNetworkReceived < previousRecordedNetworkReceived) {
            previousRecordedNetworkReceived = 0;
        }
        networkReceivedUsageHistory.add((float) (currentRecordedNetworkReceived - previousRecordedNetworkReceived));
        GamedockSDK.getInstance(context).getStorageUtil().putFloat(StorageUtil.Keys.PreviouslyNetworkReceived, currentRecordedNetworkReceived);
    }

    /**
     * Method used to record the FPS values received from the game.
     *
     * @param fpsValue The FPS value.
     */
    public void recordFPSStat(double fpsValue) {
        fpsUsageHistory.add((float) fpsValue);
    }

    /**
     * Method used to gather all the periodic methods and populate them into values that will be used by the heartbeat event.
     *
     * @return A list of all the metrics recorded by the SDK.
     */
    public ArrayList<PerformanceMetric> populatePerformanceMetrics() {
        ArrayList<PerformanceMetric> metrics = new ArrayList<>();

        //Populating memory info
        PerformanceMetric memory = populateSpecificMetric(PerformanceMetric.MetricEnum.Memory, memoryUsageHistory);
        metrics.add(memory);

        //Populating cpu info
        PerformanceMetric cpu = populateSpecificMetric(PerformanceMetric.MetricEnum.CPU, cpuUsageHistory);
        metrics.add(cpu);

        //Populating network sent info
        PerformanceMetric networkSent = populateSpecificMetric(PerformanceMetric.MetricEnum.NetworkTrafficSent, networkSentUsageHistory);
        metrics.add(networkSent);

        //Populating network received info
        PerformanceMetric networkReceived = populateSpecificMetric(PerformanceMetric.MetricEnum.NetworkTrafficReceived, networkReceivedUsageHistory);
        metrics.add(networkReceived);

        //Populating FPS info
        if (!fpsUsageHistory.isEmpty()) {
            PerformanceMetric fps = populateSpecificMetric(PerformanceMetric.MetricEnum.FPS, fpsUsageHistory);
            metrics.add(fps);
        }

        return metrics;
    }

    /**
     * Method used to populate the performance value of a specific @{@link PerformanceMetric.MetricEnum}
     *
     * @param metric       The metric value that needs to be populated.
     * @param usageHistory The usage history of a specific metric.
     * @return The {@link PerformanceMetric} object based on the {@link PerformanceMetric.MetricEnum} passed.
     */
    private PerformanceMetric populateSpecificMetric(PerformanceMetric.MetricEnum metric, ArrayList<Float> usageHistory) {
        float maxValue = 0;
        float minValue = 100000;
        float totalValue = 0;
        float averageValue = 0;
        float currentValue = 0;

        for (Float value : usageHistory) {
            if (value > maxValue) {
                maxValue = value;
            }
            if (value < minValue) {
                minValue = value;
            }

            totalValue = totalValue + value;
        }
        String averageValueString = String.format(Locale.ENGLISH, "%.02f", totalValue / usageHistory.size());
        averageValueString = averageValueString.replace(",", ".");
        averageValue = Float.parseFloat(averageValueString);

        switch (metric) {
            case Memory:
                currentValue = -1;
                break;
            case CPU:
                currentValue = -1;
                totalValue = -1;
                if (averageValue == -1) {
                    maxValue = -1;
                    minValue = -1;
                }
                break;
            case NetworkTrafficSent:
                currentValue = NetworkInfo.getNetworkTrafficSent(context) - GamedockSDK.getInstance(context).getStorageUtil().getFloat(StorageUtil.Keys.PreviouslyNetworkSent, 0);
                if (currentValue > maxValue) {
                    maxValue = currentValue;
                }
                if (currentValue < minValue) {
                    minValue = currentValue;
                }
                totalValue = NetworkInfo.getNetworkTrafficSent(context);
                break;
            case NetworkTrafficReceived:
                currentValue = NetworkInfo.getNetworkTrafficReceived(context) - GamedockSDK.getInstance(context).getStorageUtil().getFloat(StorageUtil.Keys.PreviouslyNetworkReceived, 0);
                if (currentValue > maxValue) {
                    maxValue = currentValue;
                }
                if (currentValue < minValue) {
                    minValue = currentValue;
                }
                totalValue = NetworkInfo.getNetworkTrafficReceived(context);
                break;
            case FPS:
                currentValue = -1;
                totalValue = -1;
                break;
        }

        return new PerformanceMetric(metric, currentValue, averageValue, totalValue, maxValue, minValue);
    }

    public synchronized void clearMetrics() {
        memoryUsageHistory.clear();
        cpuUsageHistory.clear();
        networkSentUsageHistory.clear();
        networkReceivedUsageHistory.clear();
        fpsUsageHistory.clear();
    }
}
