package io.gamedock.sdk.utils.features;

import com.google.gson.Gson;
import com.google.gson.reflect.TypeToken;

import java.util.HashMap;

import io.gamedock.sdk.models.feature.Feature;
import io.gamedock.sdk.utils.logging.LoggingUtil;
import io.gamedock.sdk.utils.storage.StorageUtil;

public class FeaturesUtil {
    private static HashMap<String, Long> featuresVersionIds = new HashMap<>();
    private static HashMap<String, Feature> features = new HashMap<>();
    private static Gson gson;
    private static StorageUtil storageUtil;

    public static void initFeaturesInformation(StorageUtil storageUtilValue, Gson gsonValue) {
        storageUtil = storageUtilValue;
        gson = gsonValue;

        if (storageUtil.contains(StorageUtil.Keys.FeaturesInfo)) {
            features = gson.fromJson(storageUtil.getString(StorageUtil.Keys.FeaturesInfo, "{}"), new TypeToken<HashMap<String, Feature>>() {
            }.getType());
        }

        if (storageUtil.contains(StorageUtil.Keys.FeaturesVersionIds)) {
            featuresVersionIds = gson.fromJson(storageUtil.getString(StorageUtil.Keys.FeaturesVersionIds, "{}"), new TypeToken<HashMap<String, Long>>() {
            }.getType());
        }
    }

    public static HashMap<String, Feature> getFeatures() {
        return features;
    }

    public static long getFeaturesVersionId(String feature) {
        if (featuresVersionIds.containsKey(feature)) {
            return featuresVersionIds.get(feature);
        }
        return 0;
    }

    public static void setFeatureVersionId(String feature, Long id) {
        featuresVersionIds.put(feature, id);
        String featuresIdsToSave = gson.toJson(featuresVersionIds);
        storageUtil.putString(StorageUtil.Keys.FeaturesVersionIds, featuresIdsToSave);
    }

    public static boolean isFeatureEnabled(String featureName) {
        if (features.isEmpty()) {
            LoggingUtil.d("Features list is empty. SDK will initialise and use the feature " + featureName + " by default");
            return true;
        }

        Feature feature = features.get(featureName);
        if (feature == null) {
            LoggingUtil.d("Feature " + featureName + " not found in the features list. SDK will not initialise and use the feature by default");
            return false;
        }

        if (feature.isEnabled()) {
            return true;
        }

        LoggingUtil.d("Feature " + featureName + " is not enabled. Make sure to enable the feature in the Gamedock Console");
        if (feature.getDependencies() != null) {
            LoggingUtil.d("Feature " + featureName + " requires the following features: " + gson.toJson(feature.getDependencies().getRequired()));
            LoggingUtil.d("Feature " + featureName + " optionally needs the following features: " + gson.toJson(feature.getDependencies().getOptional()));
        }
        return false;
    }
}
