package io.gamedock.sdk.utils.dialog;

import android.annotation.SuppressLint;
import android.content.Context;
import android.content.DialogInterface;
import android.content.res.ColorStateList;
import android.graphics.Paint;
import android.graphics.Typeface;
import android.graphics.drawable.Drawable;
import android.os.Build;
import android.view.LayoutInflater;
import android.view.View;
import android.view.ViewGroup;
import android.view.WindowManager;
import android.widget.FrameLayout;
import android.widget.ImageView;
import android.widget.TextView;

import androidx.annotation.ColorInt;
import androidx.annotation.ColorRes;
import androidx.annotation.DrawableRes;
import androidx.annotation.LayoutRes;
import androidx.annotation.NonNull;
import androidx.annotation.Nullable;
import androidx.annotation.StringRes;
import androidx.annotation.UiThread;
import androidx.core.content.res.ResourcesCompat;

import io.gamedock.sdk.R;
import io.gamedock.sdk.utils.dialog.internal.DialogAction;
import io.gamedock.sdk.utils.dialog.internal.DialogUtils;
import io.gamedock.sdk.utils.dialog.internal.GravityEnum;
import io.gamedock.sdk.utils.dialog.internal.MDButton;
import io.gamedock.sdk.utils.dialog.internal.MDRootLayout;
import io.gamedock.sdk.utils.dialog.internal.RippleHelper;
import io.gamedock.sdk.utils.dialog.internal.Theme;
import io.gamedock.sdk.utils.dialog.internal.ThemeSingleton;
import io.gamedock.sdk.utils.dialog.internal.TypefaceHelper;

public class MaterialDialog extends DialogBase
        implements View.OnClickListener {

    protected final Builder builder;
    protected ImageView icon;
    protected TextView title;
    protected TextView content;

    View titleFrame;
    FrameLayout customViewFrame;
    MDButton positiveButton;
    MDButton neutralButton;
    MDButton negativeButton;

    @SuppressLint("InflateParams")
    protected MaterialDialog(Builder builder) {
        super(builder.context, DialogInit.getTheme(builder));
        this.builder = builder;
        final LayoutInflater inflater = LayoutInflater.from(builder.context);
        view = (MDRootLayout) inflater.inflate(DialogInit.getInflateLayout(builder), null);
        DialogInit.init(this);
    }

    public final Builder getBuilder() {
        return builder;
    }

    public final void setTypeface(TextView target, Typeface t) {
        if (t == null) {
            return;
        }
        int flags = target.getPaintFlags() | Paint.SUBPIXEL_TEXT_FLAG;
        target.setPaintFlags(flags);
        target.setTypeface(t);
    }

    /* package */ Drawable getButtonSelector(DialogAction which, boolean isStacked) {
        if (isStacked) {
            if (builder.btnSelectorStacked != 0) {
                return ResourcesCompat.getDrawable(builder.context.getResources(), builder.btnSelectorStacked, null);
            }
            final Drawable d = DialogUtils.resolveDrawable(builder.context, R.attr.md_btn_stacked_selector);
            if (d != null) {
                return d;
            }
            return DialogUtils.resolveDrawable(getContext(), R.attr.md_btn_stacked_selector);
        } else {
            switch (which) {
                default: {
                    if (builder.btnSelectorPositive != 0) {
                        return ResourcesCompat.getDrawable(builder.context.getResources(), builder.btnSelectorPositive, null);
                    }
                    Drawable d = DialogUtils.resolveDrawable(builder.context, R.attr.md_btn_positive_selector);
                    if (d != null) {
                        return d;
                    }
                    d = DialogUtils.resolveDrawable(getContext(), R.attr.md_btn_positive_selector);
                    if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.LOLLIPOP) {
                        RippleHelper.applyColor(d, builder.buttonRippleColor);
                    }
                    return d;
                }
                case NEUTRAL: {
                    if (builder.btnSelectorNeutral != 0) {
                        return ResourcesCompat.getDrawable(builder.context.getResources(), builder.btnSelectorNeutral, null);
                    }
                    Drawable d = DialogUtils.resolveDrawable(builder.context, R.attr.md_btn_neutral_selector);
                    if (d != null) {
                        return d;
                    }
                    d = DialogUtils.resolveDrawable(getContext(), R.attr.md_btn_neutral_selector);
                    if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.LOLLIPOP) {
                        RippleHelper.applyColor(d, builder.buttonRippleColor);
                    }
                    return d;
                }
                case NEGATIVE: {
                    if (builder.btnSelectorNegative != 0) {
                        return ResourcesCompat.getDrawable(builder.context.getResources(), builder.btnSelectorNegative, null);
                    }
                    Drawable d = DialogUtils.resolveDrawable(builder.context, R.attr.md_btn_negative_selector);
                    if (d != null) {
                        return d;
                    }
                    d = DialogUtils.resolveDrawable(getContext(), R.attr.md_btn_negative_selector);
                    if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.LOLLIPOP) {
                        RippleHelper.applyColor(d, builder.buttonRippleColor);
                    }
                    return d;
                }
            }
        }
    }

    @Override
    public final void onClick(View v) {
        DialogAction tag = (DialogAction) v.getTag();
        switch (tag) {
            case POSITIVE: {
                if (builder.onPositiveCallback != null) {
                    builder.onPositiveCallback.onClick(this, tag);
                }

                if (builder.autoDismiss) {
                    dismiss();
                }
                break;
            }
            case NEGATIVE: {
                if (builder.onNegativeCallback != null) {
                    builder.onNegativeCallback.onClick(this, tag);
                }
                if (builder.autoDismiss) {
                    cancel();
                }
                break;
            }
            case NEUTRAL: {
                if (builder.onNeutralCallback != null) {
                    builder.onNeutralCallback.onClick(this, tag);
                }
                if (builder.autoDismiss) {
                    dismiss();
                }
                break;
            }
        }
    }

    @Override
    @UiThread
    public void show() {
        try {
            super.show();
        } catch (WindowManager.BadTokenException e) {
            throw new DialogException("Bad window token, you cannot show a dialog before an Activity is created or after it's hidden.");
        }
    }

    /**
     * Retrieves the view representing the dialog as a whole. Be careful with this.
     */
    public final View getView() {
        return view;
    }

    /**
     * Retrieves the custom view that was inflated or set to the MaterialDialog during building.
     *
     * @return The custom view that was passed into the Builder.
     */
    @Nullable
    public final View getCustomView() {
        return builder.customView;
    }

    /**
     * Updates an action button's title, causing invalidation to check if the action buttons should be
     * stacked. Setting an action button's text to null is a shortcut for hiding it, too.
     *
     * @param which The action button to update.
     * @param title The new title of the action button.
     */
    @SuppressWarnings("WeakerAccess")
    @UiThread
    public final void setActionButton(@NonNull final DialogAction which, final CharSequence title) {
        switch (which) {
            default:
                builder.positiveText = title;
                positiveButton.setText(title);
                positiveButton.setVisibility(title == null ? View.GONE : View.VISIBLE);
                break;
            case NEUTRAL:
                builder.neutralText = title;
                neutralButton.setText(title);
                neutralButton.setVisibility(title == null ? View.GONE : View.VISIBLE);
                break;
            case NEGATIVE:
                builder.negativeText = title;
                negativeButton.setText(title);
                negativeButton.setVisibility(title == null ? View.GONE : View.VISIBLE);
                break;
        }
    }

    /**
     * Updates an action button's title, causing invalidation to check if the action buttons should be
     * stacked.
     *
     * @param which    The action button to update.
     * @param titleRes The string resource of the new title of the action button.
     */
    public final void setActionButton(DialogAction which, @StringRes int titleRes) {
        setActionButton(which, getContext().getText(titleRes));
    }

    /**
     * Gets whether or not the positive, neutral, or negative action button is visible.
     *
     * @return Whether or not 1 or more action buttons is visible.
     */
    public final boolean hasActionButtons() {
        return numberOfActionButtons() > 0;
    }

    /**
     * Gets the number of visible action buttons.
     *
     * @return 0 through 3, depending on how many should be or are visible.
     */
    @SuppressWarnings("WeakerAccess")
    public final int numberOfActionButtons() {
        int number = 0;
        if (builder.positiveText != null && positiveButton.getVisibility() == View.VISIBLE) {
            number++;
        }
        if (builder.neutralText != null && neutralButton.getVisibility() == View.VISIBLE) {
            number++;
        }
        if (builder.negativeText != null && negativeButton.getVisibility() == View.VISIBLE) {
            number++;
        }
        return number;
    }

    @UiThread
    @Override
    public final void setTitle(CharSequence newTitle) {
        title.setText(newTitle);
    }

    @SuppressWarnings("unused")
    @UiThread
    public final void setTitle(@StringRes int newTitleRes, @Nullable Object... formatArgs) {
        setTitle(builder.context.getString(newTitleRes, formatArgs));
    }

    @Override
    public final void onShow(DialogInterface dialog) {
        super.onShow(dialog);
    }

    @Override
    public void dismiss() {
        super.dismiss();
    }

    /**
     * An alternate way to define a single callback.
     */
    public interface SingleButtonCallback {

        void onClick(@NonNull MaterialDialog dialog, @NonNull DialogAction which);
    }

    private static class DialogException extends WindowManager.BadTokenException {

        DialogException(@SuppressWarnings("SameParameterValue") String message) {
            super(message);
        }
    }

    /**
     * The class used to construct a MaterialDialog.
     */
    @SuppressWarnings({"WeakerAccess", "unused"})
    public static class Builder {

        protected final Context context;
        protected CharSequence title;
        protected GravityEnum titleGravity = GravityEnum.START;
        protected GravityEnum contentGravity = GravityEnum.START;
        protected GravityEnum btnStackedGravity = GravityEnum.END;
        protected GravityEnum itemsGravity = GravityEnum.START;
        protected GravityEnum buttonsGravity = GravityEnum.START;
        protected int buttonRippleColor = 0;
        protected int titleColor = -1;
        protected CharSequence positiveText;
        protected CharSequence neutralText;
        protected CharSequence negativeText;
        protected boolean positiveFocus;
        protected boolean neutralFocus;
        protected boolean negativeFocus;
        protected View customView;
        protected int widgetColor;
        protected ColorStateList positiveColor;
        protected ColorStateList negativeColor;
        protected ColorStateList neutralColor;
        protected Drawable positiveDrawable;
        protected Drawable negativeDrawable;
        protected Drawable neutralDrawable;
        protected SingleButtonCallback onPositiveCallback;
        protected SingleButtonCallback onNegativeCallback;
        protected SingleButtonCallback onNeutralCallback;
        protected Theme theme = Theme.LIGHT;
        protected boolean cancelable = true;
        protected boolean canceledOnTouchOutside = true;
        protected float contentLineSpacingMultiplier = 1.2f;
        protected boolean autoDismiss = true;
        protected Typeface regularFont;
        protected Typeface mediumFont;
        protected OnDismissListener dismissListener;
        protected OnCancelListener cancelListener;
        protected OnKeyListener keyListener;
        protected OnShowListener showListener;
        protected boolean wrapCustomViewInScroll;
        protected int dividerColor;
        protected int backgroundColor;

        protected boolean titleColorSet = false;
        protected boolean contentColorSet = false;
        protected boolean itemColorSet = false;
        protected boolean positiveColorSet = false;
        protected boolean neutralColorSet = false;
        protected boolean negativeColorSet = false;
        protected boolean widgetColorSet = false;
        protected boolean dividerColorSet = false;

        protected boolean positiveDrawableSet = false;
        protected boolean negativeDrawableSet = false;
        protected boolean neutralDrawableSet = false;

        @DrawableRes
        protected int btnSelectorStacked;
        @DrawableRes
        protected int btnSelectorPositive;
        @DrawableRes
        protected int btnSelectorNeutral;
        @DrawableRes
        protected int btnSelectorNegative;

        public Builder(@NonNull Context context) {
            this.context = context;

            // Retrieve default accent colors, which are used on the action buttons and progress bars
            this.widgetColor = DialogUtils.getColor(context, R.color.md_material_blue_600);
            if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.LOLLIPOP) {
                this.widgetColor = DialogUtils.resolveColor(context, android.R.attr.colorAccent, this.widgetColor);
            }

            this.positiveColor = DialogUtils.getActionTextStateList(context, this.widgetColor);
            this.negativeColor = DialogUtils.getActionTextStateList(context, this.widgetColor);
            this.neutralColor = DialogUtils.getActionTextStateList(context, this.widgetColor);

            this.buttonRippleColor = DialogUtils.getColor(context, R.color.md_btn_selected);

            // Set the default theme based on the Activity theme's primary color darkness (more white or more black)
            final int primaryTextColor = DialogUtils.resolveColor(context, android.R.attr.textColorPrimary);
            this.theme = DialogUtils.isColorDark(primaryTextColor) ? Theme.LIGHT : Theme.DARK;

            // Load theme values from the ThemeSingleton if needed
            checkSingleton();

            // Retrieve gravity settings from global theme attributes if needed
            this.titleGravity = DialogUtils.resolveGravityEnum(context, R.attr.md_title_gravity, this.titleGravity);
            this.contentGravity = DialogUtils.resolveGravityEnum(context, R.attr.md_content_gravity, this.contentGravity);
            this.btnStackedGravity = DialogUtils.resolveGravityEnum(context, R.attr.md_btnstacked_gravity, this.btnStackedGravity);
            this.itemsGravity = DialogUtils.resolveGravityEnum(context, R.attr.md_items_gravity, this.itemsGravity);
            this.buttonsGravity = DialogUtils.resolveGravityEnum(context, R.attr.md_buttons_gravity, this.buttonsGravity);

            final String mediumFont = DialogUtils.resolveString(context, R.attr.md_medium_font);
            final String regularFont = DialogUtils.resolveString(context, R.attr.md_regular_font);
            try {
                typeface(mediumFont, regularFont);
            } catch (Throwable ignored) {
            }

            if (this.mediumFont == null) {
                try {
                    if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.LOLLIPOP) {
                        this.mediumFont = Typeface.create("sans-serif-medium", Typeface.NORMAL);
                    } else {
                        this.mediumFont = Typeface.create("sans-serif", Typeface.BOLD);
                    }
                } catch (Throwable ignored) {
                    this.mediumFont = Typeface.DEFAULT_BOLD;
                }
            }
            if (this.regularFont == null) {
                try {
                    this.regularFont = Typeface.create("sans-serif", Typeface.NORMAL);
                } catch (Throwable ignored) {
                    this.regularFont = Typeface.SANS_SERIF;
                    if (this.regularFont == null) {
                        this.regularFont = Typeface.DEFAULT;
                    }
                }
            }
        }

        public final Context getContext() {
            return context;
        }

        @SuppressWarnings("ConstantConditions")
        private void checkSingleton() {
            if (ThemeSingleton.get(false) == null) {
                return;
            }
            ThemeSingleton s = ThemeSingleton.get();
            if (s.darkTheme) {
                this.theme = Theme.DARK;
            }
            if (s.titleColor != 0) {
                this.titleColor = s.titleColor;
            }
            if (s.positiveColor != null) {
                this.positiveColor = s.positiveColor;
            }
            if (s.neutralColor != null) {
                this.neutralColor = s.neutralColor;
            }
            if (s.negativeColor != null) {
                this.negativeColor = s.negativeColor;
            }
            if (s.backgroundColor != 0) {
                this.backgroundColor = s.backgroundColor;
            }
            if (s.dividerColor != 0) {
                this.dividerColor = s.dividerColor;
            }
            if (s.btnSelectorStacked != 0) {
                this.btnSelectorStacked = s.btnSelectorStacked;
            }
            if (s.btnSelectorPositive != 0) {
                this.btnSelectorPositive = s.btnSelectorPositive;
            }
            if (s.btnSelectorNeutral != 0) {
                this.btnSelectorNeutral = s.btnSelectorNeutral;
            }
            if (s.btnSelectorNegative != 0) {
                this.btnSelectorNegative = s.btnSelectorNegative;
            }
            if (s.widgetColor != 0) {
                this.widgetColor = s.widgetColor;
            }
            this.titleGravity = s.titleGravity;
            this.contentGravity = s.contentGravity;
            this.btnStackedGravity = s.btnStackedGravity;
            this.itemsGravity = s.itemsGravity;
            this.buttonsGravity = s.buttonsGravity;
        }

        public Builder title(@NonNull CharSequence title) {
            this.title = title;
            return this;
        }

        public Builder titleGravity(@NonNull GravityEnum gravity) {
            this.titleGravity = gravity;
            return this;
        }

        public Builder buttonRippleColor(@ColorInt int color) {
            this.buttonRippleColor = color;
            return this;
        }

        public Builder buttonRippleColorRes(@ColorRes int colorRes) {
            return buttonRippleColor(DialogUtils.getColor(this.context, colorRes));
        }

        public Builder titleColor(@ColorInt int color) {
            this.titleColor = color;
            this.titleColorSet = true;
            return this;
        }

        public Builder titleColorRes(@ColorRes int colorRes) {
            return titleColor(DialogUtils.getColor(this.context, colorRes));
        }

        /**
         * Sets the fonts used in the dialog, by file names. This also uses TypefaceHelper in order to
         * avoid any un-needed allocations (it recycles typefaces for you).
         *
         * @param medium  The name of font in assets/fonts used on titles and action buttons (null uses
         *                device default). E.g. [your-project]/app/main/assets/fonts/[medium]
         * @param regular The name of font in assets/fonts used everywhere else, like content and list
         *                items (null uses device default). E.g. [your-project]/app/main/assets/fonts/[regular]
         * @return The Builder instance so you can chain calls to it.
         */
        public Builder typeface(@Nullable String medium, @Nullable String regular) {
            if (medium != null && !medium.trim().isEmpty()) {
                this.mediumFont = TypefaceHelper.get(this.context, medium);
                if (this.mediumFont == null) {
                    throw new IllegalArgumentException("No font asset found for \"" + medium + "\"");
                }
            }
            if (regular != null && !regular.trim().isEmpty()) {
                this.regularFont = TypefaceHelper.get(this.context, regular);
                if (this.regularFont == null) {
                    throw new IllegalArgumentException("No font asset found for \"" + regular + "\"");
                }
            }
            return this;
        }

        public Builder positiveText(@NonNull CharSequence message) {
            this.positiveText = message;
            return this;
        }

        public Builder positiveColor(@ColorInt int color) {
            return positiveColor(DialogUtils.getActionTextStateList(context, color));
        }

        public Builder positiveColorRes(@ColorRes int colorRes) {
            return positiveColor(DialogUtils.getActionTextColorStateList(this.context, colorRes));
        }

        public Builder positiveColor(@NonNull ColorStateList colorStateList) {
            this.positiveColor = colorStateList;
            this.positiveColorSet = true;
            return this;
        }

        public Builder positiveDrawable(@NonNull Drawable drawable) {
            this.positiveDrawable = drawable;
            this.positiveDrawableSet = true;
            return this;
        }

        public Builder neutralText(@NonNull CharSequence message) {
            this.neutralText = message;
            return this;
        }

        public Builder negativeColor(@ColorInt int color) {
            return negativeColor(DialogUtils.getActionTextStateList(context, color));
        }

        public Builder negativeColorRes(@ColorRes int colorRes) {
            return negativeColor(DialogUtils.getActionTextColorStateList(this.context, colorRes));
        }

        public Builder negativeColor(@NonNull ColorStateList colorStateList) {
            this.negativeColor = colorStateList;
            this.negativeColorSet = true;
            return this;
        }

        public Builder negativeText(@NonNull CharSequence message) {
            this.negativeText = message;
            return this;
        }

        public Builder negativeDrawable(@NonNull Drawable drawable) {
            this.negativeDrawable = drawable;
            this.negativeDrawableSet = true;
            return this;
        }

        public Builder neutralColor(@ColorInt int color) {
            return neutralColor(DialogUtils.getActionTextStateList(context, color));
        }

        public Builder neutralColorRes(@ColorRes int colorRes) {
            return neutralColor(DialogUtils.getActionTextColorStateList(this.context, colorRes));
        }

        public Builder neutralColor(@NonNull ColorStateList colorStateList) {
            this.neutralColor = colorStateList;
            this.neutralColorSet = true;
            return this;
        }

        public Builder neutralDrawable(@NonNull Drawable drawable) {
            this.neutralDrawable = drawable;
            this.neutralDrawableSet = true;
            return this;
        }

        public Builder customView(@LayoutRes int layoutRes, boolean wrapInScrollView) {
            LayoutInflater li = LayoutInflater.from(this.context);
            return customView(li.inflate(layoutRes, null), wrapInScrollView);
        }

        public Builder customView(@NonNull View view, boolean wrapInScrollView) {
            if (view.getParent() != null && view.getParent() instanceof ViewGroup) {
                ((ViewGroup) view.getParent()).removeView(view);
            }
            this.customView = view;
            this.wrapCustomViewInScroll = wrapInScrollView;
            return this;
        }

        public Builder backgroundColor(@ColorInt int color) {
            this.backgroundColor = color;
            return this;
        }

        public Builder backgroundColorRes(@ColorRes int colorRes) {
            return backgroundColor(DialogUtils.getColor(this.context, colorRes));
        }

        public Builder onPositive(@NonNull SingleButtonCallback callback) {
            this.onPositiveCallback = callback;
            return this;
        }

        public Builder onNegative(@NonNull SingleButtonCallback callback) {
            this.onNegativeCallback = callback;
            return this;
        }

        public Builder onNeutral(@NonNull SingleButtonCallback callback) {
            this.onNeutralCallback = callback;
            return this;
        }

        public Builder cancelable(boolean cancelable) {
            this.cancelable = cancelable;
            this.canceledOnTouchOutside = cancelable;
            return this;
        }

        public Builder canceledOnTouchOutside(boolean canceledOnTouchOutside) {
            this.canceledOnTouchOutside = canceledOnTouchOutside;
            return this;
        }

        /**
         * This defaults to true. If set to false, the dialog will not automatically be dismissed when
         * an action button is pressed, and not automatically dismissed when the user selects a list
         * item.
         *
         * @param dismiss Whether or not to dismiss the dialog automatically.
         * @return The Builder instance so you can chain calls to it.
         */
        public Builder autoDismiss(boolean dismiss) {
            this.autoDismiss = dismiss;
            return this;
        }

        public Builder showListener(@NonNull OnShowListener listener) {
            this.showListener = listener;
            return this;
        }

        public Builder dismissListener(@NonNull OnDismissListener listener) {
            this.dismissListener = listener;
            return this;
        }

        public Builder cancelListener(@NonNull OnCancelListener listener) {
            this.cancelListener = listener;
            return this;
        }

        public Builder keyListener(@NonNull OnKeyListener listener) {
            this.keyListener = listener;
            return this;
        }

        @UiThread
        public MaterialDialog build() {
            return new MaterialDialog(this);
        }

        @UiThread
        public MaterialDialog show() {
            MaterialDialog dialog = build();
            dialog.show();
            return dialog;
        }
    }
}