package io.gamedock.sdk.utils.device;

import android.app.ActivityManager;
import android.content.ActivityNotFoundException;
import android.content.Context;
import android.content.Intent;
import android.content.SharedPreferences;
import android.content.pm.PackageInfo;
import android.content.pm.PackageManager;
import android.database.Cursor;
import android.net.Uri;
import android.os.Handler;
import android.os.Looper;
import android.provider.Settings;

import androidx.annotation.NonNull;

import java.util.Iterator;
import java.util.List;
import java.util.Locale;

import io.gamedock.sdk.GamedockSDK;
import io.gamedock.sdk.R;
import io.gamedock.sdk.events.Event;
import io.gamedock.sdk.utils.dialog.MaterialDialog;
import io.gamedock.sdk.utils.dialog.MaterialStyledDialog;
import io.gamedock.sdk.utils.dialog.internal.DialogAction;
import io.gamedock.sdk.utils.logging.LoggingUtil;
import io.gamedock.sdk.utils.storage.StorageUtil;

import static android.content.Context.ACTIVITY_SERVICE;

/**
 * Utility Class that handles the logic regarding app install reporting
 */
public class DeviceUtil {

    private enum AppStart {
        FIRST_TIME, UPDATE, NORMAL
    }

    /**
     * Method that reports on the status of an application.
     * It can either check if it is freshly installed or has been updated.
     * Sends an event to the backend with the appropriate information.
     *
     * @param context The activity context.
     */
    public static void reportAppStatus(Context context) {
        try {

            Event event = new Event(context);

            DeviceUtil.AppStart start = checkAppStart(context);

            if (start == DeviceUtil.AppStart.FIRST_TIME) {
                LoggingUtil.d("First time start");

                event.setName("install");
                event.addCustomData("store", GamedockSDK.getInstance(context).getStorageUtil().getString(StorageUtil.Keys.Store, null));

                String referrer = getInstallReferrer(context);
                if (referrer != null) {
                    event.addCustomData("referrer", referrer);
                }

                GamedockSDK.getInstance(context).trackEvent(event, null);
            } else if (start == DeviceUtil.AppStart.UPDATE) {
                LoggingUtil.d("Update start ");
                event.setName("update");
                GamedockSDK.getInstance(context).trackEvent(event, null);
            }

            if (start == DeviceUtil.AppStart.NORMAL) {
                LoggingUtil.d("Normal start");
            } else {
                LoggingUtil.d("Special start");
            }
        } catch (Exception e) {
            e.printStackTrace();
        }
    }

    /**
     * Method that internally check if an apps staus has changed based on the version codes stored.
     *
     * @param context The activity context.
     * @return Returns the status of the application. Can be FIRST_TIME, UPDATE or NORMAL.
     */
    private static DeviceUtil.AppStart checkAppStart(Context context) {
        DeviceUtil.AppStart appStart = DeviceUtil.AppStart.NORMAL;
        try {
            PackageInfo pInfo = context.getPackageManager().getPackageInfo(context.getPackageName(), 0);

            int lastVersionCode = GamedockSDK.getInstance(context).getStorageUtil().getInt(StorageUtil.Keys.AppVersion, -1);

            int currentVersionCode = pInfo.versionCode;

            LoggingUtil.d("current version code " + currentVersionCode + " previous vc " + lastVersionCode);

            appStart = checkVersionCode(context, currentVersionCode, lastVersionCode);

            // Update version in preferences
            GamedockSDK.getInstance(context).getStorageUtil().putInt(StorageUtil.Keys.AppVersion, currentVersionCode);

        } catch (PackageManager.NameNotFoundException e) {
            LoggingUtil.w("Unable to determine current app version from pacakge manager. Defenisvely assuming normal app start.");
        }
        return appStart;
    }

    /**
     * Method that checks if the application version codes have changed or if they are present.
     *
     * @param context            The application context.
     * @param currentVersionCode The current version code received from the app in the {@link SharedPreferences}.
     * @param lastVersionCode    The version code stored in the {@link SharedPreferences}.
     * @return Returns the status of the application. Can be FIRST_TIME, UPDATE or NORMAL.
     */
    private static DeviceUtil.AppStart checkVersionCode(Context context, int currentVersionCode, int lastVersionCode) {
        boolean install = GamedockSDK.getInstance(context).getStorageUtil().getBoolean(StorageUtil.Keys.SpilSdkInstall, true);

        if ((lastVersionCode == -1) && install) {
            GamedockSDK.getInstance(context).getStorageUtil().putBoolean(StorageUtil.Keys.SpilSdkInstall, false);
            return DeviceUtil.AppStart.FIRST_TIME;
        } else if (lastVersionCode < currentVersionCode) {
            return DeviceUtil.AppStart.UPDATE;
        } else if (lastVersionCode > currentVersionCode) {
            LoggingUtil.w("Current version code (" + currentVersionCode
                    + ") is less then the one recognized on last startup ("
                    + lastVersionCode
                    + "). Defenisvely assuming normal app start.");
            return DeviceUtil.AppStart.NORMAL;
        } else {
            return DeviceUtil.AppStart.NORMAL;
        }
    }

    /**
     * Method that retrieves the install referrer link with which the application has been installed.
     *
     * @param context The activity context.
     * @return Returns the referrer link stored in the {@link StorageUtil}.
     */
    private static String getInstallReferrer(Context context) {
        return GamedockSDK.getInstance(context).getStorageUtil().getString("installReferrer", null);
    }


    /**
     * Method that retrieves the Application Name.
     *
     * @return The name of the application that is currently running the Gamedock SDK.
     */
    public static String getAppName(Context context) {
        ActivityManager am = (ActivityManager) context.getSystemService(ACTIVITY_SERVICE);
        List l = null;
        if (am != null) {
            l = am.getRunningAppProcesses();
            Iterator i = l.iterator();
            PackageManager pm = context.getPackageManager();
            while (i.hasNext()) {
                ActivityManager.RunningAppProcessInfo info = (ActivityManager.RunningAppProcessInfo) (i.next());
                try {
                    CharSequence c = pm.getApplicationLabel(pm.getApplicationInfo(info.processName, PackageManager.GET_META_DATA));
                    return c.toString();
                } catch (Exception e) {
                    //Name Not FOund Exception
                }
            }
        }

        return null;
    }

    /**
     * Method that check if the string (event name) is a valid alpha numeric string.
     *
     * @param eventName The string that has to be verified
     * @return True or false depending on the check.
     */
    public static boolean isAlphaNumeric(String eventName) {
        String pattern = "^[a-zA-Z0-9_-]*$";

        return eventName.matches(pattern);
    }

    /**
     * Method that gets the debug mode has been enabled.
     * It check if the "debug" flag is set in the content provider.
     *
     * @return Returns {@code true} if debug mode is enabled.
     */
    public static boolean getDebugMode(Context context) {
        String URL = "content://io.gamedock.sdk.android.application.provider.GamedockProvider";

        try {
            Uri dbUri = Uri.parse(URL);
            Cursor cursor = context.getContentResolver().query(dbUri, null, null, null, null);

            if (cursor != null && cursor.moveToFirst()) {
                String debugValue = cursor.getString(cursor.getColumnIndex("debug"));

                cursor.close();

                return debugValue.equals("true");
            }
        } catch (SecurityException e) {
            e.printStackTrace();
        }

        return false;
    }

    /**
     * Method used to show a native dialog.
     *
     * @param context    The activity context.
     * @param title      The title of the dialog.
     * @param message    The message of the dialog.
     * @param buttonText The text of the button.
     */
    public static void showNativeDialog(final Context context, final String title, final String message, final String buttonText) {
        final int dialogHeader;
        int resourceId = context.getResources().getIdentifier("permission_header_custom", "drawable", context.getPackageName());

        if (resourceId != 0) {
            dialogHeader = resourceId;
        } else {
            dialogHeader = R.drawable.permission_header;
        }

        Handler handler = new Handler(Looper.getMainLooper());
        handler.post(new Runnable() {
            @Override
            public void run() {
                MaterialStyledDialog.Builder builder = new MaterialStyledDialog.Builder(context)
                        .setTitle(title)
                        .setDescription(message)
                        .setHeaderDrawable(dialogHeader)
                        .autoDismiss(false)
                        .withDialogAnimation(true)
                        .setPositiveText(buttonText)
                        .onPositive(new MaterialDialog.SingleButtonCallback() {
                            @Override
                            public void onClick(@NonNull MaterialDialog materialDialog, @NonNull DialogAction dialogAction) {
                                materialDialog.dismiss();
                            }
                        });

                builder.show();
            }
        });
    }

    public static String formatTime(long timeSeconds) {
        if (timeSeconds < 0) {
            timeSeconds = 0;
        }

        int hours = (int) (timeSeconds / 3600);
        int minutes = (int) ((timeSeconds % 3600) / 60);
        int seconds = (int) ((timeSeconds % 3600) % 60);

        return String.format(Locale.ENGLISH,"%02d", hours) + ":" + String.format(Locale.ENGLISH, "%02d", minutes) + ":" + String.format(Locale.ENGLISH, "%02d", seconds);
    }

    public static boolean isClass(String className) {
        try  {
            Class.forName(className);
            return true;
        }  catch (NoClassDefFoundError | ClassNotFoundException e) {
            return false;
        }
    }

    public static void showAppSettings(Context context) {
        try {
            Intent intent = new Intent(Settings.ACTION_APPLICATION_DETAILS_SETTINGS);
            Uri uri = Uri.fromParts("package", context.getPackageName(), null);
            intent.setData(uri);
            context.startActivity(intent);
        } catch (NullPointerException | ActivityNotFoundException e) {
            e.printStackTrace();
        }
    }
}