package io.gamedock.sdk.utils.IAP;


import android.app.Activity;
import android.content.Context;
import android.os.Bundle;
import android.os.RemoteException;

import androidx.annotation.NonNull;
import androidx.annotation.Nullable;

import com.android.billingclient.api.BillingClient;
import com.android.billingclient.api.BillingClientStateListener;
import com.android.billingclient.api.BillingResult;
import com.android.billingclient.api.PurchasesUpdatedListener;
import com.android.billingclient.api.SkuDetailsParams;
import com.android.billingclient.api.SkuDetailsResponseListener;
import com.google.gson.JsonElement;
import com.google.gson.JsonObject;

import org.json.JSONException;
import org.json.JSONObject;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;

import io.gamedock.sdk.GamedockSDK;
import io.gamedock.sdk.config.internal.GamedockConfigManager;
import io.gamedock.sdk.events.Event;
import io.gamedock.sdk.events.EventManager;
import io.gamedock.sdk.events.internal.IAPEvent;
import io.gamedock.sdk.gamedata.promotions.PromotionsManager;
import io.gamedock.sdk.models.iap.IAPProduct;
import io.gamedock.sdk.utils.IAP.google.IabHelper;
import io.gamedock.sdk.utils.IAP.google.IabResult;
import io.gamedock.sdk.utils.IAP.google.Inventory;
import io.gamedock.sdk.utils.IAP.google.Purchase;
import io.gamedock.sdk.utils.IAP.google.SkuDetails;
import io.gamedock.sdk.utils.device.DeviceUtil;
import io.gamedock.sdk.utils.logging.LoggingUtil;
import io.reactivex.Observable;
import io.reactivex.ObservableEmitter;
import io.reactivex.ObservableOnSubscribe;
import io.reactivex.schedulers.Schedulers;

/**
 * Utility Class used to handle all the application IAP.
 */
public class IAPUtil {

    public static String store;

    public static IabHelper mHelper;
    public static BillingClient billingClient;
    private static boolean isInitialised;
    private static GoogleIAPLibrary iapLibrary = GoogleIAPLibrary.NONE;
    private static final HashMap<String, List<HashMap<String, String>>> queuedIAPPurchasedSkus = new HashMap<>();
    private static HashMap<String, Boolean> validatedSubscriptions = new HashMap<>();

    /**
     * Method that initialises the IAP Library.
     *
     * @param context The activity context.
     */
    public static void initIAP(final Context context, String store) {
        IAPUtil.store = store;
        if (store == null || store.equals("GooglePlay")) {
            initGoogleIAP(context);
        } else if (store.equals("Amazon")) {

        }
    }

    private static void initGoogleIAP(Context context) {
        String googleIAPKey = GamedockConfigManager.getInstance(context).googleServices.IAPKey;

        if (context != null && googleIAPKey != null && googleIAPKey.length() > 0 && !isInitialised) {
            if (DeviceUtil.isClass("com.android.vending.billing.IInAppBillingService$Stub")) {
                try {
                    iapLibrary = GoogleIAPLibrary.AIDL;
                    mHelper = new IabHelper(context, googleIAPKey);
                    // Set up IAP
                    mHelper.startSetup(new IabHelper.OnIabSetupFinishedListener() {
                        public void onIabSetupFinished(IabResult result) {
                            if (!result.isSuccess()) {
                                // There was a problem.
                                LoggingUtil.d("GamedockSDK IAP: Problem setting up In-app Billing: " + result);
                            } else {
                                LoggingUtil.d("GamedockSDK IAP: In-app Billing with AIDL is fully set up!");
                                isInitialised = true;
                            }
                        }
                    });
                } catch (Error e) {
                    e.printStackTrace();
                    LoggingUtil.e("GamedockSDK IAP: Could not initialise Google in-app purchases library.");
                }
            } else if (DeviceUtil.isClass("com.android.billingclient.api.BillingClient")) {
                try {
                    billingClient = BillingClient.newBuilder(context)
                            .enablePendingPurchases()
                            .setListener(new PurchasesUpdatedListener() {
                                @Override
                                public void onPurchasesUpdated(@NonNull BillingResult billingResult, @Nullable List<com.android.billingclient.api.Purchase> list) {
                                }
                            })
                            .build();

                    billingClient.startConnection(new BillingClientStateListener() {
                        @Override
                        public void onBillingSetupFinished(@NonNull BillingResult billingResult) {
                            if (billingResult.getResponseCode() == BillingClient.BillingResponseCode.OK) {
                                isInitialised = true;
                                iapLibrary = GoogleIAPLibrary.BILLING_LIBRARY;

                                LoggingUtil.d("GamedockSDK IAP: In-app Billing with Billing Library is fully set up!");
                            } else {
                                LoggingUtil.d("GamedockSDK IAP: Failed to initialize Billing Library");
                            }
                        }

                        @Override
                        public void onBillingServiceDisconnected() {

                        }
                    });
                } catch (Exception | Error e) {
                    e.printStackTrace();
                    LoggingUtil.e("GamedockSDK IAP: Could not initialise Google in-app purchases library.");
                }
            } else {
                LoggingUtil.e("GamedockSDK IAP: Could not initialise Google in-app purchases library. No IAP library found. Please either use AIDL or the Billing Library");
            }
        } else {
            LoggingUtil.e("GamedockSDK IAP: Could not initialise Google in-app purchases library. If you're using Google in-app purchases please make sure that you fill in the googleIAPKey in the defaultGameConfig.json");
        }
    }

    /**
     * Method that checks the IAP and adds the "localPrice", "localCurrency" and "token" parameters to the IAP Event.
     *
     * @param context The activity context.
     * @param event   The {@link Event} containing the IAP information.
     * @param SKU     The id for the IAP product.
     */
    public static void processIapPurchaseEvent(final Context context, final Event event, final String SKU) {
        HashMap<String, String> parameters = new HashMap<>();

        JsonObject result = event.customData.getAsJsonObject();
        Set<Map.Entry<String, JsonElement>> entrySet = result.entrySet();

        for (Map.Entry<String, JsonElement> entry : entrySet) {
            if (store.equals("GooglePlay")) {
                if (entry.getKey().equals("localPrice") || entry.getKey().equals("localCurrency")) {
                    continue;
                }
            }
            parameters.put(entry.getKey(), entry.getValue().getAsString());
        }

        if (store == null || store.equals("GooglePlay")) {
            processGoogleIapPurchaseEvent(context, event, SKU, parameters);
        } else if (store.equals("Amazon")) {
            processAmazonIapPurchaseEvent(context, event, SKU, parameters);
        }
    }

    private static void processGoogleIapPurchaseEvent(final Context context, final Event event, final String SKU, HashMap<String, String> parameters) {
        if (iapLibrary == GoogleIAPLibrary.AIDL) {
            processGoogleIapPurchaseEventAIDL(context, event, SKU, parameters);
        } else if (iapLibrary == GoogleIAPLibrary.BILLING_LIBRARY) {
            processGoogleIapPurchaseEventBillingLibrary(context, event, SKU, parameters);
        } else {
            processGoogleIapPurchaseEventNoInfo(context, event);
        }
    }

    private static void processGoogleIapPurchaseEventAIDL(final Context context, final Event event, final String SKU, HashMap<String, String> parameters) {
        LoggingUtil.d("GamedockSDK IAP: Google IAP Purchase detected, retrieving price and currency for SKU \"" + SKU + "\" using AIDL");

        // Query the server with the supplied SKU, this will return a list of products with either one or zero entries
        // (there should never be multiple products for an SKU but the Google Play Store returns a list anyway..)
        // If a product is found we'll get its localPrice and localCurrency and send an IAPPurchased event to the Gamedock back-end.
        try {
            if (mHelper != null && isInitialised && SKU != null) { // is IAP initialised properly and SKU filled in?

                // Store the parameters sent with this event so when we get a response from the server
                // with a product we can create and send an event include the original parameters.
                synchronized (queuedIAPPurchasedSkus) { // Make sure we're not interfering with any thread running the response handler that is removing queued events
                    List<HashMap<String, String>> queuedEventsBySKU = queuedIAPPurchasedSkus.get(SKU);
                    // Put all parameters for pending events with the same SKU into a list. This way, if a second IAP with the
                    // same SKU is made before we've got a response with localPrice and localCurrency for the first IAP
                    // we can send both events to the back-end at once as soon as we get the response with data for the SKU.
                    if (queuedEventsBySKU == null) {
                        queuedEventsBySKU = new ArrayList<>();
                    }
                    queuedEventsBySKU.add(parameters);
                    queuedIAPPurchasedSkus.put(SKU, queuedEventsBySKU);
                }

                // Response handler for query by SKU
                final IabHelper.QueryInventoryFinishedListener mQueryFinishedListener = new IabHelper.QueryInventoryFinishedListener() {
                    public void onQueryInventoryFinished(IabResult result, Inventory inventory) {
                        try {
                            if (result.isFailure()) {
                                LoggingUtil.d("GamedockSDK IAP: queryAvailableProducts onQueryInventoryFinished FAILED");

                                // handle error
                                return;
                            }

                            LoggingUtil.d("GamedockSDK IAP: queryAvailableProducts onQueryInventoryFinished SUCCESS");

                            // For each product associated with this SKU get the price and currency and send an event to the Gamedock back-end
                            // Even though this is a list there should never be more than one entry.
                            List<SkuDetails> products = inventory.getAllProducts();
                            if (products.size() > 1) {
                                LoggingUtil.e("GamedockSDK IAP: Google IAP query by SKU returned more than one product, this is not supposed to happen and may cause bugs! Please inform the Gamedock SDK team.");
                            }

                            if (products.size() == 0) {
                                LoggingUtil.e("GamedockSDK IAP: Provided SKU could not retrieve any products. Please revise the SKU parameter that is being sent to the SDK!");
                            }
                            for (SkuDetails product : products) {

                                // Remove any queued events with this SKU from the queue
                                List<HashMap<String, String>> queuedIAPPurchasedEvents;
                                synchronized (queuedIAPPurchasedSkus) {  // Make sure we're not interfering with any thread that is adding events to the queue
                                    queuedIAPPurchasedEvents = queuedIAPPurchasedSkus.remove(product.getSku());
                                }
                                // It can actually happen that a request was made for a SKU but that a response handler handling the same SKU has already
                                // handled the event in the mean time so the request was actually redundant and can be ignored
                                if (queuedIAPPurchasedEvents != null && queuedIAPPurchasedEvents.size() > 0) {
                                    // For each event that was queued for this SKU send an iapPurchased event to the back-end with the original parameters + localPrice and localCurrency
                                    for (HashMap<String, String> paramsQueued : queuedIAPPurchasedEvents) {
                                        IAPEvent iapEvent = new IAPEvent(context);
                                        iapEvent.setIAPPurchasedEvent();

                                        if (paramsQueued != null) {
                                            if (!paramsQueued.isEmpty()) {
                                                for (Map.Entry<String, String> entry : paramsQueued.entrySet()) {
                                                    iapEvent.addCustomData(entry.getKey(), entry.getValue());
                                                }
                                            }
                                        }

                                        String localPrice = product.getPrice();
                                        // Google may add a currency symbol before the price, remove it
                                        if (localPrice != null && localPrice.length() > 0) {
                                            String newLocalPrice = "";
                                            for (char c : localPrice.toCharArray()) {
                                                if (Character.isDigit(c) || c == '.' || c == ',') {
                                                    newLocalPrice += c;
                                                }
                                            }
                                            localPrice = newLocalPrice;
                                        }

                                        iapEvent.addCustomData("localPrice", localPrice);
                                        iapEvent.addCustomData("localCurrency", product.getPriceCurrencyCode());
                                        iapEvent.addCustomData("rawPrice", product.getPrice());

                                        Purchase purchase = inventory.getPurchase(product.getSku());

                                        if (!iapEvent.customData.has("token") || iapEvent.customData.get("token") == null || iapEvent.customData.get("token").getAsString().equals("")) {
                                            if (purchase != null) {
                                                iapEvent.addCustomData("token", purchase.getToken());
                                            } else {
                                                String transactionId = iapEvent.getCustomDataAsString("transactionId");
                                                if (!transactionId.startsWith("GPA.")) {
                                                    iapEvent.addCustomData("token", transactionId);
                                                }
                                            }
                                        }

                                        checkAndSendIAPEvent(iapEvent, context);
                                    }
                                }
                            }
                        } catch (NullPointerException e) {
                            e.printStackTrace();
                            LoggingUtil.w("GamedockSDK IAP: One or more required values has a null value. Please verify that you send the correct IAP information!");
                        }
                    }
                };

                // Query Google Play Store for product(s) by SKU.
                // Create and send the query and pass the response handler
                final ArrayList<String> additionalSkuList = new ArrayList<>();
                additionalSkuList.add(SKU);

                ((Activity) context).runOnUiThread(new Thread(new Runnable() {
                    public void run() {
                        try {
                            mHelper.queryInventoryAsync(true, additionalSkuList, null, mQueryFinishedListener);
                        } catch (IabHelper.IabAsyncInProgressException ex) {
                            LoggingUtil.d("GamedockSDK IAP: Error: " + ex.toString());
                        }
                    }
                }));
            } else {
                if (mHelper == null) {
                    LoggingUtil.e("GamedockSDK IAP: GamedockSDK could not retrieve localPrice and localCurrency because IAP was not initialised properly. If you're using Google in-app purchases then please make sure that you fill in the googleIAPKey in the defaultGameConfig.json");
                }
                if (SKU == null) {
                    LoggingUtil.e("GamedockSDK IAP: GamedockSDK could not retrieve localPrice and localCurrency because SKU was null.");
                }

                checkAndSendIAPEvent(event, context);
            }
        } catch (NullPointerException e) {
            e.printStackTrace();
            LoggingUtil.w("GamedockSDK IAP: One or more required values has a null value. Please verify that you send the correct IAP information!");
        }
    }

    private static void processGoogleIapPurchaseEventBillingLibrary(final Context context, final Event event, final String SKU, final HashMap<String, String> parameters) {
        LoggingUtil.d("GamedockSDK IAP: Google IAP Purchase detected, retrieving price and currency for SKU \"" + SKU + "\" using Billing Library");

        try {
            List<String> skuList = new ArrayList<> ();
            skuList.add(SKU);
            SkuDetailsParams.Builder params = SkuDetailsParams.newBuilder();

            if (event.customData.has("isSubscription")) {
                params.setSkusList(skuList).setType(BillingClient.SkuType.SUBS);
            } else {
                params.setSkusList(skuList).setType(BillingClient.SkuType.INAPP);
            }

            billingClient.querySkuDetailsAsync(params.build(), new SkuDetailsResponseListener() {
                @Override
                public void onSkuDetailsResponse(@NonNull BillingResult billingResult, @Nullable List<com.android.billingclient.api.SkuDetails> list) {
                    try {
                        if (billingResult.getResponseCode() == BillingClient.BillingResponseCode.OK && list != null) {
                            for (com.android.billingclient.api.SkuDetails skuDetails : list) {
                                if (skuDetails.getSku().equals(SKU)) {
                                    IAPEvent iapEvent = new IAPEvent(context);
                                    iapEvent.setIAPPurchasedEvent();

                                    if (parameters != null) {
                                        if (!parameters.isEmpty()) {
                                            for (Map.Entry<String, String> entry : parameters.entrySet()) {
                                                iapEvent.addCustomData(entry.getKey(), entry.getValue());
                                            }
                                        }
                                    }

                                    String localPrice = skuDetails.getPrice();
                                    // Google may add a currency symbol before the price, remove it
                                    if (localPrice.length() > 0) {
                                        String newLocalPrice = "";
                                        for (char c : localPrice.toCharArray()) {
                                            if (Character.isDigit(c) || c == '.' || c == ',') {
                                                newLocalPrice += c;
                                            }
                                        }
                                        localPrice = newLocalPrice;
                                    }

                                    iapEvent.addCustomData("localPrice", localPrice);
                                    iapEvent.addCustomData("localCurrency", skuDetails.getPriceCurrencyCode());
                                    iapEvent.addCustomData("rawPrice", skuDetails.getPrice());

                                    checkAndSendIAPEvent(iapEvent, context);
                                }
                            }
                        }
                    } catch (Exception e) {
                        e.printStackTrace();
                        LoggingUtil.e("GamedockSDK IAP: GamedockSDK could not retrieve localPrice and localCurrency because IAP was not initialised properly. If you're using Google in-app purchases then please make sure that you fill in the googleIAPKey in the defaultGameConfig.json");
                        checkAndSendIAPEvent(event, context);
                    }
                }
            });
        } catch (NoClassDefFoundError | NullPointerException e) {
            e.printStackTrace();
            LoggingUtil.e("GamedockSDK IAP: GamedockSDK could not retrieve localPrice and localCurrency because IAP was not initialised properly. If you're using Google in-app purchases then please make sure that you fill in the googleIAPKey in the defaultGameConfig.json");
            checkAndSendIAPEvent(event, context);
        }
    }

    private static void processGoogleIapPurchaseEventNoInfo(final Context context, final Event event) {
        LoggingUtil.d("GamedockSDK IAP: No billing implementation detected. Will send event without price and currency code.");

        checkAndSendIAPEvent(event, context);
    }

    public static String getGooglePlayIapDetails(Context context, ArrayList<String> skuIds) {
        if (!isInitialised) {
            LoggingUtil.e("GamedockSDK IAP: GamedockSDK could not retrieve localPrice and localCurrency because IAP was not initialised properly. If you're using Google in-app purchases then please make sure that you fill in the googleIAPKey in the defaultGameConfig.json");
            return "[]";
        }

        if (skuIds == null) {
            LoggingUtil.e("GamedockSDK IAP: GamedockSDK could not retrieve localPrice and localCurrency because SKU list was null.");
            return "[]";
        }

        final ArrayList<IAPProduct> iapProducts = new ArrayList<>();

        try {
            if (iapLibrary == GoogleIAPLibrary.AIDL) {
                Bundle querySkus = new Bundle();
                querySkus.putStringArrayList(IabHelper.GET_SKU_DETAILS_ITEM_LIST, skuIds);
                Bundle skuDetails = mHelper.mService.getSkuDetails(3, context.getPackageName(), IabHelper.ITEM_TYPE_INAPP, querySkus);

                if (!skuDetails.containsKey(IabHelper.RESPONSE_GET_SKU_DETAILS_LIST)) {
                    int response = mHelper.getResponseCodeFromBundle(skuDetails);
                    if (response != IabHelper.BILLING_RESPONSE_RESULT_OK) {
                        LoggingUtil.d("GamedockSDK IAP: getSkuDetails() failed: " + IabHelper.getResponseDesc(response));
                        return "[]";
                    } else {
                        LoggingUtil.d("GamedockSDK IAP: getSkuDetails() returned a bundle with neither an error nor a detail list.");
                        return "[]";
                    }
                }

                ArrayList<String> responseList = skuDetails.getStringArrayList(
                        IabHelper.RESPONSE_GET_SKU_DETAILS_LIST);

                if (responseList != null) {
                    for (String thisResponse : responseList) {
                        SkuDetails product = new SkuDetails(IabHelper.ITEM_TYPE_INAPP, thisResponse);

                        IAPProduct iapProduct = new IAPProduct();
                        iapProduct.skuId = product.getSku();
                        iapProduct.price = product.getPrice();
                        iapProduct.currency = product.getPriceCurrencyCode();

                        iapProducts.add(iapProduct);
                    }
                }
            } else if (iapLibrary == GoogleIAPLibrary.BILLING_LIBRARY) {
                iapProducts.addAll(getIapDetailsAsync(billingClient, skuIds)
                        .subscribeOn(Schedulers.newThread())
                        .blockingSingle());
            }
        } catch (JSONException | RuntimeException | Error | RemoteException e) {
            LoggingUtil.d("GamedockSDK IAP: getSkuDetails() failed: \n" + e.getMessage());
        }

        return GamedockSDK.getInstance(context).getGson().toJson(iapProducts);
    }

    public static Observable<ArrayList<IAPProduct>> getIapDetailsAsync(final BillingClient billingClient, final ArrayList<String> skuIds) {
        return Observable.create(new ObservableOnSubscribe<ArrayList<IAPProduct>>() {
            @Override
            public void subscribe(@io.reactivex.annotations.NonNull final ObservableEmitter<ArrayList<IAPProduct>> observableEmitter) throws Exception {
                final ArrayList<IAPProduct> iapProducts = new ArrayList<>();

                SkuDetailsParams.Builder params = SkuDetailsParams.newBuilder();
                params.setSkusList(skuIds).setType(BillingClient.SkuType.INAPP);

                if (billingClient == null) {
                    observableEmitter.onNext(iapProducts);
                    observableEmitter.onComplete();
                    return;
                }

                billingClient.querySkuDetailsAsync(params.build(), new SkuDetailsResponseListener() {
                    @Override
                    public void onSkuDetailsResponse(@NonNull BillingResult billingResult, @Nullable List<com.android.billingclient.api.SkuDetails> list) {
                        if (billingResult.getResponseCode() == BillingClient.BillingResponseCode.OK && list != null) {
                            for (com.android.billingclient.api.SkuDetails skuDetails : list) {
                                IAPProduct iapProduct = new IAPProduct();
                                iapProduct.skuId = skuDetails.getSku();
                                iapProduct.price = skuDetails.getPrice();
                                iapProduct.currency = skuDetails.getPriceCurrencyCode();

                                iapProducts.add(iapProduct);
                            }
                        }

                        observableEmitter.onNext(iapProducts);
                        observableEmitter.onComplete();
                    }
                });
            }
        });
    }

    private static void processAmazonIapPurchaseEvent(final Context context, final Event event, final String SKU, HashMap<String, String> parameters) {
        try {
            IAPEvent iapEvent = new IAPEvent(context);
            iapEvent.setIAPPurchasedEvent();

            if (parameters != null) {
                if (!parameters.isEmpty()) {
                    for (Map.Entry<String, String> entry : parameters.entrySet()) {
                        iapEvent.addCustomData(entry.getKey(), entry.getValue());
                    }
                }
            }
            checkAndSendIAPEvent(iapEvent, context);
        } catch (Exception e) {
            checkAndSendIAPEvent(event, context);
        }

    }

    public static String getAmazonIapDetails(Context context, ArrayList<String> skuIds) {
        return "";
    }

    /**
     * Method that checks for IAP duplicate events that could be sent by the SDK
     *
     * @param event   The {@link Event} containing the IAP information.
     * @param context The activity context.
     */
    private synchronized static void checkAndSendIAPEvent(Event event, Context context) {
        String SKU = event.getCustomDataAsString("skuId");
        String transactionId = event.getCustomDataAsString("transactionId");

        String token = null;
        if (event.customData.has("token")) {
            token = event.getCustomDataAsString("token");
        }

        if (token == null || token.equals("")) {
            LoggingUtil.w("GamedockSDK IAP: IAP Purchased is missing the token! Canceling sending the original event! Sending a NO TOKEN event!");
            event.setName("iapPurchasedNoToken");
            GamedockSDK.getInstance(context).trackEvent(event, null);
            return;
        }

        ArrayList<Event> recentIapEvents = EventManager.getInstance(context).getRecentIAPEvents().list;
        for (int i = 0; i < recentIapEvents.size(); i++) {
            if (SKU.equals(recentIapEvents.get(i).getCustomDataAsString("skuId")) &&
                    transactionId.equals(recentIapEvents.get(i).getCustomDataAsString("transactionId")) &&
                    recentIapEvents.get(i).customData.has("token") &&
                    token.equals(recentIapEvents.get(i).getCustomDataAsString("token"))) {

                LoggingUtil.w("GamedockSDK IAP: IAP Purchased event already sent! Removing duplicate! Creating duplicate event for tracking!");
                event.setName("iapPurchasedDuplicate");
                GamedockSDK.getInstance(context).trackEvent(event, null);
                return;
            }
        }

        if (PromotionsManager.getInstance(context).getPackagePromotionObject(SKU) != null) {
            int amountPurchased = PromotionsManager.getInstance(context).getPackagePromotionObject(SKU).getAmountPurchased();
            PromotionsManager.getInstance(context).getPackagePromotionObject(SKU).setAmountPurchased(amountPurchased + 1);
            PromotionsManager.getInstance(context).sendBoughtPromotion(SKU);
        }

        EventManager.getInstance(context).getRecentIAPEvents().add(event);
        GamedockSDK.getInstance(context).trackEvent(event, null);
    }


    /**
     * Method that notifies the application if an IAP has been validated correctly with the Gamedock Backend.
     *
     * @param context The activity context.
     * @param iapJSON The response from Gamedock containing the validation.
     */
    public static void processValidateIAP(Context context, String iapJSON) {
        try {
            JSONObject iap = new JSONObject(iapJSON);

            boolean valid = iap.getBoolean("valid");
            boolean isSubscription = false;
            String skuId = "";
            String message = null;

            if (iap.has("reason")) {
                message = iap.getString("reason");
            }

            if (iap.has("isSubscription")) {
                isSubscription = iap.getBoolean("isSubscription");
            }

            if (iap.has("skuId")) {
                skuId = iap.getString("skuId");
            }

            if (valid) {
                if (isSubscription) {
                    validatedSubscriptions.put(skuId, true);
                }
                GamedockSDK.getInstance(context).getIapCallbacks().iapValid(skuId);
            } else {
                GamedockSDK.getInstance(context).getIapCallbacks().iapInvalid(skuId, message);
            }
        } catch (JSONException e) {
            e.printStackTrace();
        }
    }

    public static void validateSubscription(Context context, String skuId, String transactionId, String token) {
        if (validatedSubscriptions.containsKey(skuId)) {
            GamedockSDK.getInstance(context).getIapCallbacks().iapValid(skuId);
            return;
        }

        IAPEvent iapEvent = new IAPEvent(context);
        iapEvent.setValidateSubscription();

        iapEvent.addCustomData("skuId", skuId);
        iapEvent.addCustomData("transactionId", transactionId);
        iapEvent.addCustomData("token", token);

        GamedockSDK.getInstance(context).trackEvent(iapEvent, null);
    }

    private enum GoogleIAPLibrary {
        NONE,
        AIDL,
        BILLING_LIBRARY
    }
}
