package io.gamedock.sdk.userdata.playerdata.functions;

import android.content.Context;

import java.util.ArrayList;
import java.util.Map;

import io.gamedock.sdk.gamedata.GamedockGameDataManager;
import io.gamedock.sdk.models.gamedata.items.Item;
import io.gamedock.sdk.models.userdata.UserData;
import io.gamedock.sdk.models.userdata.inventory.Inventory;
import io.gamedock.sdk.models.userdata.inventory.PlayerItem;
import io.gamedock.sdk.models.userdata.inventory.UniquePlayerItem;
import io.gamedock.sdk.models.userdata.wallet.PlayerCurrency;
import io.gamedock.sdk.models.userdata.wallet.Wallet;
import io.gamedock.sdk.userdata.UserDataManager;
import io.gamedock.sdk.userdata.playerdata.PlayerDataManager;
import io.gamedock.sdk.utils.storage.StorageUtil;

/**
 * Internal class used by the {@link PlayerDataManager} to perform all processing functions.
 */
public class PlayerDataProcessing {

    private static final int WALLET_CALCULATE_CASE_INIT = 1;
    private static final int WALLET_CALCULATE_CASE_UPDATE = 2;

    /**
     * Method that handles the {@link Wallet} initialisation that comes from the Gamedock backend.
     *
     * @param context        The activity context.
     * @param userData       The {@link UserData} object that contains the {@link Wallet} that needs to be updated.
     * @param receivedWallet The {@link Wallet} object that has been received from the Gamedock backend.
     * @param externalChange Flag informing if another device did updates on the backend.
     * @return Returns the {@link UserData} object initialised with the {@link Wallet} information received from the Gamedock backend.
     */
    public static UserData walletInit(Context context, UserData userData, Wallet receivedWallet, boolean externalChange) {
        if (externalChange && !receivedWallet.getCurrenciesMap().isEmpty()) {
            userData = calculateWallet(userData, receivedWallet, WALLET_CALCULATE_CASE_INIT);
        } else if (userData.getWallet().getOffset() == 0 && receivedWallet.getOffset() != 0) {
            if (receivedWallet.getCurrenciesMap().isEmpty()) {
                userData = UserDataManager.getInstance(context).getPlayerDataManager().setWalletInitialValues(userData);
                if (userData == null) {
                    return null;
                } else {
                    UserDataManager.getInstance(context).getPlayerDataManager().initialValue = true;
                }
            } else if (!receivedWallet.getCurrenciesMap().isEmpty() && !UserDataManager.getInstance(context).getPlayerDataManager().storageUtil.getBoolean(StorageUtil.Keys.WalletInit, false)) {
                userData = calculateWallet(userData, receivedWallet, WALLET_CALCULATE_CASE_INIT);
            }
        } else if (userData.getWallet().getOffset() < receivedWallet.getOffset() && !receivedWallet.getCurrenciesMap().isEmpty()) {
            userData = calculateWallet(userData, receivedWallet, WALLET_CALCULATE_CASE_UPDATE);
        }

        UserDataManager.getInstance(context).getPlayerDataManager().storageUtil.putBoolean(StorageUtil.Keys.WalletInit, true);

        //Update offset and logic from the server
        userData.getWallet().setOffset(receivedWallet.getOffset());
        userData.getWallet().setLogic(receivedWallet.getLogic());

        return userData;
    }

    /**
     * Method that handles the updating of the {@link Wallet} object with information coming from the Gamedock backend.
     *
     * @param userData       The {@link UserData} object that contains the {@link Wallet} that needs to be updated.
     * @param receivedWallet The {@link Wallet} object that has been received from the Gamedock backend.
     * @return Returns the {@link UserData} object updated with the {@link Wallet} information received from the Gamedock backend.
     */
    public static UserData walletUpdate(UserData userData, Wallet receivedWallet) {

        //Check if the received offset is higher than the current offset and if the sent currencies list is not empty
        if (userData.getWallet().getOffset() < receivedWallet.getOffset() && !receivedWallet.getCurrenciesMap().isEmpty()) {
            userData = calculateWallet(userData, receivedWallet, WALLET_CALCULATE_CASE_UPDATE);

            //Update offset and logic from the server
            if (userData.getWallet().getOffset() != 0) {
                userData.getWallet().setOffset(receivedWallet.getOffset());
            }

            userData.getWallet().setLogic(receivedWallet.getLogic());
        }

        return userData;
    }

    /**
     * Method that performs the calculations of the values for the {@link Wallet} that come from the Gamedock backend.
     *
     * @param userData       The {@link UserData} object that contains the {@link Wallet} that needs to be updated.
     * @param receivedWallet The {@link Wallet} object that has been received from the Gamedock backend.
     * @param calculateCase  The calculation case that has to be applied. It can either be WALLET_CALCULATE_CASE_INIT or WALLET_CALCULATE_CASE_UPDATE.
     * @return Returns the {@link UserData} object initialised with the {@link Wallet} information received from the Gamedock backend.
     */
    private static UserData calculateWallet(UserData userData, Wallet receivedWallet, int calculateCase) {
        //Iterate through the received wallet
        for (Map.Entry<Integer, PlayerCurrency> receivedCurrency : receivedWallet.getCurrenciesMap().entrySet()) {
            if (receivedWallet.getLogic().equals(PlayerDataManager.Client)) {
                if (userData.getWallet().getCurrenciesMap().containsKey(receivedCurrency.getKey())) {
                    if (calculateCase == WALLET_CALCULATE_CASE_INIT) {
                        userData.getWallet().getCurrenciesMap().get(receivedCurrency.getKey()).setCurrentBalance(receivedCurrency.getValue().getCurrentBalance());
                    } else if (calculateCase == WALLET_CALCULATE_CASE_UPDATE) {
                        if (receivedCurrency.getValue().getDelta() != 0) {
                            //Calculate updated balance based on the delta received from the server
                            int updatedBalance = userData.getWallet().getCurrenciesMap().get(receivedCurrency.getKey()).getCurrentBalance() + receivedCurrency.getValue().getDelta();

                            //Reset updated balance to 0 if the value is negative
                            if (updatedBalance < 0) {
                                updatedBalance = 0;
                            }

                            //Check for currency limit and overflow
                            int currencyLimit = userData.getWallet().getCurrenciesMap().get(receivedCurrency.getKey()).getLimit();
                            if (currencyLimit > 0 && updatedBalance > currencyLimit) {
                                int newOverflow = (updatedBalance - currencyLimit) + userData.getWallet().getCurrenciesMap().get(receivedCurrency.getKey()).getOverflow();
                                userData.getWallet().getCurrenciesMap().get(receivedCurrency.getKey()).setOverflow(newOverflow);
                                updatedBalance = currencyLimit;
                            }

                            userData.getWallet().getCurrenciesMap().get(receivedCurrency.getKey()).setCurrentBalance(updatedBalance);
                            userData.getWallet().getCurrenciesMap().get(receivedCurrency.getKey()).setDelta(receivedCurrency.getValue().getDelta());
                        }
                    }

                    userData.updated = true;
                }
            }
        }

        return userData;
    }

    /**
     * Method that handles the {@link Inventory} initialisation that comes from the Gamedock backend.
     *
     * @param context           The activity context.
     * @param userData          The {@link UserData} object that contains the {@link Inventory} that needs to be updated.
     * @param receivedInventory The {@link Inventory} object that has been received from the Gamedock backend.
     * @return Returns the {@link UserData} object initialised with the {@link Inventory} information received from the Gamedock backend.
     */
    public static UserData inventoryInit(Context context, UserData userData, Inventory receivedInventory, boolean externalChange) {
        if (externalChange && (!receivedInventory.getItemsMap().isEmpty() || !receivedInventory.getUniqueItemsMap().isEmpty())) {
            userData = calculateInventory(context, userData, receivedInventory, true);
        } else if (userData.getInventory().getOffset() == 0 && receivedInventory.getOffset() != 0) {
            if (receivedInventory.getItemsMap().isEmpty()) {
                userData = UserDataManager.getInstance(context).getPlayerDataManager().setInventoryInitialValues(userData);
                if (userData == null) {
                    return null;
                } else {
                    UserDataManager.getInstance(context).getPlayerDataManager().initialValue = true;
                }
            } else if (!receivedInventory.getItemsMap().isEmpty() && !UserDataManager.getInstance(context).getPlayerDataManager().storageUtil.getBoolean(StorageUtil.Keys.InventoryInit, false)) {
                userData = calculateInventory(context, userData, receivedInventory, false);
            }
        } else if (userData.getInventory().getOffset() < receivedInventory.getOffset() && (!receivedInventory.getItemsMap().isEmpty() || !receivedInventory.getUniqueItemsMap().isEmpty())) {
            userData = calculateInventory(context, userData, receivedInventory, false);
        }

        UserDataManager.getInstance(context).getPlayerDataManager().storageUtil.putBoolean(StorageUtil.Keys.InventoryInit, true);

        //Update offset and logic from the server
        userData.getInventory().setOffset(receivedInventory.getOffset());
        userData.getInventory().setLogic(receivedInventory.getLogic());

        return userData;
    }

    /**
     * Method that handles the updating of the {@link Inventory} object with information coming from the Gamedock backend.
     *
     * @param context           The activity context.
     * @param userData          The {@link UserData} object that contains the {@link Inventory} that needs to be updated.
     * @param receivedInventory The {@link Wallet} object that has been received from the Gamedock backend.
     * @return Returns the {@link UserData} object updated with the {@link Inventory} information received from the Gamedock backend.
     */
    public static UserData inventoryUpdate(Context context, UserData userData, Inventory receivedInventory) {

        //Check if the received offset is higher than the current offset and if the sent currencies list is not empty
        if (userData.getInventory().getOffset() < receivedInventory.getOffset() && (!receivedInventory.getItemsMap().isEmpty() || !receivedInventory.getUniqueItemsMap().isEmpty())) {
            userData = calculateInventory(context, userData, receivedInventory, false);

            //Update offset and logic from the server
            if (userData.getWallet().getOffset() != 0) {
                userData.getInventory().setOffset(receivedInventory.getOffset());
            }
            userData.getInventory().setLogic(receivedInventory.getLogic());
        }

        return userData;
    }

    /**
     * Method that performs the calculations of the values for the {@link Inventory} that come from the Gamedock backend.
     *
     * @param context           The activity context.
     * @param userData          The {@link UserData} object that contains the {@link Inventory} that needs to be updated.
     * @param receivedInventory The {@link Wallet} object that has been received from the Gamedock backend.
     * @return Returns the {@link UserData} object updated with the {@link Inventory} information received from the Gamedock backend.
     */
    private static UserData calculateInventory(Context context, UserData userData, Inventory receivedInventory, boolean clear) {
        ArrayList<PlayerItem> itemsToBeAdded = new ArrayList<>();

        //Items
        for (Map.Entry<Integer, PlayerItem> receivedItem : receivedInventory.getItemsMap().entrySet()) {
            if (receivedInventory.getLogic().equals(PlayerDataManager.Client)) {
                if (userData.getInventory().getItemsMap().containsKey(receivedItem.getKey())) {
                    //Calculate updated balance based on the delta received from the server
                    int updatedAmount = userData.getInventory().getItemsMap().get(receivedItem.getKey()).getAmount();
                    if (clear) {
                        updatedAmount = receivedItem.getValue().getAmount();
                    } else if (receivedItem.getValue().getDelta() != 0) {
                        updatedAmount = updatedAmount + receivedItem.getValue().getDelta();
                    } else {
                        continue;
                    }

                    //Check for item limit and overflow
                    int itemLimit = userData.getInventory().getItemsMap().get(receivedItem.getKey()).getLimit();
                    if (itemLimit > 0 && updatedAmount > itemLimit) {
                        int newOverflow = (updatedAmount - itemLimit) + userData.getInventory().getItemsMap().get(receivedItem.getKey()).getOverflow();
                        userData.getInventory().getItemsMap().get(receivedItem.getKey()).setOverflow(newOverflow);
                        updatedAmount = itemLimit;
                    }

                    userData.getInventory().getItemsMap().get(receivedItem.getKey()).setAmount(updatedAmount);
                    userData.getInventory().getItemsMap().get(receivedItem.getKey()).setDelta(receivedItem.getValue().getDelta());

                    userData.updated = true;
                } else {
                    itemsToBeAdded.add(receivedItem.getValue());
                }
            }
        }

        for (int i = 0; i < itemsToBeAdded.size(); i++) {
            Item gameItem = GamedockGameDataManager.getInstance(context).getItem(itemsToBeAdded.get(i).getId());
            if (gameItem != null && itemsToBeAdded.get(i).getAmount() > 0) {
                PlayerItem item = new PlayerItem(gameItem);

                int updatedAmount = itemsToBeAdded.get(i).getAmount();

                //Check for item limit and overflow
                int itemLimit = item.getLimit();
                if (itemLimit > 0 && updatedAmount > itemLimit) {
                    int newOverflow = (updatedAmount - itemLimit) + item.getOverflow();
                    item.setOverflow(newOverflow);
                    updatedAmount = itemLimit;
                }

                item.setAmount(updatedAmount);
                item.setDelta(itemsToBeAdded.get(i).getDelta());
                item.setValue(itemsToBeAdded.get(i).getValue());

                userData.getInventory().getItemsMap().put(item.getId(), item);

                userData.updated = true;
            }
        }

        //Unique Items
        if (clear) {
            userData.getInventory().getUniqueItemsMap().clear();
            userData.getInventory().getUniqueItems().clear();
            userData.getInventory().setUniqueItemsMap(receivedInventory.getUniqueItemsMap());

            userData.updated = true;
        } else {
            for (Map.Entry<String, UniquePlayerItem> receivedUniqueItem : receivedInventory.getUniqueItemsMap().entrySet()) {
                if (receivedInventory.getLogic().equals(PlayerDataManager.Client)) {
                    if (userData.getInventory().getUniqueItemsMap().containsKey(receivedUniqueItem.getKey())) {
                        if (receivedUniqueItem.getValue().getStatus().equals("UPDATE")) {
                            userData.getInventory().getUniqueItemsMap().get(receivedUniqueItem.getKey()).setStatus("UPDATE");
                            userData.getInventory().getUniqueItemsMap().get(receivedUniqueItem.getKey()).setUniqueProperties(receivedUniqueItem.getValue().getUniqueProperties());

                            userData.updated = true;
                        } else if (receivedUniqueItem.getValue().getStatus().equals("REMOVE")) {
                            userData.getInventory().getUniqueItemsMap().get(receivedUniqueItem.getKey()).setStatus("REMOVE");
                            userData.getInventory().getUniqueItemsMap().get(receivedUniqueItem.getKey()).setAmount(0);
                            userData.getInventory().getUniqueItemsMap().get(receivedUniqueItem.getKey()).setDelta(-1);

                            userData.updated = true;
                        }
                    }
                }
            }
        }


        return userData;
    }
}
