package io.gamedock.sdk.userdata.playerdata.functions;

import android.content.Context;
import android.util.Log;

import com.google.gson.Gson;

import org.json.JSONException;
import org.json.JSONObject;

import java.util.ArrayList;
import java.util.Calendar;
import java.util.Random;
import java.util.UUID;

import io.gamedock.sdk.GamedockSDK;
import io.gamedock.sdk.gamedata.GamedockGameDataManager;
import io.gamedock.sdk.gamedata.promotions.PromotionsManager;
import io.gamedock.sdk.models.gamedata.GamedockGameData;
import io.gamedock.sdk.models.gamedata.bundles.Bundle;
import io.gamedock.sdk.models.gamedata.bundles.BundleItem;
import io.gamedock.sdk.models.gamedata.bundles.BundlePrice;
import io.gamedock.sdk.models.gamedata.gacha.GachaContent;
import io.gamedock.sdk.models.gamedata.items.Item;
import io.gamedock.sdk.models.gamedata.perk.PerkAddition;
import io.gamedock.sdk.models.gamedata.perk.PerkItem;
import io.gamedock.sdk.models.gamedata.perk.PerkPriceReduction;
import io.gamedock.sdk.models.gamedata.promotion.ExtraEntity;
import io.gamedock.sdk.models.gamedata.promotion.PriceOverride;
import io.gamedock.sdk.models.gamedata.promotion.Promotion;
import io.gamedock.sdk.models.tier.TieredEvent;
import io.gamedock.sdk.models.tier.TieredEventProgress;
import io.gamedock.sdk.models.tier.TieredEventTier;
import io.gamedock.sdk.models.userdata.UpdatedUserData;
import io.gamedock.sdk.models.userdata.UserData;
import io.gamedock.sdk.models.userdata.inventory.Inventory;
import io.gamedock.sdk.models.userdata.inventory.PlayerItem;
import io.gamedock.sdk.models.userdata.inventory.UniquePlayerItem;
import io.gamedock.sdk.models.userdata.wallet.PlayerCurrency;
import io.gamedock.sdk.models.userdata.wallet.Wallet;
import io.gamedock.sdk.tier.TieredEventsManager;
import io.gamedock.sdk.userdata.UserDataCallbacks;
import io.gamedock.sdk.userdata.UserDataManager;
import io.gamedock.sdk.userdata.playerdata.PlayerDataManager;
import io.gamedock.sdk.userdata.playerdata.PlayerDataUpdateReasons;
import io.gamedock.sdk.utils.error.ErrorCodes;
import io.gamedock.sdk.utils.logging.LoggingUtil;
import io.gamedock.sdk.web.WebViewActivity;

/**
 * Internal class used by the {@link PlayerDataManager} to perform all operations functions.
 */
public class PlayerDataOperations {

    public static int gachaId = 0;
    public static int gachaPosition = 0;

    /**
     * Method that resets the whole {@link UserData} and adds the initial value.
     * Calls the resetWallet and resetInventory methods.
     *
     * @param userData     The {@link UserData} object that contains all the information that needs to be resetted.
     * @param gamedockGameData The {@link GamedockGameData} object which hold all the information regarding the initial value.
     * @return Returns the resetted {@link UserData} object.
     */
    public static UserData resetUserProfile(UserData userData, GamedockGameData gamedockGameData) {
        userData = resetWallet(userData, gamedockGameData);
        userData = resetInventory(userData, gamedockGameData);

        return userData;
    }

    /**
     * Method that resets the {@link Wallet} and adds the initial values if present.
     *
     * @param userData     The {@link UserData} object that contains all the information that needs to be resetted.
     * @param gamedockGameData The {@link GamedockGameData} object which hold all the information regarding the initial value.
     * @return Returns the resetted {@link UserData} object.
     */
    public static UserData resetWallet(UserData userData, GamedockGameData gamedockGameData) {
        for (PlayerCurrency currency : userData.getWallet().getCurrenciesMap().values()) {
            try {
                int newDelta = gamedockGameData.getCurrenciesMap().get(currency.getId()).getInitialValue() - currency.getCurrentBalance();

                currency.setCurrentBalance(gamedockGameData.getCurrenciesMap().get(currency.getId()).getInitialValue());
                currency.setDelta(newDelta + currency.getDelta());

                userData.getWallet().updateCurrency(currency);
            }catch (Exception e){

            }

        }

        return userData;
    }

    /**
     * Method that resets the {@link Inventory} and adds the initial values if present.
     *
     * @param userData     The {@link UserData} object that contains all the information that needs to be resetted.
     * @param gamedockGameData The {@link GamedockGameData} object which hold all the information regarding the initial value.
     * @return Returns the resetted {@link UserData} object.
     */
    public static UserData resetInventory(UserData userData, GamedockGameData gamedockGameData) {
        for (PlayerItem item : userData.getInventory().getItemsMap().values()) {
            int newDelta = gamedockGameData.getItemsMap().get(item.getId()).getInitialValue() - item.getAmount();

            item.setAmount(gamedockGameData.getItemsMap().get(item.getId()).getInitialValue());
            item.setDelta(newDelta + item.getDelta());

            userData.getInventory().updateItem(item);
        }

        userData.getInventory().getUniqueItemsMap().clear();

        return userData;
    }

    /**
     * Method that updates the wallet currency with an amount and then sends an updatePlayerData event with the data and a reason to the server.
     *
     * @param context       The activity context.
     * @param userData      The {@link UserData} object that contains all the information regarding the player.
     * @param currency      The {@link PlayerCurrency} object which needs to be updated.
     * @param gson          The Gson object used for serializing and deserializing objects.
     * @param delta         The value with which the {@link PlayerCurrency} has to be updated. Can be both a positive or a negative value.
     * @param reason        The reason for which the update was triggered. A standard list of reasons can be found at {@link PlayerDataUpdateReasons}.
     * @param reasonDetails The details of the reason. This parameter is most of the time defined by the developer in combination with the BI team.
     * @param location      The location from which the update event has been triggered.
     * @param transactionId The transaction id for the IAP if the update was triggered by an IAP. Helps linking the event on the backend.
     */
    public static UpdatedUserData updateWallet(Context context, UserData userData, PlayerCurrency currency, Gson gson, int delta, String reason, String reasonDetails, String location, String transactionId, ArrayList<PerkItem> perkItems, boolean isTransaction, UserDataCallbacks userDataCallback) {
        int currentBalance = currency.getCurrentBalance();
        int updatedBalance = currentBalance + delta;

        //Sends an error to the developer if the currency balance is negative
        if (updatedBalance < 0) {
            userDataCallback.userDataError(ErrorCodes.NotEnoughCurrency);
            return null;
        }

        //Check for currency limit and overflow
        int currencyLimit = currency.getLimit();
        if (currencyLimit > 0 && updatedBalance > currencyLimit) {
            int newOverflow = (updatedBalance - currencyLimit) + currency.getOverflow();
            currency.setOverflow(newOverflow);
            updatedBalance = currencyLimit;
        }

        int updatedDelta = delta + currency.getDelta();

        //If delta is 0 then it means the updates cancel each other
        //We take the delta from the most recent update
        if (updatedDelta == 0) {
            updatedDelta = delta;
        }

        currency.setDelta(updatedDelta);
        currency.setCurrentBalance(updatedBalance);

        UpdatedUserData updatedUserData = new UpdatedUserData();
        updatedUserData.currencies.add(currency.clone());

        //Saves updated currency
        userData.getWallet().updateCurrency(currency);

        //Updates UserData Version and Meta
        userData = UserDataManager.getInstance(context).updateUserDataVersion(userData);
        userData = UserDataManager.getInstance(context).updateUserDataMeta(userData);

        //Saves updated user data
        UserDataManager.getInstance(context).updateUserData(userData);

        for (PlayerItem playerItem : userData.getInventory().getItemsMap().values()) {
            if (playerItem.isGacha() && playerItem.getDelta() != 0) {
                boolean bFound = false;
                for (PlayerItem existingItem : updatedUserData.items) {
                    if (playerItem.getId() == existingItem.getId()) {
                        bFound = true;
                        break;
                    }
                }
                if (!bFound) {
                    updatedUserData.items.add(playerItem.clone());
                }
            }
        }

        if (gachaId != 0) {
            updatedUserData.gachaId = gachaId;
            updatedUserData.gachaPosition = gachaPosition;
        }

        if (!isTransaction) {
            PlayerDataSending.sendUpdatePlayerDataEvent(context, gson, userData, null, reason, reasonDetails, location, transactionId, null, perkItems);

            userDataCallback.playerDataUpdated(reason, updatedUserData, GamedockSDK.getInstance(context).getWallet(), GamedockSDK.getInstance(context).getInventory());
        }

        PlayerDataOperations.updateTieredEvent(context, currency.getId(), delta, PlayerDataManager.Currency);

        return updatedUserData;
    }

    /**
     * Method that updates the inventory item with an amount and then sends an updatePlayerData event with the data and a reason to the server.
     *
     * @param context       The activity context.
     * @param userData      The {@link UserData} object that contains all the information regarding the player.
     * @param gameItem      The {@link Item} object that needs to be added to the {@link Inventory}.
     * @param gson          The Gson object used for serializing and deserializing objects.
     * @param amount        The value with which the {@link PlayerItem} has to be updated. Can only be positive.
     * @param action        The action that has to be done by the update. Can be "add" or "subtract".
     * @param reason        The reason for which the update was triggered. A standard list of reasons can be found at {@link PlayerDataUpdateReasons}.
     * @param reasonDetails The details of the reason. This parameter is most of the time defined by the developer in combination with the BI team.
     * @param location      The location from which the update event has been triggered.
     * @param transactionId The transaction id for the IAP if the update was triggered by an IAP. Helps linking the event on the backend.
     */
    public static UpdatedUserData updateInventoryWithItem(Context context, UserData userData, Item gameItem, Gson gson, int amount, String action, String reason, String reasonDetails, String location, String transactionId, ArrayList<PerkItem> perkItems, boolean isTransaction, UserDataCallbacks userDataCallback) {
        UpdatedUserData updatedUserData = new UpdatedUserData();

        PlayerItem item = new PlayerItem(gameItem);
        item.setDelta(amount);

        //Check for item limit and overflow
        int itemLimit = item.getLimit();
        if (itemLimit > 0 && amount > itemLimit) {
            int newOverflow = (amount - itemLimit) + item.getOverflow();
            item.setOverflow(newOverflow);
            amount = itemLimit;
        }

        item.setAmount(amount);

        PlayerItem inventoryItem = userData.getInventory().getItemsMap().get(item.getId());

        //Checks if the item to be added exists already in the inventory or not
        //If it exists, it either adds to the amount or removes it completely from the inventory
        //Else it creates a new entry in the inventory
        if (inventoryItem != null) {
            int inventoryItemAmount = inventoryItem.getAmount();
            int inventoryDeltaAmount = 0;

            if (action.equals(PlayerDataManager.Add)) {
                inventoryItemAmount = inventoryItemAmount + amount;
                inventoryDeltaAmount = amount;
            } else if (action.equals(PlayerDataManager.Subtract)) {
                inventoryItemAmount = inventoryItemAmount - amount;
                inventoryDeltaAmount = -amount;

                if (inventoryItemAmount < 0) {
                    userDataCallback.userDataError(ErrorCodes.ItemAmountToLow);
                    return null;
                }
            }

            //Check for item limit and overflow
            if (itemLimit > 0 && inventoryItemAmount > itemLimit) {
                int newOverflow = (inventoryItemAmount - itemLimit) + inventoryItem.getOverflow();
                inventoryItem.setOverflow(newOverflow);
                inventoryItemAmount = itemLimit;
            }

            inventoryItem.setDelta(inventoryDeltaAmount);
            inventoryItem.setAmount(inventoryItemAmount);
            userData.getInventory().updateItem(inventoryItem);

            updatedUserData.items.add(inventoryItem.clone());
        } else {
            if (action.equals(PlayerDataManager.Add)) {
                userData.getInventory().getItemsMap().put(item.getId(), item);

                updatedUserData.items.add(item.clone());
            } else if (action.equals(PlayerDataManager.Subtract)) {
                userDataCallback.userDataError(ErrorCodes.ItemAmountToLow);
            }
        }

        //Updates UserData Version and Meta
        userData = UserDataManager.getInstance(context).updateUserDataVersion(userData);
        userData = UserDataManager.getInstance(context).updateUserDataMeta(userData);

        //Saves updated user data
        UserDataManager.getInstance(context).updateUserData(userData);

        for (PlayerItem playerItem : userData.getInventory().getItemsMap().values()) {
            if (playerItem.isGacha() && playerItem.getDelta() != 0) {
                boolean bFound = false;
                for (PlayerItem existingItem : updatedUserData.items) {
                    if (existingItem.getId() == playerItem.getId()) {
                        bFound = true;
                        break;
                    }
                }
                if (!bFound) {
                    updatedUserData.items.add(playerItem.clone());
                }
            }
        }

        if (gachaId != 0) {
            updatedUserData.gachaId = gachaId;
            updatedUserData.gachaPosition = gachaPosition;
        }

        if (!isTransaction) {
            PlayerDataSending.sendUpdatePlayerDataEvent(context, gson, userData, null, reason, reasonDetails, location, transactionId, null, perkItems);
            userDataCallback.playerDataUpdated(reason, updatedUserData, GamedockSDK.getInstance(context).getWallet(), GamedockSDK.getInstance(context).getInventory());
        }

        PlayerDataOperations.updateTieredEvent(context, gameItem.getId(), amount, gameItem.isGacha() ? PlayerDataManager.GachaCheck : PlayerDataManager.Item);

        return updatedUserData;
    }

    /**
     * Method that adds a UniquePlayerItem to the user's inventory.
     *
     * @param context          The activity context.
     * @param userData         The {@link UserData} object that contains all the information regarding the player.
     * @param gson             The Gson object used for serializing and deserializing objects.
     * @param uniquePlayerItem The UniquePlayerItem that needs to be added.
     * @param reason           The reason for which the update was triggered. A standard list of reasons can be found at {@link PlayerDataUpdateReasons}.
     * @param reasonDetails    The details of the reason. This parameter is most of the time defined by the developer in combination with the BI team.
     * @param location         The location from which the update event has been triggered.
     * @param transactionId    The transaction id for the IAP if the update was triggered by an IAP. Helps linking the event on the backend.
     */
    public static UpdatedUserData addUniqueItemToInventory(Context context, UserData userData, Gson gson, UniquePlayerItem uniquePlayerItem, String reason, String reasonDetails, String location, String transactionId, boolean isTransaction, UserDataCallbacks userDataCallback) {
        UpdatedUserData updatedUserData = new UpdatedUserData();

        if (userData.getInventory().getUniqueItemsMap().containsKey(uniquePlayerItem.getUniqueId())) {
            userDataCallback.userDataError(ErrorCodes.UniqueItemAdd);
            return null;
        }

        uniquePlayerItem.setAmount(1);
        uniquePlayerItem.setDelta(1);
        uniquePlayerItem.setStatus("CREATE");

        userData.getInventory().getUniqueItemsMap().put(uniquePlayerItem.getUniqueId(), uniquePlayerItem);

        updatedUserData.uniqueItems.add(uniquePlayerItem.clone());

        //Updates UserData Version and Meta
        userData = UserDataManager.getInstance(context).updateUserDataVersion(userData);
        userData = UserDataManager.getInstance(context).updateUserDataMeta(userData);

        //Saves updated user data
        UserDataManager.getInstance(context).updateUserData(userData);

        if (!isTransaction) {
            PlayerDataSending.sendUpdatePlayerDataEvent(context, gson, userData, null, reason, reasonDetails, location, transactionId, null, null);
            userDataCallback.playerDataUpdated(reason, updatedUserData, GamedockSDK.getInstance(context).getWallet(), GamedockSDK.getInstance(context).getInventory());
        }

        return updatedUserData;
    }

    /**
     * Method that updates a UniquePlayerItem to the user's inventory.
     *
     * @param context          The activity context.
     * @param userData         The {@link UserData} object that contains all the information regarding the player.
     * @param gson             The Gson object used for serializing and deserializing objects.
     * @param uniquePlayerItem The UniquePlayerItem that needs to be updated.
     * @param reason           The reason for which the update was triggered. A standard list of reasons can be found at {@link PlayerDataUpdateReasons}.
     * @param reasonDetails    The details of the reason. This parameter is most of the time defined by the developer in combination with the BI team.
     * @param location         The location from which the update event has been triggered.
     * @param transactionId    The transaction id for the IAP if the update was triggered by an IAP. Helps linking the event on the backend.
     */
    public static UpdatedUserData updateUniqueItemFromInventory(Context context, UserData userData, Gson gson, UniquePlayerItem uniquePlayerItem, String reason, String reasonDetails, String location, String transactionId, boolean isTransaction, UserDataCallbacks userDataCallback) {
        if (!userData.getInventory().getUniqueItemsMap().containsKey(uniquePlayerItem.getUniqueId())) {
            userDataCallback.userDataError(ErrorCodes.ItemOperation);
            return null;
        }

        UpdatedUserData updatedUserData = new UpdatedUserData();

        uniquePlayerItem.setAmount(1);
        uniquePlayerItem.setDelta(0);
        uniquePlayerItem.setStatus("UPDATE");

        userData.getInventory().getUniqueItemsMap().put(uniquePlayerItem.getUniqueId(), uniquePlayerItem);

        updatedUserData.uniqueItems.add(uniquePlayerItem.clone());

        //Updates UserData Version and Meta
        userData = UserDataManager.getInstance(context).updateUserDataVersion(userData);
        userData = UserDataManager.getInstance(context).updateUserDataMeta(userData);

        //Saves updated user data
        UserDataManager.getInstance(context).updateUserData(userData);

        if (!isTransaction) {
            PlayerDataSending.sendUpdatePlayerDataEvent(context, gson, userData, null, reason, reasonDetails, location, transactionId, null, null);
            userDataCallback.playerDataUpdated(reason, updatedUserData, GamedockSDK.getInstance(context).getWallet(), GamedockSDK.getInstance(context).getInventory());
        }

        return updatedUserData;
    }

    /**
     * Method that removes a UniquePlayerItem to the user's inventory.
     *
     * @param context          The activity context.
     * @param userData         The {@link UserData} object that contains all the information regarding the player.
     * @param gson             The Gson object used for serializing and deserializing objects.
     * @param uniquePlayerItem The UniquePlayerItem that needs to be removed.
     * @param reason           The reason for which the update was triggered. A standard list of reasons can be found at {@link PlayerDataUpdateReasons}.
     * @param reasonDetails    The details of the reason. This parameter is most of the time defined by the developer in combination with the BI team.
     * @param location         The location from which the update event has been triggered.
     * @param transactionId    The transaction id for the IAP if the update was triggered by an IAP. Helps linking the event on the backend.
     */
    public static UpdatedUserData removeUniqueItemFromInventory(Context context, UserData userData, Gson gson, UniquePlayerItem uniquePlayerItem, String reason, String reasonDetails, String location, String transactionId, boolean isTransaction, UserDataCallbacks userDataCallback) {
        if (!userData.getInventory().getUniqueItemsMap().containsKey(uniquePlayerItem.getUniqueId())) {
            userDataCallback.userDataError(ErrorCodes.ItemOperation);
            return null;
        }

        UpdatedUserData updatedUserData = new UpdatedUserData();

        uniquePlayerItem.setAmount(0);
        uniquePlayerItem.setDelta(-1);
        uniquePlayerItem.setStatus("REMOVE");

        userData.getInventory().getUniqueItemsMap().put(uniquePlayerItem.getUniqueId(), uniquePlayerItem);

        updatedUserData.uniqueItems.add(uniquePlayerItem.clone());

        //Updates UserData Version and Meta
        userData = UserDataManager.getInstance(context).updateUserDataVersion(userData);
        userData = UserDataManager.getInstance(context).updateUserDataMeta(userData);

        //Saves updated user data
        UserDataManager.getInstance(context).updateUserData(userData);

        if (!isTransaction) {
            PlayerDataSending.sendUpdatePlayerDataEvent(context, gson, userData, null, reason, reasonDetails, location, transactionId, null, null);
            userDataCallback.playerDataUpdated(reason, updatedUserData, GamedockSDK.getInstance(context).getWallet(), GamedockSDK.getInstance(context).getInventory());
        }

        return updatedUserData;
    }

    /**
     * Method that updates the inventory and wallet by consuming the bundle and then sends an updatePlayerData event with the data and a reason to the server.
     *
     * @param context       The activity context.
     * @param userData      The {@link UserData} object that contains all the information regarding the player.
     * @param bundle        The {@link Bundle} object that needs to be consumed.
     * @param gson          The Gson object used for serializing and deserializing objects.
     * @param reason        The reason for which the update was triggered. A standard list of reasons can be found at {@link PlayerDataUpdateReasons}.
     * @param reasonDetails The details of the reason. This parameter is most of the time defined by the developer in combination with the BI team.
     * @param location      The location from which the update event has been triggered.
     * @param transactionId The transaction id for the IAP if the update was triggered by an IAP. Helps linking the event on the backend.
     */
    public static void buyBundle(Context context, UserData userData, Bundle bundle, Gson gson, String reason, String reasonDetails, String location, String transactionId, ArrayList<PerkItem> perkItems, UserDataCallbacks userDataCallback) {
        UpdatedUserData updatedUserData = new UpdatedUserData();
        ArrayList<UniquePlayerItem> uniquePlayerItems = new ArrayList<>();

        //Check if bundle has a promotion and is valid
        Promotion promotion = PromotionsManager.getInstance(context).getBundlePromotionObject(bundle.getId());
        boolean isPromotionValid = false;

        if (promotion != null) {
            isPromotionValid = promotion.isValid();
        }

        //Update Player Currency
        //Similar logic as the Case 1 method
        ArrayList<BundlePrice> bundlePrices = new ArrayList<>();

        if (isPromotionValid) {
            for (PriceOverride priceOverride : promotion.getPriceOverride()) {
                BundlePrice bundlePrice = new BundlePrice();
                bundlePrice.setCurrencyId(priceOverride.getId());
                bundlePrice.setValue(priceOverride.getAmount());

                bundlePrices.add(bundlePrice);
            }
        }

        if (bundlePrices.isEmpty()) {
            for (BundlePrice price : bundle.getPrices()) {
                BundlePrice bundlePrice = new BundlePrice();
                bundlePrice.setCurrencyId(price.getCurrencyId());
                bundlePrice.setValue(price.getValue());

                bundlePrices.add(bundlePrice);
            }
        }

        if (perkItems != null) {
            for (int i = 0; i < bundlePrices.size(); i++) {
                int priceAfterPerk = bundlePrices.get(i).getValue();
                for (PerkItem perkItem : perkItems) {
                    for (PerkPriceReduction priceReduction : perkItem.priceReductions) {
                        if (priceReduction.currencyId == bundlePrices.get(i).getCurrencyId()) {
                            priceAfterPerk = priceAfterPerk - priceReduction.discountValue;

                            if (priceAfterPerk < 0) {
                                priceAfterPerk = 0;
                            }
                        }
                    }
                }
                bundlePrices.get(i).setValue(priceAfterPerk);
            }
        }

        for (BundlePrice bundlePrice : bundlePrices) {
            PlayerCurrency currency = UserDataManager.getInstance(context).getPlayerDataManager().getPlayerCurrency(bundlePrice.getCurrencyId());

            if (currency == null) {
                userDataCallback.userDataError(ErrorCodes.CurrencyNotFound);
                return;
            }

            int currentBalance = currency.getCurrentBalance();
            int updatedBalance = currentBalance - bundlePrice.getValue();

            if (updatedBalance < 0) {
                userDataCallback.userDataError(ErrorCodes.NotEnoughCurrency);
                return;
            }

            int updatedDelta = -bundlePrice.getValue() + currency.getDelta();

            if (updatedDelta == 0) {
                updatedDelta = -bundlePrice.getValue();
            }

            currency.setDelta(updatedDelta);
            currency.setCurrentBalance(updatedBalance);

            userData.getWallet().updateCurrency(currency);
            updatedUserData.currencies.add(currency.clone());
        }

        //Update Player Inventory
        //Similar logic as the Case 2 method
        for (BundleItem bundleItem : bundle.getItems()) {
            if (bundleItem.getType().equals(PlayerDataManager.Currency)) {
                updateCurrencyReward(context, userData, bundleItem.getId(), bundleItem.getAmount(), perkItems, updatedUserData);
            } else if (bundleItem.getType().equals(PlayerDataManager.Item) || bundleItem.getType().equals(PlayerDataManager.GachaCheck)) {
                updateItemReward(context, userData, bundleItem.getId(), bundleItem.getAmount(), uniquePlayerItems, perkItems, updatedUserData);
            }
        }

        if (isPromotionValid) {
            for (ExtraEntity extraEntity : promotion.getExtraEntities()) {
                if (extraEntity.getType().equals(PlayerDataManager.Currency)) {
                    updateCurrencyReward(context, userData, extraEntity.getId(), extraEntity.getAmount(), null, updatedUserData);
                } else if (extraEntity.getType().equals(PlayerDataManager.Item) || extraEntity.getType().equals(PlayerDataManager.GachaCheck)) {
                    updateItemReward(context, userData, extraEntity.getId(), extraEntity.getAmount(), uniquePlayerItems, null, updatedUserData);
                }
            }
        }

        if (perkItems != null) {
            ArrayList<PerkItem> clonedPerkItems = new ArrayList<PerkItem>();
            for (PerkItem perkItem : perkItems) {
                clonedPerkItems.add(perkItem.clone());
            }
            updatedUserData.perkItems = clonedPerkItems;
        }

        //Updates UserData Version and Meta
        userData = UserDataManager.getInstance(context).updateUserDataVersion(userData);
        userData = UserDataManager.getInstance(context).updateUserDataMeta(userData);

        //Saves updated user data
        UserDataManager.getInstance(context).updateUserData(userData);

        for (PlayerItem playerItem : userData.getInventory().getItemsMap().values()) {
            if (playerItem.isGacha() && playerItem.getDelta() != 0) {
                boolean bFound = false;
                for (PlayerItem existingItem : updatedUserData.items) {
                    if (playerItem.getId() == existingItem.getId()) {
                        bFound = true;
                        break;
                    }
                }
                if (!bFound) {
                    updatedUserData.items.add(playerItem.clone());
                }
            }
        }

        updatedUserData.bundleId = bundle.getId();

        final UserData finalUserData = userData;

        if (isPromotionValid) {
            PromotionsManager.getInstance(context).getBundlePromotionObject(bundle.getId()).setAmountPurchased(promotion.getAmountPurchased() + 1);
            PromotionsManager.getInstance(context).sendBoughtPromotion(bundle.getId());
        }

        sendWebViewBundleResponse(true, bundle.getId());

        PlayerDataSending.sendUpdatePlayerDataEvent(context, gson, finalUserData, bundle, reason, reasonDetails, location, transactionId, promotion, perkItems);

        for (UniquePlayerItem uniquePlayerItem : uniquePlayerItems) {
            GamedockSDK.getInstance(context).getUserDataCallbacks().playerDataNewUniqueItem(uniquePlayerItem, bundle.getId(), 0, 0, 0, reason);
        }

        for (BundlePrice bundlePrice : bundlePrices) {
            updateTieredEvent(context, bundlePrice.getCurrencyId(), -bundlePrice.getValue(), PlayerDataManager.Currency);
        }

        for (BundleItem bundleItem : bundle.getItems()) {
            updateTieredEvent(context, bundleItem.getId(), bundleItem.getAmount(), bundleItem.getType());
        }

        userDataCallback.playerDataUpdated(reason, updatedUserData, GamedockSDK.getInstance(context).getWallet(), GamedockSDK.getInstance(context).getInventory());
    }

    /**
     * Method used to consume the contents of a bundle without deducting the price for it.
     *
     * @param context       The activity context.
     * @param userData      The {@link UserData} object that contains all the information regarding the player.
     * @param bundle        The {@link Bundle} object that needs to be consumed.
     * @param gson          The Gson object used for serializing and deserializing objects.
     * @param reason        The reason for which the update was triggered. A standard list of reasons can be found at {@link PlayerDataUpdateReasons}.
     * @param reasonDetails The details of the reason. This parameter is most of the time defined by the developer in combination with the BI team.
     * @param location      The location from which the update event has been triggered.
     */
    public static void openBundle(final Context context, UserData userData, final Bundle bundle, final Gson gson, final String reason, final String reasonDetails, final String location, final ArrayList<PerkItem> perkItems, UserDataCallbacks userDataCallback) {
        UpdatedUserData updatedUserData = new UpdatedUserData();
        ArrayList<UniquePlayerItem> uniquePlayerItems = new ArrayList<>();

        for (BundleItem bundleItem : bundle.getItems()) {
            if (bundleItem.getType().equals(PlayerDataManager.Currency)) {
                updateCurrencyReward(context, userData, bundleItem.getId(), bundleItem.getAmount(), perkItems, updatedUserData);
            } else if (bundleItem.getType().equals(PlayerDataManager.Item) || bundleItem.getType().equals(PlayerDataManager.GachaCheck)) {
                updateItemReward(context, userData, bundleItem.getId(), bundleItem.getAmount(), uniquePlayerItems, perkItems, updatedUserData);
            }
        }

        //Updates UserData Version and Meta
        userData = UserDataManager.getInstance(context).updateUserDataVersion(userData);
        userData = UserDataManager.getInstance(context).updateUserDataMeta(userData);

        //Saves updated user data
        UserDataManager.getInstance(context).updateUserData(userData);

        for (PlayerItem playerItem : userData.getInventory().getItemsMap().values()) {
            if (playerItem.isGacha() && playerItem.getDelta() != 0) {
                boolean bFound = false;
                for (PlayerItem existingItem : updatedUserData.items) {
                    if (existingItem.getId() == playerItem.getId()) {
                        bFound = true;
                        break;
                    }
                }
                if (!bFound) {
                    updatedUserData.items.add(playerItem.clone());
                }
            }
        }

        updatedUserData.bundleId = bundle.getId();

        if (gachaId != 0) {
            updatedUserData.gachaId = gachaId;
            updatedUserData.gachaPosition = gachaPosition;
        }

        PlayerDataSending.sendUpdatePlayerDataEvent(context, gson, userData, bundle, reason, reasonDetails, location, null, null, perkItems);

        for (UniquePlayerItem uniquePlayerItem : uniquePlayerItems) {
            GamedockSDK.getInstance(context).getUserDataCallbacks().playerDataNewUniqueItem(uniquePlayerItem, bundle.getId(), 0, 0, 0, reason);
        }

        userDataCallback.playerDataUpdated(reason, updatedUserData, GamedockSDK.getInstance(context).getWallet(), GamedockSDK.getInstance(context).getInventory());
    }

    /**
     * Method that opens the gacha box and adds the contents to the user's {@link Wallet} and {@link Inventory}.
     *
     * @param context       The activity context.
     * @param userData      The {@link UserData} object that contains all the information regarding the player.
     * @param gacha         The gacha {@link PlayerItem} that needs to be opened.
     * @param reason        The reason for which the update was triggered. A standard list of reasons can be found at {@link PlayerDataUpdateReasons}.
     * @param reasonDetails The details of the reason. This parameter is most of the time defined by the developer in combination with the BI team.
     * @param location      The location from which the update event has been triggered.
     */
    public static UpdatedUserData openGacha(final Context context, UserData userData, final PlayerItem gacha, String reason, String reasonDetails, String location, ArrayList<PerkItem> perkItems, boolean isTransaction, UserDataCallbacks userDataCallback) {
        UpdatedUserData finalUpdatedUserData = new UpdatedUserData();

        if (gacha.getAmount() < 1) {
            userDataCallback.userDataError(ErrorCodes.NotEnoughGachaBoxes);
            return null;
        }

        if (gacha.getContent().isEmpty()) {
            userDataCallback.userDataError(ErrorCodes.GachaOperation);
            return null;
        }

        ArrayList<GachaContent> gachaContents = gacha.getContent();

        if (perkItems != null) {
            for (PerkItem perkItem : perkItems) {
                for (int i = 0; i < gachaContents.size(); i++) {
                    for (int j = 0; j < perkItem.gachaWeights.size(); j++) {
                        if ((perkItem.gachaWeights.get(j).id == gachaContents.get(i).getId()) && (perkItem.gachaWeights.get(j).type.equals(gachaContents.get(i).getType()))) {
                            gachaContents.get(i).setWeight(perkItem.gachaWeights.get(j).weight);
                        }
                    }
                }
            }
        }

        ArrayList<GachaContent> gachaContentPossibilities = new ArrayList<>();
        if (!gacha.doesAllowDuplicates()) {
            if (gacha.shouldReroll()) {
                for (int i = 0; i < gachaContents.size(); i++) {
                    boolean duplicate = isDuplicateGachaContentResult(context, userData, gachaContents.get(i));
                    if (!duplicate) {
                        gachaContentPossibilities.add(gachaContents.get(i));
                    }
                }
            } else {
                gachaContentPossibilities = gachaContents;
            }
        } else {
            gachaContentPossibilities = gachaContents;
        }

        GachaContent gachaReward = null;
        if (gachaContentPossibilities.size() == 0) {
            gachaReward = gacha.getDuplicateReward();
        } else {
            int weightSum = 0;
            for (int i = 0; i < gachaContentPossibilities.size(); i++) {
                weightSum = weightSum + gachaContentPossibilities.get(i).getWeight();
            }

            Random r = new Random();
            int rand = r.nextInt(weightSum);

            int high = 0;

            for (int i = 0; i < gachaContentPossibilities.size(); i++) {
                high += gachaContentPossibilities.get(i).getWeight();

                if (rand < high) {
                    gachaReward = gachaContentPossibilities.get(i);
                    break;
                }
            }

            if (!gacha.doesAllowDuplicates() && !gacha.shouldReroll()) {
                boolean duplicate = isDuplicateGachaContentResult(context, userData, gachaReward);
                if (duplicate) {
                    gachaReward = gacha.getDuplicateReward();
                }
            }
        }

        if (gachaReward == null) {
            userDataCallback.userDataError(ErrorCodes.GachaOperation);
            return null;
        }

        gacha.setAmount(gacha.getAmount() - 1);
        gacha.setDelta(gacha.getDelta() - 1);
        userData.getInventory().getItemsMap().put(gacha.getId(), gacha);

        //Saves updated user data
        UserDataManager.getInstance(context).updateUserData(userData);
        gachaId = gacha.getId();
        gachaPosition = gachaReward.getPosition();

        finalUpdatedUserData.items.add(gacha);

        switch (gachaReward.getType()) {
            case PlayerDataManager.Currency:
                int amountCurrency = gachaReward.getAmount();

                if (perkItems != null) {
                    for (PerkItem perkItem : perkItems) {
                        for (int j = 0; j < perkItem.additions.size(); j++) {
                            if (perkItem.additions.get(j).id == gachaReward.getId() && perkItem.additions.get(j).type.equals(PlayerDataManager.Currency)) {
                                amountCurrency = amountCurrency + perkItem.additions.get(j).additionValue;
                            }
                        }
                    }
                }

                UpdatedUserData tempUpdatedDataCurrency = UserDataManager.getInstance(context).getPlayerDataManager().updateWallet(gachaReward.getId(), amountCurrency, reason, reasonDetails, location, null, perkItems, isTransaction, userDataCallback);
                finalUpdatedUserData.currencies.addAll(tempUpdatedDataCurrency.currencies);
                break;
            case PlayerDataManager.Item:
                Item gameItem = GamedockGameDataManager.getInstance(context).getItem(gachaReward.getId());

                if (gameItem == null) {
                    userDataCallback.userDataError(ErrorCodes.ItemNotFound);
                    return null;
                }

                if (gameItem.isUnique()) {
                    UniquePlayerItem uniquePlayerItem = new UniquePlayerItem(gameItem);
                    uniquePlayerItem.setUniqueId(UUID.randomUUID().toString());

                    if (!isTransaction) {
                        PlayerDataSending.sendUpdatePlayerDataEvent(context, GamedockSDK.getInstance(context).getGson(), userData, null, reason, reasonDetails, location, null, null, perkItems);
                    }

                    userDataCallback.playerDataNewUniqueItem(uniquePlayerItem, 0, gacha.getId(), gachaReward.getPosition(),0, reason);
                } else {
                    int amountItem = gachaReward.getAmount();

                    if (perkItems != null) {
                        for (PerkItem perkItem : perkItems) {
                            for (int j = 0; j < perkItem.additions.size(); j++) {
                                if (perkItem.additions.get(j).id == gachaReward.getId() && perkItem.additions.get(j).type.equals(PlayerDataManager.Item)) {
                                    amountItem = amountItem + perkItem.additions.get(j).additionValue;
                                }
                            }
                        }
                    }

                    UpdatedUserData tempUpdatedDataItem = UserDataManager.getInstance(context).getPlayerDataManager().updateInventoryWithItem(gachaReward.getId(), amountItem, PlayerDataManager.Add, reason, reasonDetails, location, null, perkItems, isTransaction, userDataCallback);
                    finalUpdatedUserData.items.addAll(tempUpdatedDataItem.items);
                }
                break;
            case PlayerDataManager.BundleCheck:
                UserDataManager.getInstance(context).getPlayerDataManager().openBundle(gachaReward.getId(), gachaReward.getAmount(), reason, reasonDetails, location, perkItems, userDataCallback);
                break;
            case PlayerDataManager.GachaCheck:
                int amountGacha = gachaReward.getAmount();

                if (perkItems != null) {
                    for (PerkItem perkItem : perkItems) {
                        for (int j = 0; j < perkItem.additions.size(); j++) {
                            if (perkItem.additions.get(j).id == gachaReward.getId() && perkItem.additions.get(j).type.equals(PlayerDataManager.Item)) {
                                amountGacha = amountGacha + perkItem.additions.get(j).additionValue;
                            }
                        }
                    }
                }

                UpdatedUserData tempUpdatedDataGacha = UserDataManager.getInstance(context).getPlayerDataManager().updateInventoryWithItem(gachaReward.getId(), amountGacha, PlayerDataManager.Add, reason, reasonDetails, location, null, perkItems, isTransaction, userDataCallback);
                finalUpdatedUserData.items.addAll(tempUpdatedDataGacha.items);
                break;
            case PlayerDataManager.None:
                //Updates UserData Version and Meta
                userData = UserDataManager.getInstance(context).updateUserDataVersion(userData);
                userData = UserDataManager.getInstance(context).updateUserDataMeta(userData);

                UserDataManager.getInstance(context).updateUserData(userData);

                userDataCallback.playerDataEmptyGacha();
                break;
            default:
                userDataCallback.userDataError(ErrorCodes.GachaOperation);
                return null;
        }

        if (WebViewActivity.getActivity() != null) {
            try {
                JSONObject webViewData = new JSONObject();
                webViewData.put("success", true);

                JSONObject webViewDataInfo = new JSONObject();
                JSONObject gachaItemJSON = new JSONObject(new Gson().toJson(gachaReward));
                webViewDataInfo.put("gachaItem", gachaItemJSON);

                webViewData.put("data", webViewDataInfo);

                WebViewActivity.getActivity().javascriptBridge.nativeMessage("openGacha", webViewData.toString());
            } catch (JSONException e) {
                e.printStackTrace();
            }
        }

        finalUpdatedUserData.gachaId = gacha.getId();
        finalUpdatedUserData.gachaPosition = gachaReward.getPosition();

        gachaId = 0;
        gachaPosition = 0;

        return finalUpdatedUserData;
    }

    /**
     * Method used to check if the gacha reward that has been rolled is already owned by the user.
     *
     * @param context     The Activity context.
     * @param userData    The {@link UserData} object that contains all the information regarding the player.
     * @param gachaReward The gacha reward that needs to be checked.
     * @return
     */
    private static boolean isDuplicateGachaContentResult(Context context, UserData userData, GachaContent gachaReward) {
        if (gachaReward == null) {
            return false;
        }

        if (gachaReward.getType().equals("ITEM") || gachaReward.getType().equals("GACHA")) {
            Item item = GamedockGameDataManager.getInstance(context).getItem(gachaReward.getId());
            if (item != null && item.isUnique()) {
                // Check if there was already another unique item in the inventory with the same item id
                return hasUniqueItemInInventory(userData, item.getId());
            } else {
                PlayerItem playerItem = userData.getInventory().getItemsMap().get(gachaReward.getId());
                return playerItem != null && playerItem.getAmount() > 0;
            }
        }

        return false;
    }

    /**
     * Method used to check if the user has an unique item with a specific item id.
     *
     * @param userData The {@link UserData} object that contains all the information regarding the player.
     * @param itemId   The id associated with the item.
     * @return
     */
    private static boolean hasUniqueItemInInventory(UserData userData, int itemId) {
        for (UniquePlayerItem uniqueItem : userData.getInventory().getUniqueItemsMap().values()) {
            if (uniqueItem.getId() == itemId) {
                return uniqueItem.getAmount() > 0;
            }
        }

        return false;
    }

    /**
     * Method used to notify a splash screen that a bundle with a specific id has been bought.
     *
     * @param success  Value that marks if the operation was successful.
     * @param bundleId The id of the bundle that was bought.
     */
    private static void sendWebViewBundleResponse(boolean success, int bundleId) {
        if (WebViewActivity.getActivity() == null) {
            LoggingUtil.d("No Splash Screen active. No message will be sent.");
            return;
        }

        try {
            JSONObject webViewData = new JSONObject();
            webViewData.put("success", success);

            JSONObject webViewDataInfo = new JSONObject();
            webViewDataInfo.put("bundleId", bundleId);
            webViewData.put("data", webViewDataInfo);

            WebViewActivity.getActivity().javascriptBridge.nativeMessage("buyBundle", webViewData.toString());
        } catch (JSONException e) {
            e.printStackTrace();
        }
    }

    /**
     * Method used to update any tiered event that might have the specified entity types and ids as requirements.
     *
     * @param context    The Activity context.
     * @param entityId   The id of the entity that needs to be updated.
     * @param amount     The value with which it needs to be updated.
     * @param entityType The type of entity. (ex.: currency, item, etc.)
     */
    private static void updateTieredEvent(Context context, int entityId, int amount, String entityType) {
        TieredEvent selectedTieredEvent = null;

        if (TieredEventsManager.getInstance(context).getTieredEventsOverview().getTieredEvents().isEmpty()) {
            return;
        }

        for (TieredEvent tieredEvent : TieredEventsManager.getInstance(context).getTieredEventsOverview().getTieredEvents().values()) {
            if (!(tieredEvent.getEndDate() > Calendar.getInstance().getTimeInMillis())) {
                continue;
            }

            for (TieredEventTier tier : tieredEvent.getTiers()) {
                if (tier.getEntityId() == entityId && tier.getEntityType().equals(entityType)) {
                    selectedTieredEvent = tieredEvent;
                    break;
                }
            }
        }

        if (selectedTieredEvent == null) {
            return;
        }

        TieredEventProgress tieredProgress = TieredEventsManager.getInstance(context).getTieredEventsOverview().getProgress().get(selectedTieredEvent.getId());

        if (tieredProgress == null || tieredProgress.isCompleted()) {
            return;
        }

        TieredEventTier currentTier = null;

        for (TieredEventTier tier : selectedTieredEvent.getTiers()) {
            if (tier.getId() == tieredProgress.getCurrentTierId()) {
                currentTier = tier;
                break;
            }
        }

        if (currentTier == null) {
            return;
        }

        if ((selectedTieredEvent.getType().equals("spend") && amount < 0)) {
            TieredEventsManager.getInstance(context).updateTierProgress(selectedTieredEvent.getId(), currentTier.getId(), entityId, entityType, (amount * -1));
        } else if ((selectedTieredEvent.getType().equals("collect") && amount > 0)) {
            TieredEventsManager.getInstance(context).updateTierProgress(selectedTieredEvent.getId(), currentTier.getId(), entityId, entityType, amount);
        } else {
            LoggingUtil.d("Entity operation not meeting Tiered Event requirements. Tiered Event progress will not be updated.");
        }
    }

    /**
     * Method used to update the currency value. Mostly used in mission reward claiming.
     *
     * @param context         The Activity context.
     * @param userData        The {@link UserData} object that contains all the information regarding the player.
     * @param currencyId      The id of the currency that needs to be updated.
     * @param currencyAmount  The amount that needs to be updated.
     * @param perkItems       The perk items, if any that need to be applied.
     * @param updatedUserData The updated response that will be sent to the game.
     */
    public static void updateCurrencyReward(Context context, UserData userData, int currencyId, int currencyAmount, ArrayList<PerkItem> perkItems, UpdatedUserData updatedUserData) {
        PlayerCurrency currency = UserDataManager.getInstance(context).getPlayerDataManager().getPlayerCurrency(currencyId);
        int perkAdditionAmount = 0;

        if (currency == null) {
            GamedockSDK.getInstance(context).getUserDataCallbacks().userDataError(ErrorCodes.CurrencyNotFound);
            return;
        }

        if (perkItems != null) {
            for (PerkItem perkItem : perkItems) {
                for (PerkAddition perkAddition : perkItem.additions) {
                    if (perkAddition.type.equals("CURRENCY") && perkAddition.id == currency.getId()) {
                        perkAdditionAmount = perkAdditionAmount + perkAddition.additionValue;
                        break;
                    }
                }
            }
        }

        int updatedBalance = currency.getCurrentBalance() + currencyAmount + perkAdditionAmount;

        //Check for currency limit and overflow
        int currencyLimit = currency.getLimit();
        if (currencyLimit > 0 && updatedBalance > currencyLimit) {
            int newOverflow = (updatedBalance - currencyLimit) + currency.getOverflow();
            currency.setOverflow(newOverflow);
            updatedBalance = currencyLimit;
        }

        currency.setCurrentBalance(updatedBalance);
        currency.setDelta(currency.getDelta() + currencyAmount + perkAdditionAmount);

        userData.getWallet().updateCurrency(currency);

        PlayerCurrency temp = null;

        for (PlayerCurrency playerCurrency : updatedUserData.currencies) {
            if (playerCurrency.getId() == currencyId) {
                temp = playerCurrency;
                break;
            }
        }

        if (temp != null) {
            updatedUserData.currencies.remove(temp);
        }

        updatedUserData.currencies.add(currency.clone());
    }

    /**
     * Method used to update the item value. Mostly used in mission reward claiming.
     *
     * @param context           The Activity context.
     * @param userData          The {@link UserData} object that contains all the information regarding the player.
     * @param itemId            The id of the item that needs to be updated.
     * @param itemAmount        The amount that needs to be updated.
     * @param uniquePlayerItems The list of unique items that might need to be updated.
     * @param perkItems         The perk items, if any that need to be applied.
     * @param updatedUserData   The updated response that will be sent to the game.
     */
    public static void updateItemReward(Context context, UserData userData, int itemId, int itemAmount, ArrayList<UniquePlayerItem> uniquePlayerItems, ArrayList<PerkItem> perkItems, UpdatedUserData updatedUserData) {
        Item gameItem = GamedockGameDataManager.getInstance(context).getItem(itemId);

        if (gameItem == null) {
            GamedockSDK.getInstance(context).getUserDataCallbacks().userDataError(ErrorCodes.ItemNotFound);
            return;
        }

        if (gameItem.isUnique()) {
            UniquePlayerItem uniquePlayerItem = new UniquePlayerItem(gameItem);
            uniquePlayerItem.setUniqueId(UUID.randomUUID().toString());
            uniquePlayerItems.add(uniquePlayerItem);
        } else {
            PlayerItem item = new PlayerItem(gameItem);

            PlayerItem inventoryItem = userData.getInventory().getItemsMap().get(item.getId());

            int inventoryItemAmount;
            int itemLimit = item.getLimit();
            int perkAdditionAmount = 0;

            if (perkItems != null) {
                for (PerkItem perkItem : perkItems) {
                    for (PerkAddition perkAddition : perkItem.additions) {
                        if (perkAddition.type.equals("ITEM") && perkAddition.id == gameItem.getId()) {
                            perkAdditionAmount = perkAdditionAmount + perkAddition.additionValue;
                            break;
                        }
                    }
                }
            }

            if (inventoryItem != null) {
                inventoryItemAmount = inventoryItem.getAmount() + itemAmount + perkAdditionAmount;

                if (itemLimit > 0 && inventoryItemAmount > itemLimit) {
                    int newOverflow = (inventoryItemAmount - itemLimit) + inventoryItem.getOverflow();
                    inventoryItem.setOverflow(newOverflow);
                    inventoryItemAmount = itemLimit;
                }

                inventoryItem.setDelta(inventoryItem.getDelta() + itemAmount + perkAdditionAmount);
                inventoryItem.setAmount(inventoryItemAmount);
                userData.getInventory().updateItem(inventoryItem);

                PlayerItem temp = null;
                for (PlayerItem playerItem : updatedUserData.items) {
                    if (playerItem.getId() == inventoryItem.getId()) {
                        temp = playerItem;
                        break;
                    }
                }

                if (temp != null) {
                    updatedUserData.items.remove(temp);
                }

                updatedUserData.items.add(inventoryItem.clone());
            } else {
                inventoryItemAmount = itemAmount + perkAdditionAmount;

                if (itemLimit > 0 && inventoryItemAmount > itemLimit) {
                    int newOverflow = (inventoryItemAmount - itemLimit) + item.getOverflow();
                    item.setOverflow(newOverflow);
                    inventoryItemAmount = itemLimit;
                }

                item.setDelta(inventoryItemAmount);
                item.setAmount(inventoryItemAmount);
                userData.getInventory().getItemsMap().put(item.getId(), item);

                if (updatedUserData != null) {
                    updatedUserData.items.add(item.clone());
                }

            }
        }
    }
}
