package io.gamedock.sdk.userdata.playerdata.functions;

import android.content.Context;

import androidx.annotation.VisibleForTesting;

import com.google.gson.Gson;

import java.io.IOException;
import java.io.InputStream;
import java.util.Map;

import io.gamedock.sdk.GamedockSDK;
import io.gamedock.sdk.gamedata.GamedockGameDataManager;
import io.gamedock.sdk.models.gamedata.GamedockGameData;
import io.gamedock.sdk.models.gamedata.currencies.Currency;
import io.gamedock.sdk.models.gamedata.items.Item;
import io.gamedock.sdk.models.userdata.UserData;
import io.gamedock.sdk.models.userdata.UserDataVersion;
import io.gamedock.sdk.models.userdata.inventory.Inventory;
import io.gamedock.sdk.models.userdata.inventory.PlayerItem;
import io.gamedock.sdk.models.userdata.inventory.UniquePlayerItem;
import io.gamedock.sdk.models.userdata.mission.MissionData;
import io.gamedock.sdk.models.userdata.wallet.PlayerCurrency;
import io.gamedock.sdk.models.userdata.wallet.Wallet;
import io.gamedock.sdk.userdata.UserDataManager;
import io.gamedock.sdk.userdata.playerdata.PlayerDataManager;
import io.gamedock.sdk.utils.assets.FileAssetsUtil;
import io.gamedock.sdk.utils.error.ErrorCodes;
import io.gamedock.sdk.utils.logging.LoggingUtil;
import io.gamedock.sdk.utils.userid.UserIDGenerator;

/**
 * Internal class used by the {@link PlayerDataManager} to perform all initialisation functions.
 */
public class PlayerDataInitialisation {

    /**
     * Method that initialises the {@link UserData} from the local assets folder.
     * The details of the object are found in the "defaultPlayerData.json".
     *
     * @param context The activity context.
     * @param gson    The Gson object used for deserializing objects.
     * @return Returns the {@link UserData} object loaded from the assets folder.
     */
    public static UserData initUserDataFromAssets(Context context, Gson gson, FileAssetsUtil fileAssetsUtil) {
        String userProfileString = loadPlayerDataFromAssets(context, fileAssetsUtil);

        if (userProfileString == null || userProfileString.length() <= 0) {
            if (!UserDataManager.getInstance(context).getPlayerDataManager().loadFailedFired) {
                GamedockSDK.getInstance(context).getUserDataCallbacks().userDataError(ErrorCodes.LoadFailed);
                UserDataManager.getInstance(context).getPlayerDataManager().loadFailedFired = true;
            }

            return null;
        }

        GamedockGameData gamedockGameData = GamedockGameDataManager.getInstance(context).getGameData();
        UserData userData = gson.fromJson(userProfileString, UserData.class);

        if (gamedockGameData == null || userData == null) {
            if (!UserDataManager.getInstance(context).getPlayerDataManager().loadFailedFired) {
                GamedockSDK.getInstance(context).getUserDataCallbacks().userDataError(ErrorCodes.LoadFailed);
                UserDataManager.getInstance(context).getPlayerDataManager().loadFailedFired = true;
            }
            return null;
        }

        userData.Build();

        if (userData.getWallet() != null) {

            for (Map.Entry<Integer, PlayerCurrency> entry : userData.getWallet().getCurrenciesMap().entrySet()) {
                Integer key = entry.getKey();
                if (gamedockGameData.getCurrenciesMap().containsKey(key)) {
                    PlayerCurrency playerCurrency = entry.getValue();
                    Currency currency = gamedockGameData.getCurrenciesMap().get(key);

                    playerCurrency.populateValues(currency);
                }
            }

        }

        if (userData.getInventory() != null) {
            for (Map.Entry<Integer, PlayerItem> entry : userData.getInventory().getItemsMap().entrySet()) {
                Integer key = entry.getKey();
                if (gamedockGameData.getItemsMap().containsKey(key)) {
                    PlayerItem playerItem = entry.getValue();
                    Item item = gamedockGameData.getItemsMap().get(key);

                    playerItem.populateValues(item);
                }
            }

            for (Map.Entry<String, UniquePlayerItem> entry : userData.getInventory().getUniqueItemsMap().entrySet()) {
                UniquePlayerItem uniquePlayerItem = entry.getValue();
                if (gamedockGameData.getItemsMap().containsKey(uniquePlayerItem.getId())) {
                    Item item = gamedockGameData.getItemsMap().get(uniquePlayerItem.getId());

                    uniquePlayerItem.populateValues(item);
                }
            }
        }

        UserDataVersion userDataVersion = new UserDataVersion();
        userDataVersion.setDeviceId(UserIDGenerator.getUniqueDeviceId(context));
        userDataVersion.setVersion(0);

        userData.getUserDataVersions().add(userDataVersion);

        userData.setMissionData(new MissionData());

        return userData;
    }

    /**
     * Method used to initialise the {@link UserData} from the {@link android.content.SharedPreferences}.
     *
     * @param context           The activity context.
     * @param gson              The Gson object used for deserializing objects.
     * @param userProfileString The {@link UserData} object in a JSON {@link String} format.
     * @return Returns the {@link UserData} object loaded from the {@link android.content.SharedPreferences}.
     */
    public static UserData initUserDataFromPrefs(Context context, Gson gson, String userProfileString) {
        UserData userData = null;
        try {
            GamedockGameData gamedockGameData = GamedockGameDataManager.getInstance(context).getGameData();
            userData = gson.fromJson(userProfileString, UserData.class);

            if (gamedockGameData == null || userData == null) {
                if (!UserDataManager.getInstance(context).getPlayerDataManager().loadFailedFired) {
                    GamedockSDK.getInstance(context).getUserDataCallbacks().userDataError(ErrorCodes.LoadFailed);
                    UserDataManager.getInstance(context).getPlayerDataManager().loadFailedFired = true;
                }
                return null;
            }

            if (userData.getWallet() != null) {
                for (Map.Entry<Integer, PlayerCurrency> entry : userData.getWallet().getCurrenciesMap().entrySet()) {
                    Integer key = entry.getKey();
                    if (gamedockGameData.getCurrenciesMap().containsKey(key)) {
                        PlayerCurrency playerCurrency = entry.getValue();
                        Currency currency = gamedockGameData.getCurrenciesMap().get(key);

                        if (currency != null) {
                            playerCurrency.populateValues(currency);
                        }
                    }
                }
            }

            if (userData.getInventory() != null) {
                for (Map.Entry<Integer, PlayerItem> entry : userData.getInventory().getItemsMap().entrySet()) {
                    Integer key = entry.getKey();
                    if (gamedockGameData.getItemsMap().containsKey(key)) {
                        PlayerItem playerItem = entry.getValue();
                        Item item = gamedockGameData.getItemsMap().get(key);

                        if (item != null) {
                            playerItem.populateValues(item);
                        }
                    }
                }

                for (Map.Entry<String, UniquePlayerItem> entry : userData.getInventory().getUniqueItemsMap().entrySet()) {
                    UniquePlayerItem uniquePlayerItem = entry.getValue();
                    if (gamedockGameData.getItemsMap().containsKey(uniquePlayerItem.getId())) {
                        Item item = gamedockGameData.getItemsMap().get(uniquePlayerItem.getId());

                        if (item != null) {
                            uniquePlayerItem.populateValues(item);
                        }
                    }
                }
            }

            if (userData.getMissionData() == null) {
                userData.setMissionData(new MissionData());
            }
        } catch (Exception | Error e) {
            if (!UserDataManager.getInstance(context).getPlayerDataManager().loadFailedFired) {
                GamedockSDK.getInstance(context).getUserDataCallbacks().userDataError(ErrorCodes.LoadFailed);
                UserDataManager.getInstance(context).getPlayerDataManager().loadFailedFired = true;
            }
            return null;
        }

        return userData;
    }

    /**
     * Method that initialises the {@link Wallet} object with the initial values.
     * Called internally by {@link PlayerDataManager}.
     *
     * @param userData The {@link UserData} that contains the {@link Wallet} information
     * @param context  The activity context.
     * @return Returns the {@link UserData} containing the {@link Wallet} object with the initial values.
     */
    public static UserData walletInitialValues(UserData userData, Context context) {
        GamedockGameData gamedockGameData = GamedockGameDataManager.getInstance(context).getGameData();

        if (gamedockGameData == null || userData == null) {
            if (!UserDataManager.getInstance(context).getPlayerDataManager().loadFailedFired) {
                GamedockSDK.getInstance(context).getUserDataCallbacks().userDataError(ErrorCodes.LoadFailed);
                UserDataManager.getInstance(context).getPlayerDataManager().loadFailedFired = true;
            }
            return null;
        }

        for (Map.Entry<Integer, Currency> currency : gamedockGameData.getCurrenciesMap().entrySet()) {
            if (currency.getValue().getInitialValue() > 0) {
                userData.getWallet().getCurrenciesMap().get(currency.getKey()).setDelta(currency.getValue().getInitialValue());
                userData.getWallet().getCurrenciesMap().get(currency.getKey()).setCurrentBalance(currency.getValue().getInitialValue());
            }
        }

        return userData;
    }

    /**
     * Method that initialises the {@link Inventory} object with the initial values.
     * Called internally by {@link PlayerDataManager}.
     *
     * @param userData The {@link UserData} that contains the {@link Inventory} information
     * @param context  The activity context.
     * @return Returns the {@link UserData} containing the {@link Inventory} object with the initial values.
     */
    public static UserData inventoryInitialValues(UserData userData, Context context) {
        GamedockGameData gamedockGameData = GamedockGameDataManager.getInstance(context).getGameData();

        if (gamedockGameData == null || userData == null) {
            if (!UserDataManager.getInstance(context).getPlayerDataManager().loadFailedFired) {
                GamedockSDK.getInstance(context).getUserDataCallbacks().userDataError(ErrorCodes.LoadFailed);
                UserDataManager.getInstance(context).getPlayerDataManager().loadFailedFired = true;
            }
            return null;
        }

        for (Map.Entry<Integer, Item> item : gamedockGameData.getItemsMap().entrySet()) {
            if (item.getValue().getInitialValue() > 0) {
                PlayerItem playerItem = new PlayerItem(item.getValue());
                playerItem.setAmount(item.getValue().getInitialValue());
                playerItem.setDelta(item.getValue().getInitialValue());
                userData.getInventory().getItemsMap().put(playerItem.getId(), playerItem);
            }
        }
        return userData;
    }

    /**
     * Method that loads the Player Data information from the local assets folder.
     * The information is contained in the "defaultPlayerData.json" file.
     *
     * @param context The activity context.
     * @return Returns the Player Data information in JSON {@link String} format or {@code null} if no file was found.
     */
    @VisibleForTesting
    private static String loadPlayerDataFromAssets(Context context, FileAssetsUtil fileAssetsUtil) {
        String json = null;

        try {
            InputStream is = fileAssetsUtil.getPlayerDataFileAsset(context);
            int size = is.available();
            byte[] buffer = new byte[size];
            is.read(buffer);
            is.close();
            json = new String(buffer, "UTF-8");

        } catch (IOException ex) {
            LoggingUtil.e("The 'defaultPlayerData.json' file is missing from your assets folder. If you want to use the Wallet/Inventory/Shop functionality please include this file.");
        }

        return json;
    }
}
