package io.gamedock.sdk.tier;

import com.unity3d.player.UnityPlayer;

import org.json.JSONArray;
import org.json.JSONException;
import org.json.JSONObject;

import io.gamedock.sdk.models.tier.TieredEventProgress;
import io.gamedock.sdk.utils.error.ErrorCodes;
import io.gamedock.sdk.utils.logging.LoggingUtil;

/**
 * Class that handles the Tiered Events Callbacks.
 */
public class TieredEventsCallbacks {
    private OnTieredEventListener tieredEventListener;

    public TieredEventsCallbacks() {
        tieredEventListener = null;
    }

    public TieredEventsCallbacks(OnTieredEventListener tieredEventListener) {
        this.tieredEventListener = tieredEventListener;
    }

    /**
     * Method called to inform the game when a tiered event is available.
     */
    public void tieredEventsAvailable() {
        if (tieredEventListener != null) {
            tieredEventListener.TieredEventsAvailable();
        } else {
            try {
                UnityPlayer.UnitySendMessage("GamedockSDK", "TieredEventsAvailable", "");
            } catch (NoClassDefFoundError e) {
                LoggingUtil.w("You need to register the OnTieredEventsAvailable in order to receive information or something went wrong with Unity");
            }
        }
    }

    /**
     * Method called to inform the game when a tiered event is not available.
     */
    public void tieredEventsNotAvailable() {
        if (tieredEventListener != null) {
            tieredEventListener.TieredEventsNotAvailable();
        } else {
            try {
                UnityPlayer.UnitySendMessage("GamedockSDK", "TieredEventsNotAvailable", "");
            } catch (NoClassDefFoundError e) {
                LoggingUtil.w("You need to register the OnTieredEventsNotAvailable in order to receive information or something went wrong with Unity");
            }
        }
    }

    /**
     * Method called to inform the game a tiered event has been updated.
     *
     * @param tieredProgress The tiered event progress.
     */
    public void tieredEventUpdated(TieredEventProgress tieredProgress) {
        if (tieredEventListener != null) {
            tieredEventListener.TieredEventUpdated(tieredProgress);
        } else {
            try {
                JSONObject jsonObject = new JSONObject();

                jsonObject.put("tieredEventId", tieredProgress.getTieredEventId());
                jsonObject.put("currentTierId", tieredProgress.getCurrentTierId());
                jsonObject.put("previousAmount", tieredProgress.getPreviousAmount());
                jsonObject.put("currentAmount", tieredProgress.getCurrentAmount());
                jsonObject.put("completedTiers", new JSONArray(tieredProgress.getCompletedTiers()));
                jsonObject.put("claimableTiers", new JSONArray(tieredProgress.getClaimableTiers()));
                jsonObject.put("completed", tieredProgress.isCompleted());

                UnityPlayer.UnitySendMessage("GamedockSDK", "TieredEventUpdated", jsonObject.toString());
            } catch (JSONException | NoClassDefFoundError e) {
                LoggingUtil.w("You need to register the OnTieredEventUpdated in order to receive information or something went wrong with Unity");
            }
        }
    }

    /**
     * Method called to inform the game that a tiered event has been opened.
     */
    public void tieredEventProgressOpen() {
        if (tieredEventListener != null) {
            tieredEventListener.TieredEventProgressOpen();
        } else {
            try {
                UnityPlayer.UnitySendMessage("GamedockSDK", "TieredEventProgressOpen", "");
            } catch (NoClassDefFoundError e) {
                LoggingUtil.w("You need to register the OnTieredEventsAvailable in order to receive information or something went wrong with Unity");
            }
        }
    }

    /**
     * Method called to inform the game that a tiered event has been closed opened.
     */
    public void tieredEventProgressClosed() {
        if (tieredEventListener != null) {
            tieredEventListener.TieredEventProgressClosed();
        } else {
            try {
                UnityPlayer.UnitySendMessage("GamedockSDK", "TieredEventProgressClosed", "");
            } catch (NoClassDefFoundError e) {
                LoggingUtil.w("You need to register the OnTieredEventsAvailable in order to receive information or something went wrong with Unity");
            }
        }
    }

    /**
     * Method that is called when an error occurred while processing Tiered Events operations.
     * Uses the {@link ErrorCodes} class do define the error.
     */
    public void tieredEventsError(ErrorCodes error) {
        if (tieredEventListener != null) {
            tieredEventListener.TieredEventError(error);
        } else {
            try {
                JSONObject jsonObject = new JSONObject();
                jsonObject.put("id", error.getId());
                jsonObject.put("name", error.getName());
                jsonObject.put("message", error.getMessage());

                UnityPlayer.UnitySendMessage("GamedockSDK", "TieredEventsError", jsonObject.toString());
            } catch (JSONException | NoClassDefFoundError e) {
                LoggingUtil.w("Tried to send information to Unity but UnityPlayer could not be found (NoClassDefFoundError). This can happen by design and is not necessarily a problem.");
            }
        }
    }
}
