package io.gamedock.sdk.support;

import android.app.Activity;
import android.content.Context;
import android.content.pm.PackageManager;
import android.graphics.Color;
import android.graphics.Typeface;
import android.os.Build;
import android.os.Bundle;
import android.view.Gravity;
import android.view.View;
import android.view.ViewGroup;
import android.webkit.WebView;
import android.webkit.WebViewClient;
import android.widget.RelativeLayout;
import android.widget.TextView;

import org.json.JSONArray;
import org.json.JSONException;
import org.json.JSONObject;

import java.io.UnsupportedEncodingException;
import java.net.URLEncoder;
import java.util.Locale;

import io.gamedock.sdk.GamedockSDK;
import io.gamedock.sdk.utils.logging.LoggingUtil;
import io.gamedock.sdk.utils.userid.UserIDGenerator;

/**
 * Activity that handles the showing of a Zendesk {@link WebView} interface of the Customer Support.
 */
public class SupportActivity extends Activity {

    WebView webView;
    TextView userIdTV;
    TextView deviceIdTV;
    View line;

    String defaultUrl = "https://appsupport.spilgames.com/hc/en-us";

    @Override
    protected void onCreate(Bundle savedInstanceState) {
        super.onCreate(savedInstanceState);

        final Context context = this;

        Bundle extras = getIntent().getExtras();

        if (extras != null) {
            String URL = extras.getString("webViewUrl", null);
			String tags = extras.getString("tags");

            // Creating a new LinearLayout
            RelativeLayout relativeLayout = new RelativeLayout(this);
            relativeLayout.setBackgroundColor(Color.TRANSPARENT);
            relativeLayout.setPadding(0, 0, 0, 0);
            relativeLayout.setBackgroundColor(Color.WHITE);

            // Defining the RelativeLayout layout parameters.
            // In this case I want to fill its parent
            RelativeLayout.LayoutParams rlp = new RelativeLayout.LayoutParams(
                    RelativeLayout.LayoutParams.MATCH_PARENT,
                    RelativeLayout.LayoutParams.MATCH_PARENT);
            rlp.setMargins(0, 0, 0, 0);


            userIdTV = new TextView(this);
            RelativeLayout.LayoutParams textViewParams = new RelativeLayout.LayoutParams(ViewGroup.LayoutParams.MATCH_PARENT, 200);
            textViewParams.addRule(RelativeLayout.ALIGN_PARENT_BOTTOM);
            textViewParams.addRule(RelativeLayout.CENTER_HORIZONTAL);
            userIdTV.setBackgroundColor(Color.parseColor("#424242"));
            userIdTV.setTextColor(Color.WHITE);
            userIdTV.setText("User ID:\n" + UserIDGenerator.getGamedockUserId(this));
            userIdTV.setPadding(10, 10, 10, 10);
            userIdTV.setId(View.generateViewId());
            userIdTV.setTextIsSelectable(true);
            userIdTV.setGravity(Gravity.CENTER);
            userIdTV.setTypeface(Typeface.SANS_SERIF);

            relativeLayout.addView(userIdTV, textViewParams);

            userIdTV.setOnClickListener(new View.OnClickListener() {
                @Override
                public void onClick(View view) {
                    android.content.ClipboardManager clipboard = (android.content.ClipboardManager) getSystemService(Context.CLIPBOARD_SERVICE);
                    android.content.ClipData clip = android.content.ClipData.newPlainText("Copied Text", UserIDGenerator.getGamedockUserId(context));
                    clipboard.setPrimaryClip(clip);
                }
            });


            //Device ID

            deviceIdTV = new TextView(this);
            RelativeLayout.LayoutParams deviceTextViewParams = new RelativeLayout.LayoutParams(ViewGroup.LayoutParams.MATCH_PARENT, 200);
            deviceTextViewParams.addRule(RelativeLayout.ABOVE, userIdTV.getId());
            deviceTextViewParams.addRule(RelativeLayout.CENTER_HORIZONTAL);
            deviceIdTV.setBackgroundColor(Color.parseColor("#424242"));
            deviceIdTV.setTextColor(Color.WHITE);
            deviceIdTV.setText("Device ID:\n" + UserIDGenerator.getUniqueDeviceId(context));
            deviceIdTV.setPadding(10, 10, 10, 10);
            deviceIdTV.setId(View.generateViewId());
            deviceIdTV.setTextIsSelectable(true);
            deviceIdTV.setGravity(Gravity.CENTER);
            deviceIdTV.setTypeface(Typeface.SANS_SERIF);

            relativeLayout.addView(deviceIdTV, deviceTextViewParams);

            deviceIdTV.setOnClickListener(new View.OnClickListener() {
                @Override
                public void onClick(View view) {
                    android.content.ClipboardManager clipboard = (android.content.ClipboardManager) getSystemService(Context.CLIPBOARD_SERVICE);
                    android.content.ClipData clip = android.content.ClipData.newPlainText("Copied Text", UserIDGenerator.getUniqueDeviceId(context));
                    clipboard.setPrimaryClip(clip);
                }
            });

            line = new View(this);
            RelativeLayout.LayoutParams lineViewParams = new RelativeLayout.LayoutParams(ViewGroup.LayoutParams.MATCH_PARENT, 2);
            lineViewParams.addRule(RelativeLayout.ABOVE, deviceIdTV.getId());
            line.setBackgroundColor(Color.BLACK);
            line.setId(View.generateViewId());

            relativeLayout.addView(line, lineViewParams);

            webView = new WebView(this);

            webView.setBackgroundColor(Color.TRANSPARENT);

            //hack for android 4.x
            webView.setWebViewClient(new WebViewClient() {
                @Override
                public void onPageFinished(WebView view, String url) {
                    webView.setBackgroundColor(Color.TRANSPARENT);
                }
            });

            RelativeLayout.LayoutParams webViewParams = new RelativeLayout.LayoutParams(ViewGroup.LayoutParams.MATCH_PARENT, ViewGroup.LayoutParams.MATCH_PARENT);
            webViewParams.addRule(RelativeLayout.ABOVE, line.getId());
            webView.setLayoutParams(webViewParams);
            webView.clearCache(true);
            webView.clearHistory();
            webView.getSettings().setJavaScriptEnabled(true);
            webView.getSettings().setJavaScriptCanOpenWindowsAutomatically(true);
            webView.setBackgroundColor(Color.TRANSPARENT);
            webView.setLayerType(View.LAYER_TYPE_HARDWARE, null);

            relativeLayout.addView(webView, webViewParams);

			JSONObject payloadJsonObject = new JSONObject();
			try {
			    // Device data
                payloadJsonObject.put("platform", "android");
                payloadJsonObject.put("osVersion", Build.VERSION.RELEASE);
                payloadJsonObject.put("deviceModel", Build.MANUFACTURER + " " + Build.MODEL);

                // App data
                payloadJsonObject.put("bundleId", GamedockSDK.getInstance(context).getApplicationPackageName());
                try {
                    payloadJsonObject.put("appVersion", context.getPackageManager().getPackageInfo(context.getPackageName(), 0).versionName);
                } catch (PackageManager.NameNotFoundException e) {
                    payloadJsonObject.put("appVersion", "unknown");
                    e.printStackTrace();
                }

                // User data
                payloadJsonObject.put("uid", GamedockSDK.getInstance(context).getGamedockUID());
                payloadJsonObject.put("deviceId", UserIDGenerator.getUniqueDeviceId(context));

                Locale locale = Locale.getDefault();
                if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.N) {
                    locale = context.getResources().getConfiguration().getLocales().get(0);
                } else {
                    locale = context.getResources().getConfiguration().locale;
                }
                payloadJsonObject.put("locale", locale.toString());

                // Parameters added by developer
				JSONArray tagsJsonArray = new JSONArray();
				if(tags != null) {
					tagsJsonArray = new JSONArray(tags);
				}

                payloadJsonObject.put("tags", tagsJsonArray);

            } catch(JSONException ex) {
                LoggingUtil.e("Could not parse url parameters for Support WebView, opening webview with empty parameters.");
            }

            String payloadEncodedString = "";
            try {
                payloadEncodedString = URLEncoder.encode(payloadJsonObject.toString(), "UTF-8");
            } catch(UnsupportedEncodingException ex) {
                LoggingUtil.e("Could not encode url parameters for Support WebView, opening webview with empty parameters.");
            }

			String finalUrl = (URL != null ? URL : defaultUrl) + "?" + "data=" + payloadEncodedString;
			webView.loadUrl(finalUrl);

            webView.setBackgroundColor(Color.TRANSPARENT); ///hack for android 4.x

            this.setTheme(android.R.style.Theme_Translucent_NoTitleBar); //make sure activity is full width/height
            setContentView(relativeLayout, rlp);
        }
    }

    @Override
    public void onBackPressed() {
        if (webView.canGoBack()) {
            webView.goBack();
        } else {
            super.onBackPressed();
        }
    }

    @Override
    protected void onDestroy() {
        super.onDestroy();

        GamedockSDK.getInstance(this).isShowingChildActivity = false;
    }
}
