package io.gamedock.sdk.splashscreen;


import com.unity3d.player.UnityPlayer;

import org.json.JSONException;
import org.json.JSONObject;

import io.gamedock.sdk.utils.error.ErrorCodes;
import io.gamedock.sdk.utils.logging.LoggingUtil;

/**
 * Class that handles the Splash Screen Callbacks.
 * Notifies the application when the Daily Bonus has been received and processed by the SDK.
 * Informs the application if an error occurred while processing the Splash Screen values.
 */
public class SplashScreenCallbacks {

    private OnSplashScreenListener splashScreenListener;

    public SplashScreenCallbacks() {
        splashScreenListener = null;
    }

    public SplashScreenCallbacks(OnSplashScreenListener webListener) {
        this.splashScreenListener = webListener;
    }

    /**
     * Method fired when the in-game shop has been opened from the Splash Screen.
     */
    public void splashScreenOpenShop() {
        if (splashScreenListener != null) {
            splashScreenListener.SplashScreenOpenShop();
        } else {
            try {
                UnityPlayer.UnitySendMessage("GamedockSDK", "SplashScreenOpenShop", "");
            } catch (NoClassDefFoundError e) {
                LoggingUtil.w("You need to register the SplashScreenListener in order to receive information or something went wrong with Unity");
            }
        }
    }

    /**
     * Method called when the Splash Screen Web View has been opened.
     */
    public void splashScreenOpen() {
        if (splashScreenListener != null) {
            splashScreenListener.SplashScreenOpen();
        } else {
            try {
                UnityPlayer.UnitySendMessage("GamedockSDK", "SplashScreenOpen", "");
            } catch (NoClassDefFoundError e) {
                LoggingUtil.w("You need to register the SplashScreenListener in order to receive information or something went wrong with Unity");
            }
        }
    }

    /**
     * Method called when the Splash Screen Web View has been closed.
     */
    public void splashScreenClosed() {
        if (splashScreenListener != null) {
            splashScreenListener.SplashScreenClosed();
        } else {
            try {
                UnityPlayer.UnitySendMessage("GamedockSDK", "SplashScreenClosed", "");
            } catch (NoClassDefFoundError e) {
                LoggingUtil.w("You need to register the SplashScreenListener in order to receive information or something went wrong with Unity");
            }
        }
    }

    /**
     * Method called when the Splash Screen Web View is ready.
     */
    public void splashScreenAvailable(String type) {
        if (splashScreenListener != null) {
            splashScreenListener.SplashScreenAvailable(type);
        } else {
            try {
                UnityPlayer.UnitySendMessage("GamedockSDK", "SplashScreenAvailable", type);
            } catch (NoClassDefFoundError e) {
                LoggingUtil.w("You need to register the SplashScreenListener in order to receive information or something went wrong with Unity");
            }
        }
    }

    /**
     * Method called when the Splash Screen Web View cannot be loaded.
     */
    public void splashScreenNotAvailable() {
        if (splashScreenListener != null) {
            splashScreenListener.SplashScreenNotAvailable();
        } else {
            try {
                UnityPlayer.UnitySendMessage("GamedockSDK", "SplashScreenNotAvailable", "");
            } catch (NoClassDefFoundError e) {
                LoggingUtil.w("You need to register the SplashScreenListener in order to receive information or something went wrong with Unity");
            }
        }
    }

    /**
     * Method called when the Splash Screen sends data to the game.
     */
    public void splashScreenData(String payload) {
        if (splashScreenListener != null) {
            splashScreenListener.SplashScreenData(payload);
        } else {
            try {
                UnityPlayer.UnitySendMessage("GamedockSDK", "SplashScreenData", payload);
            } catch (NoClassDefFoundError e) {
                LoggingUtil.w("You need to register the SplashScreenListener in order to receive information or something went wrong with Unity");
            }
        }
    }
    /**
     * Method that is called when an error occurred while processing the Splash Screen data.
     * Uses the {@link ErrorCodes} class do define the error.
     */
    public void splashScreenError(ErrorCodes error) {
        if (splashScreenListener != null) {
            splashScreenListener.SplashScreenError(error);
        } else {
            try {
                JSONObject jsonObject = new JSONObject();
                jsonObject.put("id", error.getId());
                jsonObject.put("name", error.getName());
                jsonObject.put("message", error.getMessage());

                UnityPlayer.UnitySendMessage("GamedockSDK", "SplashScreenError", jsonObject.toString());
            } catch (NoClassDefFoundError | JSONException e) {
                LoggingUtil.w("You need to register the SplashScreenListener in order to receive information or something went wrong with Unity");
            }
        }
    }
}
