package io.gamedock.sdk.reward;


import com.unity3d.player.UnityPlayer;

import org.json.JSONArray;
import org.json.JSONException;
import org.json.JSONObject;

import io.gamedock.sdk.events.response.ResponseEvent;
import io.gamedock.sdk.utils.error.ErrorCodes;
import io.gamedock.sdk.utils.logging.LoggingUtil;

/**
 * Class that handles the Reward Callbacks.
 * Notifies the application when the Reward has been received and processed by the SDK.
 */
public class RewardCallbacks {

    private OnRewardListener rewardListener;

    public RewardCallbacks() {
        rewardListener = null;
    }

    public RewardCallbacks(OnRewardListener rewardListener) {
        this.rewardListener = rewardListener;
    }

    /**
     * Method that is called when a reward has been received from Gamedock backend and needs to be sent to Native/Unity.
     *
     * @param token      The token associated with the reward. Can only be used once.
     * @param reward     The reward json received from Gamedock. For display purposes.
     * @param rewardType The type of reward that has been received. Can be deeplink, push notification etc.
     */
    public void rewardTokenReceived(String token, JSONArray reward, String rewardType) {
        JSONObject jsonObject = new JSONObject();
        try {
            jsonObject.put("token", token);
            jsonObject.put("reward", reward);
            jsonObject.put("rewardType", rewardType);
        } catch (JSONException e) {
            e.printStackTrace();
        }

        if (rewardListener != null) {
            rewardListener.onRewardTokenReceived(token, reward, rewardType);
        } else {
            try {
                UnityPlayer.UnitySendMessage("GamedockSDK", "RewardTokenReceived", jsonObject.toString());
            } catch (NoClassDefFoundError e) {
                LoggingUtil.w("Tried to send information to Unity but UnityPlayer could not be found (NoClassDefFoundError). This can happen by design and is not necessarily a problem.");
            }
        }
    }

    /**
     * Method that is called when a reward has been claimed from Gamedock backend and needs to be sent to Native/Unity.
     *
     * @param reward     The reward json received from Gamedock backend. For display purposes.
     * @param rewardType The type of reward that has been received. Can be deeplink, push notification etc.
     */
    public void rewardTokenClaimed(JSONArray reward, String rewardType) {
        JSONObject jsonObject = new JSONObject();
        try {
            jsonObject.put("reward", reward);
            jsonObject.put("rewardType", rewardType);
        } catch (JSONException e) {
            e.printStackTrace();
        }

        if (rewardListener != null) {
            rewardListener.onRewardTokenClaimed(reward, rewardType);
        } else {
            try {
                UnityPlayer.UnitySendMessage("GamedockSDK", "RewardTokenClaimed", jsonObject.toString());
            } catch (NoClassDefFoundError e) {
                LoggingUtil.w("Tried to send information to Unity but UnityPlayer could not be found (NoClassDefFoundError). This can happen by design and is not necessarily a problem.");
            }
        }
    }

    /**
     * Method that is called when a reward has failed to be claimed from Gamedock backend and the message needs to be sent to Native/Unity.
     *
     * @param rewardType The type of reward that has been received. Can be deeplink, push notification etc.
     * @param error      The error information received from Gamedock backend.
     */
    public void rewardTokenClaimFailed(String rewardType, ErrorCodes error) {
        JSONObject jsonObject = new JSONObject();
        try {
            jsonObject.put("rewardType", rewardType);

            JSONObject errorJSON = new JSONObject();
            if (error != null) {
                errorJSON.put("id", error.getId());
                errorJSON.put("name", error.getName());
                errorJSON.put("message", error.getMessage());
            }

            jsonObject.put("error", errorJSON);
        } catch (JSONException e) {
            e.printStackTrace();
        }

        if (rewardListener != null) {
            rewardListener.onRewardTokenClaimFailed(rewardType, error);
        } else {
            try {
                UnityPlayer.UnitySendMessage("GamedockSDK", "RewardTokenClaimFailed", jsonObject.toString());
            } catch (NoClassDefFoundError e) {
                LoggingUtil.w("Tried to send information to Unity but UnityPlayer could not be found (NoClassDefFoundError). This can happen by design and is not necessarily a problem.");
            }
        }
    }

    /**
     * Method that is called when the notification reward has been handled by the SDK and needs to be sent to Native/Unity.
     *
     * @param notificationData The notification reward data formatted as a JSON {@link String}.
     */
    public void sendNotificationRewards(String notificationData) {
        ResponseEvent responseEvent = new ResponseEvent();
        responseEvent.setName("reward");
        responseEvent.setType("notificationReward");
        responseEvent.setAction("receive");

        JSONObject jsonObject = new JSONObject();
        try {
            JSONObject nData = new JSONObject(notificationData);

            jsonObject.put("currencyName", nData.getString("currencyName"));
            jsonObject.put("currencyId", nData.getString("currencyId"));
            jsonObject.put("reward", nData.getInt("reward"));
        } catch (JSONException e) {
            e.printStackTrace();
        }

        responseEvent.addData("eventData", jsonObject);

        if (rewardListener != null) {
            rewardListener.onNotificationReceived(responseEvent.toStringNotification());
        } else {
            try {
                UnityPlayer.UnitySendMessage("GamedockSDK", "OnResponseReceived", responseEvent.toStringNotification());
            } catch (NoClassDefFoundError e) {
                LoggingUtil.w("Tried to send information to Unity but UnityPlayer could not be found (NoClassDefFoundError). This can happen by design and is not necessarily a problem.");
            }
        }
    }
}
