package io.gamedock.sdk.initialization;

import com.unity3d.player.UnityPlayer;

import io.gamedock.sdk.utils.logging.LoggingUtil;

/**
 * Class that handles the SDK Initialisation Callbacks.
 * Notifies the application when a user id change is needed or if the game needs to inform the user he needs to update.
 */
public class InitializationDataCallbacks {
    private OnIntializationData intializationDataListener;

    public InitializationDataCallbacks() {
        intializationDataListener = null;
    }

    public InitializationDataCallbacks(OnIntializationData intializationDataListener) {
        this.intializationDataListener = intializationDataListener;
    }

    public void initializationCompleted() {
        if (intializationDataListener != null) {
            intializationDataListener.InitializationCompleted();
        } else {
            try {
                UnityPlayer.UnitySendMessage("GamedockSDK", "InitializationCompleted", "");
            } catch (NoClassDefFoundError e) {
                LoggingUtil.w("Tried to send information to Unity but UnityPlayer could not be found (NoClassDefFoundError). This can happen by design and is not necessarily a problem.");
            }
        }
    }

    /**
     * Method called to notify the game that a user id change has been requested from the backend.
     *
     * @param newUid The new user id the user will have.
     */
    public void userIdChangeRequest(String newUid) {
        if (intializationDataListener != null) {
            intializationDataListener.UserIdChangeRequest(newUid);
        } else {
            try {
                UnityPlayer.UnitySendMessage("GamedockSDK", "UserIdChangeRequest", newUid);
            } catch (NoClassDefFoundError e) {
                LoggingUtil.w("Tried to send information to Unity but UnityPlayer could not be found (NoClassDefFoundError). This can happen by design and is not necessarily a problem.");
            }
        }
    }

    /**
     * Method called to notify the game that the user id change has been completed.
     */
    public void userIdChangeCompleted() {
        if (intializationDataListener != null) {
            intializationDataListener.UserIdChangeCompleted();
        } else {
            try {
                UnityPlayer.UnitySendMessage("GamedockSDK", "UserIdChangeCompleted", "");
            } catch (NoClassDefFoundError e) {
                LoggingUtil.w("Tried to send information to Unity but UnityPlayer could not be found (NoClassDefFoundError). This can happen by design and is not necessarily a problem.");
            }
        }
    }

    /**
     * Method called to notify the game if a new version should be downloaded or it is recommended to be downloaded.
     *
     * @param status The status can either be FORCED or RECOMMENDED.
     */
    public void gameVersionStatus(String status) {
        if (intializationDataListener != null) {
            intializationDataListener.GameVersionStatus(status);
        } else {
            try {
                UnityPlayer.UnitySendMessage("GamedockSDK", "GameVersionStatus", status);
            } catch (NoClassDefFoundError e) {
                LoggingUtil.w("Tried to send information to Unity but UnityPlayer could not be found (NoClassDefFoundError). This can happen by design and is not necessarily a problem.");
            }
        }
    }
}
