package io.gamedock.sdk.google.dynamiclinks;

import android.app.Activity;
import android.content.Context;
import android.net.Uri;

import androidx.annotation.NonNull;

import com.google.android.gms.tasks.OnFailureListener;
import com.google.android.gms.tasks.OnSuccessListener;
import com.google.firebase.dynamiclinks.FirebaseDynamicLinks;
import com.google.firebase.dynamiclinks.PendingDynamicLinkData;

import org.json.JSONArray;
import org.json.JSONException;

import java.io.UnsupportedEncodingException;
import java.net.URLDecoder;

import io.gamedock.sdk.GamedockSDK;
import io.gamedock.sdk.events.Event;
import io.gamedock.sdk.utils.logging.LoggingUtil;

/**
 * Class used to manage the Firebase Dynamic Link and Adjust Deep Link operations.
 */
public class DeepLinkManager {

    public static final String DeepLink = "deeplink";
    public static String processedDeepLinkUrl;

    /**
     * Method used to check if a Dynamic Link is available from Firebase.
     *
     * @param context The context of the Activity.
     */
    public static void requestFirebaseDynamicLinkApi(final Context context) {
        Activity activity;
        try {
            activity = (Activity) context;

            FirebaseDynamicLinks.getInstance()
                    .getDynamicLink(activity.getIntent())
                    .addOnSuccessListener(activity, new OnSuccessListener<PendingDynamicLinkData>() {
                        @Override
                        public void onSuccess(PendingDynamicLinkData pendingDynamicLinkData) {
                            if (pendingDynamicLinkData != null) {
                                String deepLink = pendingDynamicLinkData.getLink().toString();

                                LoggingUtil.d("Deep Link found from Firebase: " + deepLink);

                                processDeepLink(context, deepLink);
                            } else {
                                LoggingUtil.d("No deep link found for Firebase!");
                            }
                        }
                    })
                    .addOnFailureListener(activity, new OnFailureListener() {
                        @Override
                        public void onFailure(@NonNull Exception e) {
                            LoggingUtil.d("No deep link found for Firebase!");
                        }
                    });
        } catch (NullPointerException | IllegalStateException | ClassCastException e) {
            LoggingUtil.w("Error processing the Deeplink request! Context not of type Activity");
        }
    }

    /**
     * Method used to process the value of a deep link that has been found in either Firebase or Adjust.
     *
     * @param context  The context of the Activity.
     * @param deepLink The deep link url.
     */
    public static void processDeepLink(Context context, String deepLink) {

        if (processedDeepLinkUrl != null && processedDeepLinkUrl.equals(deepLink)) {
            return;
        }

        processedDeepLinkUrl = deepLink;

        Event event = new Event(context);
        event.setName("deepLinkClicked");
        event.addCustomData("deepLink", deepLink);

        GamedockSDK.getInstance(context).trackEvent(event, null);

        Uri uri = Uri.parse(deepLink);

        String type = uri.getQueryParameter("type");

        if (type != null) {
            if (type.equals("attribution")) {
                Event attributionEvent = new Event(context);
                attributionEvent.setName("attribution");

                for (String value : uri.getQueryParameterNames()) {
                    attributionEvent.addCustomData(value, uri.getQueryParameter(value));
                }

                GamedockSDK.getInstance(context).trackEvent(attributionEvent, null);
            }
        } else {
            String token = uri.getQueryParameter("token");
            String reward = uri.getQueryParameter("reward");

            if (token != null) {
                JSONArray rewardArray = new JSONArray();
                if (reward != null) {
                    try {
                        reward = URLDecoder.decode(reward, "UTF-8");
                    } catch (UnsupportedEncodingException e) {
                        e.printStackTrace();
                    }

                    try {
                        rewardArray = new JSONArray(reward);
                        for (int i = 0; i < rewardArray.length(); i++) {
                            if (rewardArray.getJSONObject(i).has("id")) {
                                if (rewardArray.getJSONObject(i).get("id") instanceof String) {
                                    rewardArray.getJSONObject(i).put("externalId", rewardArray.getJSONObject(i).get("id"));
                                    rewardArray.getJSONObject(i).remove("id");
                                }
                            } else {
                                LoggingUtil.d("Deep Link Reward Id not found. Make sure the deep link is in the correct format.");
                            }
                        }
                    } catch (JSONException e) {
                        LoggingUtil.e("Deep Link Reward parameter could not be retrieved! Not a valid JSON format!");
                    }
                }

                GamedockSDK.getInstance(context).getRewardCallbacks().rewardTokenReceived(token, rewardArray, DeepLink);

                LoggingUtil.d("Values for link: token = " + token + ", reward = " + reward);
            } else {
                LoggingUtil.d("No reward attached to link");
            }

            String splashScreen = uri.getQueryParameter("splashscreen");

            if (splashScreen != null) {
                LoggingUtil.d("Requesting splash screen from deeplink: " + splashScreen);
                GamedockSDK.getInstance(context).requestSplashScreen(splashScreen);
            }

            GamedockSDK.getInstance(context).getDeepLinkCallbacks().deepLinkReceived(deepLink);
        }
    }
}
